/*
 * Copyright 2025 Google Inc. All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file except
 * in compliance with the License. You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under the
 * License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 * express or implied. See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.firebase.ui.auth.ui.components

import androidx.compose.foundation.Image
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.PaddingValues
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.size
import androidx.compose.foundation.layout.width
import androidx.compose.foundation.shape.RoundedCornerShape
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.filled.Star
import androidx.compose.material3.Button
import androidx.compose.material3.ButtonDefaults
import androidx.compose.material3.Icon
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.text.style.TextOverflow
import androidx.compose.ui.tooling.preview.Preview
import androidx.compose.ui.unit.dp
import com.firebase.ui.auth.configuration.auth_provider.AuthProvider
import com.firebase.ui.auth.configuration.auth_provider.Provider
import com.firebase.ui.auth.configuration.string_provider.AuthUIStringProvider
import com.firebase.ui.auth.configuration.string_provider.DefaultAuthUIStringProvider
import com.firebase.ui.auth.configuration.theme.AuthUIAsset
import com.firebase.ui.auth.configuration.theme.AuthUITheme

/**
 * A customizable button for an authentication provider.
 *
 * This button displays the icon and name of an authentication provider (e.g., Google, Facebook).
 * It is designed to be used within a list of sign-in options. The button's appearance can be
 * customized using the [style] parameter, and its text is localized via the [stringProvider].
 *
 * **Example usage:**
 * ```kotlin
 * AuthProviderButton(
 *     provider = AuthProvider.Facebook(),
 *     onClick = { /* Handle Facebook sign-in */ },
 *     stringProvider = DefaultAuthUIStringProvider(LocalContext.current)
 * )
 * ```
 *
 * @param modifier A modifier for the button
 * @param provider The provider to represent.
 * @param onClick A callback when the button is clicked
 * @param enabled If the button is enabled. Defaults to true.
 * @param style Optional custom styling for the button.
 * @param stringProvider The [AuthUIStringProvider] for localized strings
 *
 * @since 10.0.0
 */
@Composable
fun AuthProviderButton(
    modifier: Modifier = Modifier,
    provider: AuthProvider,
    onClick: () -> Unit,
    enabled: Boolean = true,
    style: AuthUITheme.ProviderStyle? = null,
    stringProvider: AuthUIStringProvider,
) {
    val context = LocalContext.current
    val providerStyle = resolveProviderStyle(provider, style)
    val providerLabel = resolveProviderLabel(provider, stringProvider, context)

    Button(
        modifier = modifier,
        contentPadding = PaddingValues(horizontal = 12.dp),
        colors = ButtonDefaults.buttonColors(
            containerColor = providerStyle.backgroundColor,
            contentColor = providerStyle.contentColor,
        ),
        shape = providerStyle.shape,
        elevation = ButtonDefaults.buttonElevation(
            defaultElevation = providerStyle.elevation
        ),
        onClick = onClick,
        enabled = enabled,
    ) {
        Row(
            modifier = modifier,
            verticalAlignment = Alignment.CenterVertically,
            horizontalArrangement = Arrangement.Start
        ) {
            val providerIcon = providerStyle.icon
            if (providerIcon != null) {
                val iconTint = providerStyle.iconTint
                if (iconTint != null) {
                    Icon(
                        modifier = Modifier
                            .size(24.dp),
                        painter = providerIcon.painter,
                        contentDescription = providerLabel,
                        tint = iconTint
                    )
                } else {
                    Image(
                        modifier = Modifier
                            .size(24.dp),
                        painter = providerIcon.painter,
                        contentDescription = providerLabel
                    )
                }
                Spacer(modifier = Modifier.width(12.dp))
            }
            Text(
                text = providerLabel,
                overflow = TextOverflow.Ellipsis,
                maxLines = 1,
            )
        }
    }
}

internal fun resolveProviderStyle(
    provider: AuthProvider,
    style: AuthUITheme.ProviderStyle?,
): AuthUITheme.ProviderStyle {
    if (style != null) return style

    val defaultStyle =
        AuthUITheme.Default.providerStyles[provider.providerId] ?: AuthUITheme.ProviderStyle.Empty

    return if (provider is AuthProvider.GenericOAuth) {
        AuthUITheme.ProviderStyle(
            icon = provider.buttonIcon ?: defaultStyle.icon,
            backgroundColor = provider.buttonColor ?: defaultStyle.backgroundColor,
            contentColor = provider.contentColor ?: defaultStyle.contentColor,
        )
    } else {
        defaultStyle
    }
}

internal fun resolveProviderLabel(
    provider: AuthProvider,
    stringProvider: AuthUIStringProvider,
    context: android.content.Context
): String = when (provider) {
    is AuthProvider.GenericOAuth -> provider.buttonLabel
    is AuthProvider.Apple -> {
        // Use Apple-specific locale if provided, otherwise use default stringProvider
        if (provider.locale != null) {
            val appleLocale = java.util.Locale.forLanguageTag(provider.locale)
            val appleStringProvider = DefaultAuthUIStringProvider(context, appleLocale)
            appleStringProvider.signInWithApple
        } else {
            stringProvider.signInWithApple
        }
    }
    else -> when (Provider.fromId(provider.providerId)) {
        Provider.GOOGLE -> stringProvider.signInWithGoogle
        Provider.FACEBOOK -> stringProvider.signInWithFacebook
        Provider.TWITTER -> stringProvider.signInWithTwitter
        Provider.GITHUB -> stringProvider.signInWithGithub
        Provider.EMAIL -> stringProvider.signInWithEmail
        Provider.PHONE -> stringProvider.signInWithPhone
        Provider.ANONYMOUS -> stringProvider.signInAnonymously
        Provider.MICROSOFT -> stringProvider.signInWithMicrosoft
        Provider.YAHOO -> stringProvider.signInWithYahoo
        Provider.APPLE -> stringProvider.signInWithApple
        null -> "Unknown Provider"
    }
}

@Preview(showBackground = true)
@Composable
private fun PreviewAuthProviderButton() {
    val context = LocalContext.current
    Column(
        modifier = Modifier
            .fillMaxSize(),
        verticalArrangement = Arrangement.Center,
        horizontalAlignment = Alignment.CenterHorizontally
    ) {
        AuthProviderButton(
            provider = AuthProvider.Email(
                emailLinkActionCodeSettings = null,
                passwordValidationRules = emptyList()
            ),
            onClick = {},
            stringProvider = DefaultAuthUIStringProvider(context)
        )
        AuthProviderButton(
            provider = AuthProvider.Phone(
                defaultNumber = null,
                defaultCountryCode = null,
                allowedCountries = null,
            ),
            onClick = {},
            stringProvider = DefaultAuthUIStringProvider(context)
        )
        AuthProviderButton(
            provider = AuthProvider.Google(
                scopes = emptyList(),
                serverClientId = null
            ),
            onClick = {},
            stringProvider = DefaultAuthUIStringProvider(context)
        )
        AuthProviderButton(
            provider = AuthProvider.Facebook(),
            onClick = {},
            stringProvider = DefaultAuthUIStringProvider(context)
        )
        AuthProviderButton(
            provider = AuthProvider.Twitter(
                customParameters = emptyMap()
            ),
            onClick = {},
            stringProvider = DefaultAuthUIStringProvider(context)
        )
        AuthProviderButton(
            provider = AuthProvider.Github(
                customParameters = emptyMap()
            ),
            onClick = {},
            stringProvider = DefaultAuthUIStringProvider(context)
        )
        AuthProviderButton(
            provider = AuthProvider.Microsoft(
                tenant = null,
                customParameters = emptyMap()
            ),
            onClick = {},
            stringProvider = DefaultAuthUIStringProvider(context)
        )
        AuthProviderButton(
            provider = AuthProvider.Yahoo(
                customParameters = emptyMap()
            ),
            onClick = {},
            stringProvider = DefaultAuthUIStringProvider(context)
        )
        AuthProviderButton(
            provider = AuthProvider.Apple(
                locale = null,
                customParameters = emptyMap()
            ),
            onClick = {},
            stringProvider = DefaultAuthUIStringProvider(context)
        )
        AuthProviderButton(
            provider = AuthProvider.Anonymous,
            onClick = {},
            stringProvider = DefaultAuthUIStringProvider(context)
        )
        AuthProviderButton(
            provider = AuthProvider.GenericOAuth(
                providerName = "Generic Provider",
                providerId = "google.com",
                scopes = emptyList(),
                customParameters = emptyMap(),
                buttonLabel = "Generic Provider",
                buttonIcon = AuthUIAsset.Vector(Icons.Default.Star),
                buttonColor = Color.Gray,
                contentColor = Color.White
            ),
            onClick = {},
            stringProvider = DefaultAuthUIStringProvider(context)
        )
        AuthProviderButton(
            provider = AuthProvider.GenericOAuth(
                providerName = "Generic Provider",
                providerId = "google.com",
                scopes = emptyList(),
                customParameters = emptyMap(),
                buttonLabel = "Custom Style",
                buttonIcon = AuthUIAsset.Vector(Icons.Default.Star),
                buttonColor = Color.Gray,
                contentColor = Color.White
            ),
            onClick = {},
            style = AuthUITheme.ProviderStyle(
                icon = AuthUITheme.Default.providerStyles[Provider.MICROSOFT.id]?.icon,
                backgroundColor = AuthUITheme.Default.providerStyles[Provider.MICROSOFT.id]!!.backgroundColor,
                contentColor = AuthUITheme.Default.providerStyles[Provider.MICROSOFT.id]!!.contentColor,
                iconTint = Color.Red,
                shape = RoundedCornerShape(24.dp),
                elevation = 6.dp
            ),
            stringProvider = DefaultAuthUIStringProvider(context)
        )
        AuthProviderButton(
            provider = AuthProvider.GenericOAuth(
                providerName = "Generic Provider",
                providerId = "unknown_provider",
                scopes = emptyList(),
                customParameters = emptyMap(),
                buttonLabel = "Unsupported Provider",
                buttonIcon = null,
                buttonColor = null,
                contentColor = null,
            ),
            onClick = {},
            stringProvider = DefaultAuthUIStringProvider(context)
        )
    }
}
