/*
 * Decompiled with CFR 0.152.
 */
package com.google.firebase.firestore.local;

import android.database.Cursor;
import androidx.annotation.NonNull;
import androidx.annotation.VisibleForTesting;
import com.google.firebase.Timestamp;
import com.google.firebase.database.collection.ImmutableSortedMap;
import com.google.firebase.firestore.core.Query;
import com.google.firebase.firestore.local.EncodedPath;
import com.google.firebase.firestore.local.IndexManager;
import com.google.firebase.firestore.local.LocalSerializer;
import com.google.firebase.firestore.local.QueryContext;
import com.google.firebase.firestore.local.RemoteDocumentCache;
import com.google.firebase.firestore.local.SQLitePersistence;
import com.google.firebase.firestore.model.Document;
import com.google.firebase.firestore.model.DocumentCollections;
import com.google.firebase.firestore.model.DocumentKey;
import com.google.firebase.firestore.model.FieldIndex;
import com.google.firebase.firestore.model.MutableDocument;
import com.google.firebase.firestore.model.ResourcePath;
import com.google.firebase.firestore.model.SnapshotVersion;
import com.google.firebase.firestore.proto.MaybeDocument;
import com.google.firebase.firestore.util.Assert;
import com.google.firebase.firestore.util.BackgroundQueue;
import com.google.firebase.firestore.util.Function;
import com.google.firebase.firestore.util.Util;
import com.google.protobuf.InvalidProtocolBufferException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import javax.annotation.Nonnull;
import javax.annotation.Nullable;

final class SQLiteRemoteDocumentCache
implements RemoteDocumentCache {
    @VisibleForTesting
    static final int BINDS_PER_STATEMENT = 9;
    private final SQLitePersistence db;
    private final LocalSerializer serializer;
    private IndexManager indexManager;
    private final DocumentTypeBackfiller documentTypeBackfiller = new DocumentTypeBackfiller();

    SQLiteRemoteDocumentCache(SQLitePersistence persistence, LocalSerializer serializer) {
        this.db = persistence;
        this.serializer = serializer;
    }

    @Override
    public void setIndexManager(IndexManager indexManager) {
        this.indexManager = indexManager;
    }

    @Override
    public void add(MutableDocument document, SnapshotVersion readTime) {
        Assert.hardAssert(!readTime.equals(SnapshotVersion.NONE), "Cannot add document to the RemoteDocumentCache with a read time of zero", new Object[0]);
        DocumentKey documentKey = document.getKey();
        Timestamp timestamp = readTime.getTimestamp();
        MaybeDocument message = this.serializer.encodeMaybeDocument(document);
        this.db.execute("INSERT OR REPLACE INTO remote_documents (path, path_length, read_time_seconds, read_time_nanos, document_type, contents) VALUES (?, ?, ?, ?, ?, ?)", EncodedPath.encode(documentKey.getPath()), documentKey.getPath().length(), timestamp.getSeconds(), timestamp.getNanoseconds(), DocumentType.forMutableDocument((MutableDocument)document).dbValue, message.toByteArray());
        this.indexManager.addToCollectionParentIndex(document.getKey().getCollectionPath());
    }

    @Override
    public void removeAll(Collection<DocumentKey> keys) {
        if (keys.isEmpty()) {
            return;
        }
        ArrayList<Object> encodedPaths = new ArrayList<Object>();
        ImmutableSortedMap deletedDocs = DocumentCollections.emptyDocumentMap();
        for (DocumentKey key : keys) {
            encodedPaths.add(EncodedPath.encode(key.getPath()));
            deletedDocs = deletedDocs.insert((Object)key, (Object)MutableDocument.newNoDocument(key, SnapshotVersion.NONE));
        }
        SQLitePersistence.LongQuery longQuery = new SQLitePersistence.LongQuery(this.db, "DELETE FROM remote_documents WHERE path IN (", encodedPaths, ")");
        while (longQuery.hasMoreSubqueries()) {
            longQuery.executeNextSubquery();
        }
        this.indexManager.updateIndexEntries(deletedDocs);
    }

    @Override
    public MutableDocument get(DocumentKey documentKey) {
        return this.getAll(Collections.singletonList(documentKey)).get(documentKey);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public Map<DocumentKey, MutableDocument> getAll(Iterable<DocumentKey> documentKeys) {
        HashMap<DocumentKey, MutableDocument> results = new HashMap<DocumentKey, MutableDocument>();
        ArrayList<Object> bindVars = new ArrayList<Object>();
        for (DocumentKey key : documentKeys) {
            bindVars.add(EncodedPath.encode(key.getPath()));
            results.put(key, MutableDocument.newInvalidDocument(key));
        }
        SQLitePersistence.LongQuery longQuery = new SQLitePersistence.LongQuery(this.db, "SELECT contents, read_time_seconds, read_time_nanos, document_type, path FROM remote_documents WHERE path IN (", bindVars, ") ORDER BY path");
        BackgroundQueue backgroundQueue = new BackgroundQueue();
        while (longQuery.hasMoreSubqueries()) {
            longQuery.performNextSubquery().forEach(row -> this.processRowInBackground(backgroundQueue, (Map<DocumentKey, MutableDocument>)results, (Cursor)row, null));
        }
        backgroundQueue.drain();
        this.documentTypeBackfiller.backfill(this.db);
        HashMap<DocumentKey, MutableDocument> hashMap = results;
        synchronized (hashMap) {
            return results;
        }
    }

    @Override
    public Map<DocumentKey, MutableDocument> getAll(String collectionGroup, FieldIndex.IndexOffset offset, int limit) {
        List<ResourcePath> collectionParents = this.indexManager.getCollectionParents(collectionGroup);
        ArrayList<ResourcePath> collections = new ArrayList<ResourcePath>(collectionParents.size());
        for (ResourcePath collectionParent : collectionParents) {
            collections.add((ResourcePath)((Object)collectionParent.append(collectionGroup)));
        }
        if (collections.isEmpty()) {
            return Collections.emptyMap();
        }
        if (9 * collections.size() < 900) {
            return this.getAll(collections, offset, limit, null);
        }
        HashMap<DocumentKey, MutableDocument> results = new HashMap<DocumentKey, MutableDocument>();
        int pageSize = 100;
        for (int i = 0; i < collections.size(); i += pageSize) {
            results.putAll(this.getAll(collections.subList(i, Math.min(collections.size(), i + pageSize)), offset, limit, null));
        }
        return Util.firstNEntries(results, limit, FieldIndex.IndexOffset.DOCUMENT_COMPARATOR);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private Map<DocumentKey, MutableDocument> getAll(List<ResourcePath> collections, FieldIndex.IndexOffset offset, int count, @Nullable DocumentType tryFilterDocumentType, @Nullable Function<MutableDocument, Boolean> filter, @Nullable QueryContext context) {
        Timestamp readTime = offset.getReadTime().getTimestamp();
        DocumentKey documentKey = offset.getDocumentKey();
        StringBuilder sql = Util.repeatSequence("SELECT contents, read_time_seconds, read_time_nanos, document_type, path FROM remote_documents WHERE path >= ? AND path < ? AND path_length = ? " + (tryFilterDocumentType == null ? "" : " AND (document_type IS NULL OR document_type = ?) ") + "AND (read_time_seconds > ? OR ( read_time_seconds = ? AND read_time_nanos > ?) OR ( read_time_seconds = ? AND read_time_nanos = ? and path > ?)) ", collections.size(), " UNION ");
        sql.append("ORDER BY read_time_seconds, read_time_nanos, path LIMIT ?");
        Object[] bindVars = new Object[(9 + (tryFilterDocumentType != null ? 1 : 0)) * collections.size() + 1];
        int i = 0;
        for (ResourcePath collection : collections) {
            String prefixPath = EncodedPath.encode(collection);
            bindVars[i++] = prefixPath;
            bindVars[i++] = EncodedPath.prefixSuccessor(prefixPath);
            bindVars[i++] = collection.length() + 1;
            if (tryFilterDocumentType != null) {
                bindVars[i++] = tryFilterDocumentType.dbValue;
            }
            bindVars[i++] = readTime.getSeconds();
            bindVars[i++] = readTime.getSeconds();
            bindVars[i++] = readTime.getNanoseconds();
            bindVars[i++] = readTime.getSeconds();
            bindVars[i++] = readTime.getNanoseconds();
            bindVars[i++] = EncodedPath.encode(documentKey.getPath());
        }
        bindVars[i] = count;
        BackgroundQueue backgroundQueue = new BackgroundQueue();
        HashMap<DocumentKey, MutableDocument> results = new HashMap<DocumentKey, MutableDocument>();
        this.db.query(sql.toString()).binding(bindVars).forEach(row -> {
            this.processRowInBackground(backgroundQueue, (Map<DocumentKey, MutableDocument>)results, (Cursor)row, filter);
            if (context != null) {
                context.incrementDocumentReadCount();
            }
        });
        backgroundQueue.drain();
        this.documentTypeBackfiller.backfill(this.db);
        HashMap<DocumentKey, MutableDocument> hashMap = results;
        synchronized (hashMap) {
            return results;
        }
    }

    private Map<DocumentKey, MutableDocument> getAll(List<ResourcePath> collections, FieldIndex.IndexOffset offset, int count, @Nullable Function<MutableDocument, Boolean> filter) {
        return this.getAll(collections, offset, count, null, filter, null);
    }

    private void processRowInBackground(BackgroundQueue backgroundQueue, Map<DocumentKey, MutableDocument> results, Cursor row, @Nullable Function<MutableDocument, Boolean> filter) {
        byte[] rawDocument = row.getBlob(0);
        int readTimeSeconds = row.getInt(1);
        int readTimeNanos = row.getInt(2);
        boolean documentTypeIsNull = row.isNull(3);
        String path = row.getString(4);
        Runnable runnable = () -> {
            MutableDocument document = this.decodeMaybeDocument(rawDocument, readTimeSeconds, readTimeNanos);
            if (documentTypeIsNull) {
                this.documentTypeBackfiller.enqueue(path, readTimeSeconds, readTimeNanos, document);
            }
            if (filter == null || ((Boolean)filter.apply(document)).booleanValue()) {
                Map map2 = results;
                synchronized (map2) {
                    results.put(document.getKey(), document);
                }
            }
        };
        if (row.isFirst() && row.isLast()) {
            runnable.run();
        } else {
            backgroundQueue.submit(runnable);
        }
    }

    @Override
    public Map<DocumentKey, MutableDocument> getDocumentsMatchingQuery(Query query, FieldIndex.IndexOffset offset, @Nonnull Set<DocumentKey> mutatedKeys) {
        return this.getDocumentsMatchingQuery(query, offset, mutatedKeys, null);
    }

    @Override
    public Map<DocumentKey, MutableDocument> getDocumentsMatchingQuery(Query query, FieldIndex.IndexOffset offset, @Nonnull Set<DocumentKey> mutatedKeys, @Nullable QueryContext context) {
        return this.getAll(Collections.singletonList(query.getPath()), offset, Integer.MAX_VALUE, DocumentType.FOUND_DOCUMENT, doc -> query.matches((Document)doc) || mutatedKeys.contains(doc.getKey()), context);
    }

    private MutableDocument decodeMaybeDocument(byte[] bytes, int readTimeSeconds, int readTimeNanos) {
        try {
            return this.serializer.decodeMaybeDocument(MaybeDocument.parseFrom(bytes)).setReadTime(new SnapshotVersion(new Timestamp((long)readTimeSeconds, readTimeNanos)));
        }
        catch (InvalidProtocolBufferException e) {
            throw Assert.fail("MaybeDocument failed to parse: %s", new Object[]{e});
        }
    }

    private static class DocumentTypeBackfiller {
        private final ConcurrentHashMap<BackfillKey, DocumentType> documentTypeByBackfillKey = new ConcurrentHashMap();

        private DocumentTypeBackfiller() {
        }

        void enqueue(String path, int readTimeSeconds, int readTimeNanos, MutableDocument document) {
            BackfillKey backfillKey = new BackfillKey(path, readTimeSeconds, readTimeNanos);
            DocumentType documentType = DocumentType.forMutableDocument(document);
            this.documentTypeByBackfillKey.putIfAbsent(backfillKey, documentType);
        }

        void backfill(SQLitePersistence db) {
            BackfillSqlInfo backfillSqlInfo;
            while ((backfillSqlInfo = this.calculateBackfillSql()) != null) {
                db.execute(backfillSqlInfo.sql, backfillSqlInfo.bindings);
            }
        }

        @Nullable
        BackfillSqlInfo calculateBackfillSql() {
            if (this.documentTypeByBackfillKey.isEmpty()) {
                return null;
            }
            ArrayList<Object> bindings = new ArrayList<Object>();
            StringBuilder caseClauses = new StringBuilder();
            StringBuilder whereClauses = new StringBuilder();
            Iterator backfillKeys = this.documentTypeByBackfillKey.keySet().iterator();
            int numDocumentsAffected = 0;
            while (backfillKeys.hasNext() && bindings.size() < 900) {
                BackfillKey backfillKey = (BackfillKey)backfillKeys.next();
                DocumentType documentType = this.documentTypeByBackfillKey.remove(backfillKey);
                if (documentType == null) continue;
                ++numDocumentsAffected;
                bindings.add(backfillKey.path);
                int pathBindingNumber = bindings.size();
                bindings.add(backfillKey.readTimeSeconds);
                int readTimeSecondsBindingNumber = bindings.size();
                bindings.add(backfillKey.readTimeNanos);
                int readTimeNanosBindingNumber = bindings.size();
                bindings.add(documentType.dbValue);
                int dbValueBindingNumber = bindings.size();
                caseClauses.append(" WHEN path=?").append(pathBindingNumber).append(" AND read_time_seconds=?").append(readTimeSecondsBindingNumber).append(" AND read_time_nanos=?").append(readTimeNanosBindingNumber).append(" THEN ?").append(dbValueBindingNumber);
                if (whereClauses.length() > 0) {
                    whereClauses.append(" OR");
                }
                whereClauses.append(" (path=?").append(pathBindingNumber).append(" AND read_time_seconds=?").append(readTimeSecondsBindingNumber).append(" AND read_time_nanos=?").append(readTimeNanosBindingNumber).append(')');
            }
            if (numDocumentsAffected == 0) {
                return null;
            }
            String sql = "UPDATE remote_documents SET document_type = CASE" + caseClauses + " ELSE NULL END WHERE" + whereClauses;
            return new BackfillSqlInfo(sql, bindings.toArray(), numDocumentsAffected);
        }

        private static class BackfillKey {
            final String path;
            final int readTimeSeconds;
            final int readTimeNanos;

            BackfillKey(String path, int readTimeSeconds, int readTimeNanos) {
                this.path = path;
                this.readTimeSeconds = readTimeSeconds;
                this.readTimeNanos = readTimeNanos;
            }

            public boolean equals(Object object) {
                if (object == this) {
                    return true;
                }
                if (!(object instanceof BackfillKey)) {
                    return false;
                }
                BackfillKey other = (BackfillKey)object;
                return this.readTimeSeconds == other.readTimeSeconds && this.readTimeNanos == other.readTimeNanos && Objects.equals(this.path, other.path);
            }

            public int hashCode() {
                return Objects.hash(this.path, this.readTimeSeconds, this.readTimeNanos);
            }

            @NonNull
            public String toString() {
                return "DocumentTypeBackfiller.BackfillKey(path=" + this.path + ", readTimeSeconds=" + this.readTimeSeconds + ", readTimeNanos=" + this.readTimeNanos + ")";
            }
        }

        private static class BackfillSqlInfo {
            final String sql;
            final Object[] bindings;
            final int numDocumentsAffected;

            BackfillSqlInfo(String sql, Object[] bindings, int numDocumentsAffected) {
                this.sql = sql;
                this.bindings = bindings;
                this.numDocumentsAffected = numDocumentsAffected;
            }
        }
    }

    private static enum DocumentType {
        NO_DOCUMENT(1),
        FOUND_DOCUMENT(2),
        UNKNOWN_DOCUMENT(3),
        INVALID_DOCUMENT(4);

        final int dbValue;

        private DocumentType(int dbValue) {
            this.dbValue = dbValue;
        }

        static DocumentType forMutableDocument(MutableDocument document) {
            if (document.isNoDocument()) {
                return NO_DOCUMENT;
            }
            if (document.isFoundDocument()) {
                return FOUND_DOCUMENT;
            }
            if (document.isUnknownDocument()) {
                return UNKNOWN_DOCUMENT;
            }
            Assert.hardAssert(!document.isValidDocument(), "MutableDocument has an unknown type", new Object[0]);
            return INVALID_DOCUMENT;
        }
    }
}

