package com.talpa.tengine.store

import android.content.Context
import androidx.room.*
import androidx.room.Entity

@Entity(tableName = "entity")
internal data class EntityTable(
    @PrimaryKey(autoGenerate = false)
    @ColumnInfo(name = "key")
    val key: String,
    @ColumnInfo(name = "value")
    val value: String
)

@Dao
internal abstract class EntityDao {

    @Query(value = "SELECT value FROM entity WHERE `key`=:key")
    abstract fun queryValue(key: String): String?

    @Insert(onConflict = OnConflictStrategy.REPLACE)
    abstract fun insertValue(entity: EntityTable)
}

@Database(version = 1, entities = [EntityTable::class])
internal abstract class KVDatabase : RoomDatabase() {

    abstract fun entityDao(): EntityDao
}

internal class RoomHelper(private val database: KVDatabase) {

    companion object {

        @Volatile
        private var database: KVDatabase? = null

        @Synchronized
        fun instance(context: Context): RoomHelper {
            val data =
                database ?: Room.databaseBuilder(context, KVDatabase::class.java, "key-value.db")
                    .fallbackToDestructiveMigration().build().apply { database = this }
            return RoomHelper(data)
        }
    }

    fun queryValue(key: String): String? {
        return database.entityDao().queryValue(key)
    }

    fun insertValue(key: String, value: String) {
        database.entityDao()
            .insertValue(EntityTable(key = key, value = value))
    }

}