package com.talpa.translate.service

import android.app.Notification
import android.app.NotificationChannel
import android.app.NotificationManager
import android.app.Service
import android.content.Context
import android.content.Intent
import android.content.pm.ServiceInfo
import android.content.res.Resources
import android.graphics.Bitmap
import android.media.projection.MediaProjectionManager
import android.os.IBinder
import android.os.RemoteCallbackList
import android.os.RemoteException
import android.util.DisplayMetrics
import android.view.WindowManager
import com.talpa.translate.ocr.PhotoAnalyzer
import com.talpa.translate.record.*
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.cancel
import kotlinx.coroutines.runBlocking

/**
 * Create by chenjunsheng on 2021/6/16
 */
class RecordService: Service() {

    companion object {

        const val TAG = "RecordService"
        const val PROJECT_INTENT = "project_intent"
        const val REQUEST_CODE = 102
        const val PROJECT_CODE = "project_code"

        const val RECORD_CHANNEL_ID = TAG
        const val RECORD_CHANNEL_NAME = "[$TAG]"
    }

    private var mMediaProjectionManager: MediaProjectionManager? = null
    private var mScreenRecorder: IScreenRecorder? = null
    private var mOcrController: IOcrController? = null
    private var mScope = CoroutineScope(Dispatchers.IO)

    override fun onCreate() {
        super.onCreate()
        val builder: Notification.Builder =
            Notification.Builder(this.applicationContext)
        val manager =
            getSystemService(Context.NOTIFICATION_SERVICE) as NotificationManager

        if (android.os.Build.VERSION.SDK_INT >= android.os.Build.VERSION_CODES.O) {
            val notificationChannel = NotificationChannel(
                RECORD_CHANNEL_ID,
                RECORD_CHANNEL_NAME,
                NotificationManager.IMPORTANCE_MIN
            )
            notificationChannel.enableLights(false) //如果使用中的设备支持通知灯，则说明此通知通道是否应显示灯
            notificationChannel.setShowBadge(false) //是否显示角标
            notificationChannel.lockscreenVisibility = Notification.VISIBILITY_SECRET
            manager.createNotificationChannel(notificationChannel)
            builder.setChannelId(RECORD_CHANNEL_ID)
        }

        if (android.os.Build.VERSION.SDK_INT >= android.os.Build.VERSION_CODES.Q) {
            startForeground(1, builder.build(), ServiceInfo.FOREGROUND_SERVICE_TYPE_MEDIA_PROJECTION)
        }
        mMediaProjectionManager =
            getSystemService(Context.MEDIA_PROJECTION_SERVICE) as MediaProjectionManager
    }

    private var mProjectionIntent: Intent? = null

    override fun onBind(serviceIntent: Intent): IBinder? {
        if (mScreenRecorder == null) {
            //val code = serviceIntent.getIntExtra(PROJECT_CODE, REQUEST_CODE)
            val intent = serviceIntent.getParcelableExtra<Intent>(PROJECT_INTENT)
            if (intent == null) {
                stopSelf()
                return null
            }
            mProjectionIntent = serviceIntent
            val wm = getSystemService(Context.WINDOW_SERVICE) as WindowManager
            val display = wm.defaultDisplay
            val metrics = DisplayMetrics()
            display.getRealMetrics(metrics)
            val width = metrics.widthPixels //size.x;
            val height = metrics.heightPixels //size.y;

            mScreenRecorder = ScreenRecorder(
                mWidth = width,
                mHeight = height,
                mDpi = Resources.getSystem()
                    .displayMetrics.densityDpi
            )
            mOcrController = OcrController(this)
        }
        return mBinder
    }

    private suspend fun captureFrameInner(): Bitmap? {
        val projectionIntent = checkNotNull(mProjectionIntent)
        val record = checkNotNull(mScreenRecorder)

        val wm = getSystemService(Context.WINDOW_SERVICE) as WindowManager
        val display = wm.defaultDisplay
        val metrics = DisplayMetrics()
        display.getRealMetrics(metrics)
        val width = metrics.widthPixels //size.x;
        val height = metrics.heightPixels //size.y;
        record.setDisplay(width, height)

        val code = projectionIntent.getIntExtra(PROJECT_CODE, REQUEST_CODE)
        val intent = projectionIntent.getParcelableExtra<Intent>(PROJECT_INTENT)
            ?: throw IllegalStateException("project intent null")
        val mediaProjection =
            try {
                checkNotNull(mMediaProjectionManager!!.getMediaProjection(code, intent))
            } catch (e: IllegalStateException) {
                e.printStackTrace()
                null
            } catch (e: SecurityException) {
                e.printStackTrace()
                null
            } ?: return null
        //mScreenRecorder?.startRecord(mediaProjection)
        return record.acquireLatestImage(mediaProjection)
    }

    private var mRemoteCallbackList = RemoteCallbackList<IRecordListener>()

    private val mBinder = object : IRecordService.Stub() {

        override fun registerListener(client: IRecordListener) {
            mRemoteCallbackList.register(client)
            notifyServiceReady()
        }

        override fun captureFrameSync(): CaptureResult {
            checkStatus()
            return runBlocking {
                val bitmap = captureFrameInner() ?: return@runBlocking CaptureResult(PROCESSOR_BITMAP_FAIL)
                //val processor = PhotoAnalyzer.preprocessor(bitmap) ?: return@runBlocking CaptureResult(PROCESSOR_BITMAP_FAIL, null)
                mOcrController?.setImage(PhotoAnalyzer.compressBitmap(bitmap))
                val result = mOcrController?.getOcrResult()
                mOcrController?.clearImage()
                CaptureResult(
                    CAPTURE_BITMAP_SUCCESS,
                    bitmap, result
                )
            }
        }

        override fun destroyService() {
            stopSelf()
        }
    }

    private fun notifyServiceReady() {
        var i = mRemoteCallbackList.beginBroadcast()
        while (i > 0) {
            i--
            try {
                mRemoteCallbackList.getBroadcastItem(i).onRecordServiceReady()
            } catch (e: RemoteException) {
                // The RemoteCallbackList will take care of removing
                // the dead object for us.
                e.printStackTrace()
            }
        }
        mRemoteCallbackList.finishBroadcast()
    }

    private fun checkStatus() {
        if (mProjectionIntent == null) {
            error("record service is not initialized")
        }
    }

    override fun onDestroy() {
        mScope.cancel()
        mRemoteCallbackList.kill()
        mOcrController?.destroy()
        mScreenRecorder?.destroy()
        mScreenRecorder = null
        mOcrController = null
        mProjectionIntent = null
    }
}