package com.talpa.overlay.view.overlay

import android.app.PendingIntent
import android.content.ActivityNotFoundException
import android.content.Context
import android.content.Intent
import android.graphics.Rect
import android.text.TextUtils
import android.view.Gravity
import android.view.MotionEvent
import android.view.View
import android.view.WindowManager
import android.view.accessibility.AccessibilityNodeInfo
import android.widget.*
import com.talpa.overlay.R
import com.talpa.overlay.data.readOverlayTextLanguageTag
import com.talpa.overlay.service.nodeText
import com.talpa.tengine.STORE
import com.talpa.tengine.Trans
import com.talpa.tengine.UNKNOWN
import com.talpa.tengine.lang.LANG
import com.talpa.translate.network.TransResponse
import com.talpa.translate.ocr.result.OcrResult
import com.trello.rxlifecycle3.kotlin.bindToLifecycle
import org.jetbrains.anko.find
import org.jetbrains.anko.layoutInflater
import org.jetbrains.anko.textColorResource
import java.lang.StringBuilder
import java.util.*

/**
 *
 * @author CY 2019-10-22
 */
class GeneralOverlayView(context: Context) : OverlayImpl(context), View.OnClickListener {

    companion object {
        /**
         * Click Copy Locale Broadcast Action
         */
        const val ACTION_CLICK_COPY = "ACTION_CLICK_COPY"
        const val ACTION_CLICK_PLAY = "ACTION_CLICK_PLAY"
        const val ACTION_CLICK_FAVOR = "ACTION_CLICK_FAVOR"
        const val ACTION_CLICK_SHARE = "ACTION_CLICK_SHARE"
    }


    private val rectInScreen = Rect()

    override fun addContentView(x: Int, y: Int) {
        super.addContentView(x, y)
        val params = overlayParams(contentView.layoutParams as? WindowManager.LayoutParams)
        params.x = x
        params.y = y

        contentView.visibility = View.GONE
        try {
            if (!contentView.isAttachedToWindow) {
                windowManager.addView(contentView, params)
            } else {
                windowManager.updateViewLayout(contentView, params)
            }
        } catch (e: Exception) {
            e.printStackTrace()
        }

    }

    fun updateTransResult(location: Rect, ocrResult: OcrResult?, response: TransResponse) {
        val originLoc = contentView.getTag(R.id.id_content_view_node_info) as? Rect
        if (originLoc == location) {
            val voice = contentView.findViewById<View>(R.id.btn_voice)
            val copy = contentView.findViewById<View>(R.id.btn_copy)
            val star = contentView.findViewById<CheckBox>(R.id.btn_star)
            val tvTranslation = contentView.findViewById<TextView>(R.id.tv_translation)
            val menu = contentView.find<View>(R.id.ll_menu)
            menu.visibility = View.VISIBLE

            if (textSpeech.isLanguageAvailable(Locale.forLanguageTag(readOverlayTextLanguageTag(context) ?: LANG.EN))) {
                voice.visibility = View.VISIBLE
            }

            //  val translation = trans.result?.translation
            val stringBuilder = StringBuilder()
            response.result?.texts?.forEach {
                stringBuilder.append(it)
                stringBuilder.append("\n")
            }
            if (stringBuilder.isNotBlank()) {
                stringBuilder.dropLast(1)
            }
            if (!stringBuilder.isNotEmpty()) {
                copy.setTag(R.id.id_translation_view_trans_result, stringBuilder.toString())
                copy.visibility = View.VISIBLE
            }
            star.visibility = View.VISIBLE
            star.isChecked = false

            tvTranslation.text = stringBuilder.toString()
            tvTranslation.textColorResource = R.color.color_floating_translation
            tvTranslation.setTag(R.id.id_translation_view_trans_result, tvTranslation.text)

            hideProgressBar(contentView)
        }
    }

    fun updateContentView(rect: Rect) {
        if (!contentView.isAttachedToWindow) {
            val params = overlayParams(contentView.layoutParams as? WindowManager.LayoutParams)
            try {
                windowManager.addView(contentView, params)
            } catch (e: Exception) {
                e.printStackTrace()
            }

        }

        //println("contentView==${contentView.parent}")
        val tempNode = contentView.getTag(R.id.id_content_view_node_info)
        if (tempNode == rect && contentView.visibility == View.VISIBLE) {

            return
        }

        contentView.setTag(R.id.id_content_view_node_info, rect)
        if (contentView.visibility != View.VISIBLE) {
            contentView.visibility = View.VISIBLE
        }
        //println("nodeInfo#hashCode=${nodeInfo.hashCode()}")
        val contentParams = contentView.layoutParams as WindowManager.LayoutParams

        //val width = screenSize.x - R.dimen.dp24 * 2
        contentParams.width = WindowManager.LayoutParams.MATCH_PARENT
        contentParams.height = WindowManager.LayoutParams.WRAP_CONTENT//200
        contentParams.gravity = Gravity.TOP or Gravity.START
        //contentParams.horizontalMargin=0.5f//context.resources.getDimension(R.dimen.dp80)
        val tvContent = contentView.findViewById<TextView>(R.id.tv_content)

        val text = ""
        tvContent.text = text


        val sourceText = text.toString().trim()
        if (!TextUtils.isEmpty(sourceText)) {

            //val packageName = nodeInfo.packageName?.toString() ?: ""

            val targetLanguageTag = readOverlayTextLanguageTag(context) ?: LANG.EN
            //targetLanguageTag
            //    ?: throw Exception("please set text translate targetLanguage,to see writeOverlayTextLanguageTag()")
            preTranslate()
            //dealNodeInfo(sourceText, targetLanguageTag, packageName)
        }

        contentParams.y = contentViewY(rect)
        try {
            windowManager.updateViewLayout(contentView, contentParams)
        } catch (e: Exception) {
            e.printStackTrace()
        }
    }

    override fun updateContentView(nodeInfo: AccessibilityNodeInfo) {
        if (!contentView.isAttachedToWindow) {
            val params = overlayParams(contentView.layoutParams as? WindowManager.LayoutParams)
            try {
                windowManager.addView(contentView, params)
            } catch (e: Exception) {
                e.printStackTrace()
            }

        }

        //println("contentView==${contentView.parent}")
        val tempNode = contentView.getTag(R.id.id_content_view_node_info)
        if (tempNode == nodeInfo && contentView.visibility == View.VISIBLE) {

            return
        }

        contentView.setTag(R.id.id_content_view_node_info, nodeInfo)
        if (contentView.visibility != View.VISIBLE) {
            contentView.visibility = View.VISIBLE
        }
        //println("nodeInfo#hashCode=${nodeInfo.hashCode()}")
        val contentParams = contentView.layoutParams as WindowManager.LayoutParams

        //val width = screenSize.x - R.dimen.dp24 * 2
        contentParams.width = WindowManager.LayoutParams.MATCH_PARENT
        contentParams.height = WindowManager.LayoutParams.WRAP_CONTENT//200
        contentParams.gravity = Gravity.TOP or Gravity.START
        //contentParams.horizontalMargin=0.5f//context.resources.getDimension(R.dimen.dp80)
        val tvContent = contentView.findViewById<TextView>(R.id.tv_content)

        val text = nodeInfo.nodeText()
        tvContent.text = text


        val sourceText = text.toString().trim()
        if (!TextUtils.isEmpty(sourceText)) {

            val packageName = nodeInfo.packageName?.toString() ?: ""

            val targetLanguageTag = readOverlayTextLanguageTag(context) ?: LANG.EN
            //targetLanguageTag
            //    ?: throw Exception("please set text translate targetLanguage,to see writeOverlayTextLanguageTag()")
            preTranslate()
            dealNodeInfo(sourceText, targetLanguageTag, packageName)
        }

        contentParams.y = contentViewY(nodeInfo)
        try {
            windowManager.updateViewLayout(contentView, contentParams)
        } catch (e: Exception) {
            e.printStackTrace()
        }

    }

    override fun removeContentView() {
        if (contentView.isAttachedToWindow) {
            //windowManager.removeView(contentView)
            try {
                windowManager.removeViewImmediate(contentView)
            } catch (e: Exception) {
                e.printStackTrace()
            }
        }

        textSpeech.stopSpeak()
        super.removeContentView()
    }

    /**
     * 创建内容视图
     */
    override fun createContentView(): View {


        /* val shapePathModel = ShapePathModel().apply {
             setAllCorners(CutCornerTreatment(px2dp(5)))
             setAllEdges(TriangleEdgeTreatment(px2dp(5), true))
         }
         val materialShapeDrawable = MaterialShapeDrawable(shapePathModel)*/

        val view = context.layoutInflater.inflate(R.layout.layout_content_view, null)

        (view as? BaseOverlayView)?.setOnBackClickListener = {
            removeContentView()
            true
        }

        view.setOnTouchListener { v, event ->

            when (event.action) {
                MotionEvent.ACTION_OUTSIDE -> {
                    removeContentView()
                }
            }

            return@setOnTouchListener false
        }

        val copy = view.findViewById<View>(R.id.btn_copy)
        val voice = view.findViewById<View>(R.id.btn_voice)
        val star = view.findViewById<View>(R.id.btn_star)
        val shared = view.findViewById<View>(R.id.btn_shared)
        val settingLanguage = view.findViewById<View>(R.id.btn_setting_language)


        copy.setOnClickListener(this)
        voice.setOnClickListener(this)
        star.setOnClickListener(this)
        shared.setOnClickListener(this)
        settingLanguage.setOnClickListener(this)
        return view

    }

    private fun contentViewY(rect: Rect) : Int {

        contentView.measure(0, 0)


        //nodeInfo.getBoundsInScreen(rectInScreen)
        val rectViewY: Int = rect.top
        val rectViewHeight: Int = rect.height()
        val contentViewHeight: Int = contentView.measuredHeight
        val yOffset = context.resources.getDimension(R.dimen.dp30).toInt()

        val topHeight = contentViewHeight + yOffset

        val bottomHeight = rectViewHeight + yOffset

        var y = rectViewY - topHeight

        if (y <= 0) {
            y = rectViewY + bottomHeight
        }

        return y
    }

    /**
     * 计算内容视图Y轴
     */
    private fun contentViewY(nodeInfo: AccessibilityNodeInfo): Int {

        contentView.measure(0, 0)


        nodeInfo.getBoundsInScreen(rectInScreen)
        val rectViewY: Int = rectInScreen.top
        val rectViewHeight: Int = rectInScreen.height()
        val contentViewHeight: Int = contentView.measuredHeight
        val yOffset = context.resources.getDimension(R.dimen.dp30).toInt()

        val topHeight = contentViewHeight + yOffset

        val bottomHeight = rectViewHeight + yOffset

        var y = rectViewY - topHeight

        if (y <= 0) {
            y = rectViewY + bottomHeight
        }

        return y
    }


    /**
     * 处理节点信息
     */
    private fun dealNodeInfo(sourceText: String, targetLanguageTag: String, packageName: String) {

        val tvTranslation = contentView.findViewById<TextView>(R.id.tv_translation)

        tvTranslation.text = ""

        logStartTranslate(
            sourceText = sourceText,
            targetLanguageTag = targetLanguageTag,
            packageName = packageName
        )

        /**
         *  it.text,
         *  it.translation,
         *  it.sourceLanguageLocaleTag,
         *  it.targetLanguageLocaleTag,
         *  it.star
         */
        val d = readTranslateHistory(sourceText, targetLanguageTag)
            .bindToLifecycle(tvTranslation)
            .subscribe({

                if (it.size == 5 && it[4] is Boolean) {

                    val (text, translation, sourceLanguageLocaleTag, targetLanguageLocaleTag, isChecked) = it
                    postTranslate(
                        targetLanguageLocaleTag.toString(),
                        translation.toString(),
                        isChecked as Boolean
                    )
                    logSuccessTranslate(
                        sourceText = sourceText,
                        sourceLanguage = sourceLanguageLocaleTag.toString(),
                        targetLanguage = targetLanguageLocaleTag.toString(),
                        packageName = packageName,
                        isCached = true,
                        source = STORE
                    )
                } else {
                    translate(sourceText, targetLanguageTag, packageName)
                }
            }, {
                translate(sourceText, targetLanguageTag, packageName)
            })
    }


    private fun translate(sourceText: String, targetLanguageTag: String, packageName: String) {

        val tvTranslation = contentView.findViewById<TextView>(R.id.tv_translation)

//        if (!context.isNetworkConnected()) {
//            tvTranslation.setText(R.string.network_no_connect)
//            tvTranslation.textColorResource = R.color.color_floating_failure
//            hideProgressBar(contentView)
//            return
//        }


        //  val targetLanguageTag = readOverlayTextLanguageTag(tvTranslation.context) ?: return
        val onSuccess = { trans: Trans ->
            val translation = trans.result?.translation
            if (!TextUtils.isEmpty(translation)) {
                postTranslate(targetLanguageTag, translation!!, isStar = false)
                val sourceLanguage = trans.result!!.detectLang ?: "AUTO"
                logSuccessTranslate(
                    sourceText = sourceText,
                    sourceLanguage = sourceLanguage,
                    targetLanguage = targetLanguageTag,
                    packageName = packageName,
                    source = trans.result?.source ?: UNKNOWN
                )

            } else {
                hideProgressBar(contentView)
            }

        }

        val onError = { trans: Trans ->
            logFailTranslate(
                sourceText = sourceText,
                targetLanguageTag = targetLanguageTag,
                packageName = packageName,
                errorMessage = trans.result?.errorMessage ?: "unknown"
            )
            hideProgressBar(contentView)
        }
        translateForOverlayText(tvTranslation, sourceText, targetLanguageTag, onSuccess, onError)
    }

    private fun showProgressBar(contentView: View?) {
        val progressBar = contentView?.find<ProgressBar>(R.id.loading_progress_bar)
        progressBar?.visibility = View.VISIBLE
    }

    private fun hideProgressBar(contentView: View?) {
        val progressBar = contentView?.find<ProgressBar>(R.id.loading_progress_bar)
        progressBar?.visibility = View.GONE
    }

    /**
     * 翻译前
     */
    private fun preTranslate() {
        val tvTranslation = contentView.findViewById<TextView>(R.id.tv_translation)

        val menu = contentView.find<View>(R.id.ll_menu)
        menu.visibility = View.GONE
        tvTranslation.text = ""

        showProgressBar(contentView)

    }

    /**
     * 翻译后
     */
    private fun postTranslate(targetLanguageTag: String, translation: String, isStar: Boolean) {
        val voice = contentView.findViewById<View>(R.id.btn_voice)
        val copy = contentView.findViewById<View>(R.id.btn_copy)
        val star = contentView.findViewById<CheckBox>(R.id.btn_star)
        val tvTranslation = contentView.findViewById<TextView>(R.id.tv_translation)
        val menu = contentView.find<View>(R.id.ll_menu)
        menu.visibility = View.VISIBLE

        if (textSpeech.isLanguageAvailable(Locale.forLanguageTag(targetLanguageTag))) {
            voice.visibility = View.VISIBLE
        }

        //  val translation = trans.result?.translation
        if (!TextUtils.isEmpty(translation)) {
            copy.setTag(R.id.id_translation_view_trans_result, translation)
            copy.visibility = View.VISIBLE
        }
        star.visibility = View.VISIBLE
        star.isChecked = isStar

        tvTranslation.text = translation
        tvTranslation.textColorResource = R.color.color_floating_translation
        tvTranslation.setTag(R.id.id_translation_view_trans_result, tvTranslation.text)

        hideProgressBar(contentView)
    }

    override fun onClick(v: View?) {

        when (v?.id) {
            R.id.btn_copy -> {


                val translation = v.getTag(R.id.id_translation_view_trans_result) as String
                val intent = Intent("BROADCAST_ACTION_CLIPBOARD_TEXT")
                intent.putExtra("label", "floating")
                intent.putExtra("text", translation)

                Toast.makeText(
                    context.applicationContext,
                    R.string.text_copy_success,
                    Toast.LENGTH_SHORT
                ).show()

                localBroadcastManager.sendBroadcast(intent)
                localBroadcastManager.sendBroadcast(Intent(ACTION_CLICK_COPY))
            }
            R.id.btn_voice -> {
                val tvTranslation = contentView.findViewById<TextView>(R.id.tv_translation)
                val languageTag = readOverlayTextLanguageTag(context)
                if (languageTag != null) {

                    if (!textSpeech.isSpeaking()) {
                        val state =
                            textSpeech.speak(tvTranslation.text, Locale.forLanguageTag(languageTag))
                        if (!state) {
                            Toast.makeText(
                                context.applicationContext,
                                R.string.playback_error,
                                Toast.LENGTH_SHORT
                            )
                                .show()
                        }
                    } else {
                        textSpeech.stopSpeak()
                    }

                }

                localBroadcastManager.sendBroadcast(Intent(ACTION_CLICK_PLAY))
            }
            R.id.btn_star -> {


                val isChecked = (v as Checkable).isChecked

                val tvContent = contentView.findViewById<TextView>(R.id.tv_content)
                val tvTranslation = contentView.findViewById<TextView>(R.id.tv_translation)
                val text = tvContent.text.toString().trim()
                val translation = tvTranslation.text.toString().trim()

                updateTranslateHistory(text, translation, isChecked)

                localBroadcastManager.sendBroadcast(Intent(ACTION_CLICK_FAVOR))
            }
            R.id.btn_setting_language -> {
                val tvTranslation = contentView.findViewById<TextView>(R.id.tv_translation)
                val sourceText = tvTranslation.text.toString()
            }
            R.id.btn_shared -> {
                //send()
                //val intent = Intent(Intent.ACTION_SEND)
                val tvTranslation = contentView.findViewById<TextView>(R.id.tv_translation)
                val sourceText = tvTranslation.text.toString()
                v.context.pendingShare(sourceText)

                localBroadcastManager.sendBroadcast(Intent(ACTION_CLICK_SHARE))
                v.postDelayed(this::removeContentView, 400)
            }
        }
    }

}

fun Context.pendingShare(text: String, subject: String = ""): Boolean {
    return try {
        val intent = Intent(Intent.ACTION_SEND)
        intent.type = "text/plain"
        intent.putExtra(Intent.EXTRA_SUBJECT, subject)
        intent.putExtra(Intent.EXTRA_TEXT, text)
        val selectIntent = Intent.createChooser(intent, null)
        selectIntent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK)
        val pendingIntent =
            PendingIntent.getActivity(this, 100, selectIntent, PendingIntent.FLAG_CANCEL_CURRENT)
        pendingIntent.send()
        true
    } catch (e: ActivityNotFoundException) {
        e.printStackTrace()
        false
    }
}
