package com.talpa.overlay.view

import android.app.ActivityManager
import android.content.*
import android.content.res.Configuration
import android.graphics.PixelFormat
import android.graphics.Point
import android.graphics.Rect
import android.os.*
import android.preference.PreferenceManager
import android.util.AttributeSet
import android.util.Log
import android.view.*
import android.view.MotionEvent.ACTION_MOVE
import android.view.MotionEvent.ACTION_UP
import android.view.accessibility.AccessibilityNodeInfo
import android.widget.EditText
import android.widget.Toast
import androidx.constraintlayout.widget.ConstraintLayout
import androidx.core.widget.ContentLoadingProgressBar
import androidx.localbroadcastmanager.content.LocalBroadcastManager
import com.talpa.PhotoTranslateController
import com.talpa.overlay.BuildConfig
import com.talpa.overlay.R
import com.talpa.overlay.RxRelay
import com.talpa.overlay.RxRelay.EVENT_NODE_INFO_DOWN
import com.talpa.overlay.data.ActivityChangedEvent
import com.talpa.overlay.data.readOverlayEditTextLanguageTag
import com.talpa.overlay.data.readOverlayTextLanguageTag
import com.talpa.overlay.data.readShowTips1
import com.talpa.overlay.service.AccessService
import com.talpa.overlay.state.FloatingStateMachine
import com.talpa.overlay.state.FloatingStateMachine.WHAT_LIGHT_ENTER
import com.talpa.overlay.tools.checkAccessibilityMalfunctioning
import com.talpa.overlay.tools.isAccessibilityEnabled
import com.talpa.overlay.tools.isInsideWhiteList
import com.talpa.overlay.translate.detectLanguage
import com.talpa.overlay.view.alert.AlertView
import com.talpa.overlay.view.overlay.*
import com.talpa.overlay.view.overlay.OverlayImpl.Companion.MODULE_SCREEN
import com.talpa.overlay.view.overlay.OverlayImpl.Companion.PT_TRANS_AREA_CONFIRM
import com.talpa.tengine.SERVER
import com.talpa.tengine.lang.LANG
import com.talpa.translate.HiTranslator
import com.talpa.translate.base.utils.logEvent
import com.talpa.translate.network.TransResponse
import com.talpa.translate.record.CaptureTransResult
import com.talpa.translate.repository.box.grammar.GrammarNew
import com.talpa.translate.repository.box.grammar.Operation
import com.talpa.translate.repository.box.grammar.ResultsV2
import com.talpa.translate.repository.box.grammar.getOperation
import com.talpa.translate.repository.grammar.GrammarRepo
import io.reactivex.android.schedulers.AndroidSchedulers
import io.reactivex.disposables.Disposable
import io.reactivex.subjects.PublishSubject
import kotlinx.coroutines.*
import org.jetbrains.anko.configuration
import java.lang.StringBuilder
import java.lang.ref.WeakReference
import java.util.*
import java.util.concurrent.TimeUnit
import kotlin.math.abs

typealias TranslationController = PhotoTranslateController
typealias TextRecognizeListener = PhotoTranslateController.TextRecognizeListener

/**
 * 悬浮视图
 * @author CY 2019-6-11
 */
class FloatingContainer : ConstraintLayout, TextRecognizeListener {

    companion object {

        /**
         * 检测敏感度时间控制
         */
        private const val DETECT_DELAY_MILLIS = 300L//15L

        /**
         * 速率时间
         */
        private const val VELOCITY_TIME_MILLIS = 300

        /**
         * 速率最小距离
         */
        private const val VELOCITY_DES = 80

        /**
         * Detect Event
         */
        private const val WHAT_DETECT = 100

        /**
         * recycler
         */
        private const val WHAT_RECYCLER = 200

        /**
         * SUCCESS TRANSLATE
         */
        internal const val WHAT_TRANSLATE_SUCCESS = 300

        /**
         * WHAT_TRANSLATE_FAILURE
         */
        internal const val WHAT_TRANSLATE_FAILURE = 400

        private const val DELAY_RECYCLER_MILLS = 2000

        internal const val WHAT_ASK_PERMISSION = 5000
        internal const val WHAT_ASK_PERMISSION_WHOLE = 5001

        private const val TAG = "Floating"

        private const val OVERLAY_MODE_GENERAL = 0
        private const val OVERLAY_MODE_SIMPLE = 1
        private const val OVERLAY_MODE_SPEECH = 2

        /**
         * 最小移动距离
         */
        const val MIN_DISTANCE = 300

        @JvmField
        var ALERT_DEBOUNCD = 700L
    }

    /**
     * 上一次位置x,y
     */
    private var lastX = 0f
    private var lastY = 0f

    /**
     * 每次的偏移量
     */
    private var dx = 0f
    private var dy = 0f

    private var selfX = 0f
    private var selfY = 0f


    /**
     * WindowManager
     */
    private val windowManager by lazy { context.getSystemService(Context.WINDOW_SERVICE) as WindowManager }

    /**
     * 识别视图
     */
    private val recognizerView by lazy { findViewById<View>(R.id.floating) }

    /**
     * 识别视图的中心位置
     */
    private val recognizerCenterPoint = intArrayOf(0, 0)

    /**
     * 节点信息
     */
    private var nodeSet: LinkedHashSet<AccessibilityNodeInfo>? = null

    private var currentNodeInfo: AccessibilityNodeInfo? = null

    private val rectInScreen = Rect()

    /**
     * 矩形背景视图
     */
    private val rectView: View by lazy {
        LayoutInflater.from(context.applicationContext).inflate(R.layout.layout_rect_view, null)
    }

    /**
     * 子线程 Handler
     */
    private val localHandler by lazy {
        val handlerThread = HandlerThread("Floating").apply { start() }
        LocalHandler(handlerThread.looper, this@FloatingContainer)
    }

    private val generalOverlayView: GeneralOverlayView by lazy { GeneralOverlayView(context) }
    private val simpleOverlayView: SimpleOverlayView by lazy { SimpleOverlayView(context) }
    private val speechOverlayView: SpeechOverlayView by lazy { SpeechOverlayView(context) }
    private val editTextOverlayView: EditTextOverlayView by lazy {
        EditTextOverlayView(
            context,
            localHandler
        )
    }
    private val firstOverlayView: FirstGeneralOverlayView by lazy {
        FirstGeneralOverlayView(context)
    }

    private var currentOverlayStyle = -1

    /**
     * 防止双击
     */
    private var downTime: Long = 0

    /**
     * 手势
     */
    private val gestureDetector by lazy { GestureDetector(context, onGestureListener) }

    /**
     * Touch
     */
    private var motionEvent: MotionEvent? = null

    /**
     * 速率
     */
    private lateinit var velocityTracker: VelocityTracker//? = null

    private var relayDisposable: Disposable? = null


    /**
     * 本地广播
     */
    val localBroadcastManager by lazy { LocalBroadcastManager.getInstance(context) }

    /**
     * 当前配置
     */
    var currentConfiguration: Configuration? = null

    private val lock = Any()

    constructor(context: Context) : super(context)
    constructor(context: Context, attrs: AttributeSet?) : super(context, attrs)
    constructor(context: Context, attrs: AttributeSet?, defStyleAttr: Int) : super(
        context,
        attrs,
        defStyleAttr
    )

    override fun onAttachedToWindow() {
        super.onAttachedToWindow()

        currentConfiguration = context.configuration

        relayDisposable = RxRelay.subscribe(this::onNext)

        try {
            context.registerReceiver(
                broadcastReceiver,
                IntentFilter(Intent.ACTION_CLOSE_SYSTEM_DIALOGS)
            )

        } catch (e: Exception) {
            e.printStackTrace()
        }

        velocityTracker = VelocityTracker.obtain()
    }


    override fun onDetachedFromWindow() {
        removeContentView()
        if (relayDisposable?.isDisposed == false) {
            relayDisposable?.dispose()
        }
        try {
            context.unregisterReceiver(broadcastReceiver)
        } catch (e: Exception) {
            e.printStackTrace()
        }

        super.onDetachedFromWindow()
        velocityTracker.recycle()
        mJob?.cancel()

    }

    private val broadcastReceiver = object : BroadcastReceiver() {
        override fun onReceive(context: Context?, intent: Intent?) {
            when (intent?.action) {
                Intent.ACTION_CLOSE_SYSTEM_DIALOGS -> {
                    removeContentView()
                    removeRectView()
                }
            }
        }
    }

    fun handleMessage(msg: Message) {

        when (msg.what) {
            WHAT_DETECT -> {
                val findNodeInfo: () -> Unit = msg.obj as () -> Unit
                findNodeInfo.invoke()
            }
            WHAT_RECYCLER -> {
                recycleNodes()
            }
            WHAT_TRANSLATE_SUCCESS -> {
                recycleNodes()
            }
            WHAT_TRANSLATE_FAILURE -> {
                recycleNodes()
            }
        }
    }

    private fun recycleNodes() {
        nodeSet?.let {
            synchronized(it) {
                it.map { info -> info.recycle() }
                it.clear()
            }
        }
        nodeSet = null
    }

    private class LocalHandler(looper: Looper, floatingContainer: FloatingContainer) :
        Handler(looper) {
        private val weakReference = WeakReference(floatingContainer)

        override fun handleMessage(msg: Message) {
            super.handleMessage(msg)
            weakReference.get()?.handleMessage(msg)

//            when (msg.what) {
//                WHAT_DETECT -> {
//                    val findNodeInfo: () -> Unit = msg.obj as () -> Unit
//                    findNodeInfo.invoke()
//                }
//                WHAT_RECYCLER -> {
//                    nodeSet?.let {
//                        synchronized(it) {
//                            it.map { info -> info.recycle() }
//                            it.clear()
//                        }
//                    }
//                    nodeSet = null
//                }
//            }
        }
    }

    private var mTopActivity: ActivityChangedEvent? = null

    private fun onNext(message: Message) {
        when (message.what) {
            EVENT_NODE_INFO_DOWN -> {

                GlobalScope.launch {
                    synchronized(lock) {
                        val nodeSet = message.obj as LinkedHashSet<AccessibilityNodeInfo>
                        this@FloatingContainer.nodeSet = nodeSet
                    }
                }

            }
            RxRelay.EVENT_ACTIVITY_CHANGE -> {
                mTopActivity = message.obj as ActivityChangedEvent


                /*if (currentNodeInfo?.packageName != packageName) {
                    //  removeContentView()
                }*/

            }
        }
    }

    private var mAccessServiceEnable = false
    private var mAccessMalfunction = false
    private var mSubject = PublishSubject.create<Int>()

    private var mAlertView: AlertView? = null

    init {
        val d = mSubject
            .throttleFirst(800, TimeUnit.MILLISECONDS)
            .debounce(ALERT_DEBOUNCD, TimeUnit.MILLISECONDS)
            .observeOn(AndroidSchedulers.mainThread())
            .subscribe { code ->
                if (code == WHAT_ASK_PERMISSION || code == WHAT_ASK_PERMISSION_WHOLE) {
                    val alertView = mAlertView ?: AlertView.Builder()
                        .setContext(context)
                        .build().also { mAlertView = it }
                    if (!alertView.isShowing) {
                        val from = if (code == WHAT_ASK_PERMISSION) {
                            FloatingManager.PART
                        } else {
                            FloatingManager.WHOLE
                        }
                        alertView.show(from)
                        localBroadcastManager.sendBroadcast(Intent(FloatingManager.BROADCAST_ACTION_ACCESS_SHOW).also {
                            it.putExtra(
                                FloatingManager.FROM, from
                            )
                        })
                    }
                    //context.requestAccessPermission()
                }
            }
        PhotoTranslateController.setTextRecognizeListener(this)
    }

    override fun onRecognizeServiceReady() {

    }

    /*override fun postTranslateSingle(from: String, to: String, texts: List<String>): TransResponse {
        if (grammarCheckEnable) {
            return runBlocking {
                var grammarResult = GrammarNew(ResultsV2(emptyList()))
                val grammarCheckList = texts.map {
                    val detectLan = detectLanguage(it)
                    if (detectLan == "en") {
                        val repo = GrammarRepo()
                        grammarResult = try {
                            repo.requestGrammarCheck(it)
                        } catch (e: Exception) {
                            e.printStackTrace()
                            GrammarNew(ResultsV2(emptyList()))
                        }
                        val sb = StringBuilder()
                        if (grammarResult.results?.alerts.isNullOrEmpty()) {
                            sb.append(it)
                        } else {
                            sb.append(it)
                            val parseList = grammarResult.results?.alerts!!.filter {
                                !it.title.isNullOrEmpty()
                            }

                            val sortList = parseList?.sortedByDescending {
                                it.highlightBegin
                            }

                            sortList.forEach {
                                when (getOperation(it)) {
                                    Operation.Replace -> {
                                        if (!it.replacements.isNullOrEmpty()) {
                                            sb.replace(
                                                it.highlightBegin,
                                                it.highlightEnd,
                                                it.replacements?.get(0) ?: ""
                                            )
                                        }
                                    }
                                    Operation.Add -> {
                                        if (!it.replacements.isNullOrEmpty()) {
                                            sb.indexOf(it.replacements?.get(0) ?: "", it.highlightBegin)
                                        }
                                    }

                                }
                            }
                        }
                        return@map sb.toString()
                    } else {
                        return@map it
                    }
                }
                return@runBlocking super.postTranslateSingle(from, to, grammarCheckList).apply {
                    sourceText = texts.firstOrNull()
                    grammarNew = grammarResult
                }
            }
        } else {
            return super.postTranslateSingle(from, to, texts)
        }
    }*/

    override fun onAreaConfirmed(rect: Rect) {
        /*if (readShowTips1()) {
            firstOverlayView.updateContentView(rect)
            return
        }*/
        when (translationStyle()) {
            OVERLAY_MODE_GENERAL -> {
                generalOverlayView.updateContentView(rect)
            }
            OVERLAY_MODE_SIMPLE -> {
                simpleOverlayView.updateContentView(rect)
            }
            OVERLAY_MODE_SPEECH -> {
                speechOverlayView.updateContentView(rect)
            }
        }
    }

    override fun getTransTargetLanguage(): String {
        return readOverlayTextLanguageTag(context) ?: LANG.EN
    }

    override fun getTransSourceLanguage(): String {
        return readOverlayEditTextLanguageTag(context) ?: Locale.getDefault().language
    }

    override fun onRecognizedFail(rect: Rect) {
        /*if (readShowTips1()) {
            firstOverlayView.updateTransFailed(rect)
            return
        }*/
        when (translationStyle()) {
            OVERLAY_MODE_GENERAL -> {
                generalOverlayView.updateTransFailed(rect)
            }
            OVERLAY_MODE_SIMPLE -> {
                simpleOverlayView.updateTransFailed(rect)
            }
            OVERLAY_MODE_SPEECH -> {
                speechOverlayView.updateTransFailed(rect)
            }
        }
    }

    private var mJob: Job? = null

    override fun onRecognized(index: Int, result: CaptureTransResult) {
        val location = result.captureResult.ocrResult?.blocks?.get(index)?.rect ?: return
        val ocrResult = result.captureResult.ocrResult

        mJob = CoroutineScope(Dispatchers.IO).launch {
            //val list = arrayListOf<String>(ocrResult?.blocks?.get(index)?.text ?: return@launch)

            val response = result.transMap?.get(index)
            /*val response = HiTranslator.getInstance(context).postTranslate(
                from = LANG.AUTO,
                to = readOverlayTextLanguageTag(context) ?: LANG.EN, texts = list
            )*/

            /*val s = StringBuilder()
            list.forEach {
                s.append(it)
            }*/
            if (response?.result?.texts == null) {
                logFailTranslate(
                    SERVER,
                    readOverlayTextLanguageTag(context) ?: LANG.EN,
                    context.packageName
                )
            } else {
                logSuccessTranslate(
                    SERVER,
                    Locale.getDefault().language,
                    readOverlayTextLanguageTag(context) ?: LANG.EN,
                    context.packageName,
                    false,
                    moduleType = MODULE_SCREEN
                )
            }

            withContext(Dispatchers.Main) {
                /*if (readShowTips1()) {
                    firstOverlayView.updateTransResult(location, ocrResult, response)
                    return@withContext
                }*/
                when (translationStyle()) {
                    OVERLAY_MODE_GENERAL -> {
                        generalOverlayView.updateTransResult(index, location, ocrResult, response)
                    }
                    OVERLAY_MODE_SIMPLE -> {
                        simpleOverlayView.updateTransResult(location, response)
                    }
                    OVERLAY_MODE_SPEECH -> {
                        speechOverlayView.updateTransResult(location, response)
                    }
                }
            }

        }


    }

    private var mDownX = 0f
    private var mDownY = 0f

    override fun onTouchEvent(event: MotionEvent): Boolean {
        velocityTracker.addMovement(event)
        val newEvent = MotionEvent.obtain(
            event.downTime, event.eventTime, event.action,
            (event.rawX - recognizerView.width), (event.rawY - recognizerView.height / 2),
            event.metaState
        )
        when (event.action) {
            MotionEvent.ACTION_DOWN -> {
                mAccessServiceEnable = context.isAccessibilityEnabled()
                mAccessMalfunction = checkAccessibilityMalfunctioning(context)
                PhotoTranslateController.delegateRecTouchEvent(newEvent)

                lastX = event.rawX
                lastY = event.rawY

                mDownX = event.rawX
                mDownY = event.rawY

                val params = layoutParams as WindowManager.LayoutParams
                selfX = params.x.toFloat()
                selfY = params.y.toFloat()

                if (mAccessServiceEnable && !mAccessMalfunction) {
                    startAccessService()
                }
                removeRectView()
                removeContentView()

                val current = System.currentTimeMillis()
                if (current - downTime > 120) {//防止快速双击
                    addRectView(lastX.toInt(), lastY.toInt())
                    addContentView(lastX.toInt(), lastY.toInt())
                    downTime = System.currentTimeMillis()
                }

            }
            ACTION_MOVE -> {

                val rawX = event.rawX
                val rawY = event.rawY

                dx = rawX - lastX
                dy = rawY - lastY

                selfX += dx
                selfY += dy

                lastX = rawX
                lastY = rawY
                if (abs(mDownX - rawX) < 20 && abs(mDownY - rawY) < 20) {
                    return gestureDetector.onTouchEvent(event)
                }

                if ((isInsideWhiteList(mTopActivity) || mAccessMalfunction) && FloatingStateMachine.currentState()
                    !is FloatingStateMachine.FloatingState.WorkingState.FindLocationState
                ) {
                    PhotoTranslateController.delegateRecTouchEvent(newEvent)
                    toLeftBottom()
                } else {
                    when (FloatingStateMachine.currentState()) {
                        is FloatingStateMachine.FloatingState.WorkingState.DetectState -> {
                            //速率计算
                            velocityTracker.computeCurrentVelocity(VELOCITY_TIME_MILLIS)
                            val xVelocity = velocityTracker.xVelocity
                            val yVelocity = velocityTracker.yVelocity
                            if (BuildConfig.DEBUG) println("xVelocity=$xVelocity   yVelocity=$yVelocity")

                            if (!localHandler.hasMessages(WHAT_DETECT)) {
                                if (abs(xVelocity) <= VELOCITY_DES && abs(yVelocity) <= VELOCITY_DES) {

                                    if (!mAccessServiceEnable) {
                                        //先询问权限
                                        if (FloatingStateMachine.currentState() !is FloatingStateMachine.FloatingState.WorkingState.FindLocationState) {
                                            mSubject.onNext(WHAT_ASK_PERMISSION)
                                        }
                                    } else {
                                        val message =
                                            Message.obtain(
                                                localHandler,
                                                WHAT_DETECT,
                                                this::findNodeInfo
                                            )
                                        localHandler.sendMessageDelayed(
                                            message,
                                            DETECT_DELAY_MILLIS
                                        )
                                    }
                                }
                            }
                            //速率过大时，取消识别
                            if (abs(xVelocity) > VELOCITY_DES || abs(yVelocity) > VELOCITY_DES) {
                                localHandler.removeMessages(WHAT_DETECT)
                            }
                            toLeftBottom()
                        }
                        is FloatingStateMachine.FloatingState.HighLightState,
                        is FloatingStateMachine.FloatingState.CopyState -> return true
                        is FloatingStateMachine.FloatingState.StaticState.IdleState -> {
                            return gestureDetector.onTouchEvent(event)
                        }
                    }
                }

                //if (BuildConfig.DEBUG) println("rawX=$rawX  rawY=$rawY  dx=$dx  dy=$dy")
                updateViewLayout()
            }
            ACTION_UP -> {
                currentNodeInfo = null
                //延迟移除，防止快速点击，松开导致视图未添加完，就已经执行移除方法
                postDelayed({ removeRectView() }, 400)

                if (motionEvent?.action == ACTION_MOVE) {
                    onGestureListener.onScrollUp(event)
                }
                actionUpOrOutSide()
                if ((isInsideWhiteList(mTopActivity) || mAccessMalfunction) && FloatingStateMachine.currentState()
                    !is FloatingStateMachine.FloatingState.WorkingState.FindLocationState
                ) {
                    PhotoTranslateController.delegateRecTouchEvent(newEvent)
                }
            }

            MotionEvent.ACTION_OUTSIDE -> {
                removeRectView()
                actionUpOrOutSide()
            }
            MotionEvent.ACTION_CANCEL -> {
                actionUpOrOutSide()
            }
        }
        motionEvent = event
        newEvent.recycle()
        return gestureDetector.onTouchEvent(event)//super.onTouchEvent(event)
    }

    /**
     * Offset
     */
    fun detectViewOffset(offsetX: Int, offsetY: Int) {
        selfX += offsetX
        selfY += offsetY
    }

    /**
     * 松开事件
     */
    private fun actionUpOrOutSide() {
        when (FloatingStateMachine.currentState()) {
            is FloatingStateMachine.FloatingState.WorkingState.DetectState -> {
                FloatingStateMachine.sendMessage(WHAT_LIGHT_ENTER)
            }
            is FloatingStateMachine.FloatingState.WorkingState.FindLocationState -> {
                findDockPosition()
                FloatingStateMachine.sendMessage(WHAT_LIGHT_ENTER)
            }
            is FloatingStateMachine.FloatingState.MenuState -> {
                FloatingStateMachine.sendMessage(WHAT_LIGHT_ENTER)
            }
        }
        localHandler.removeMessages(WHAT_DETECT)

        //   localHandler.sendEmptyMessageDelayed(WHAT_RECYCLER, 2000)
    }

    /**
     * 找出识别中心点的Node Info
     */
    @Synchronized
    private fun findNodeInfo() {

        val nodes = nodeSet?.let {
            synchronized(it) {
                try {
                    linkedSetOf<AccessibilityNodeInfo>().apply {
                        addAll(it)
                    }
                } catch (e: java.lang.Exception) {
                    e.printStackTrace()
                    return
                }

            }
        } ?: return

        /**
         * 捕捉中心点
         */
        val (cx, cy) = recognizerLocation()
        val nodeInfo = findNodeByPosition(cx, cy, nodes) ?: return
        nodes.clear()

        /**
         * 重复节点
         */
        if (currentNodeInfo == nodeInfo) {
            return
        }

        currentNodeInfo = nodeInfo

        post { dispatchNodeInfo(nodeInfo) }


    }

    /**
     * 分发节点信息
     */
    private fun dispatchNodeInfo(nodeInfo: AccessibilityNodeInfo) {

        when (nodeInfo.className) {
            /**
             * 输入框
             */
            EditText::class.java.name -> {
                removeContentView()
                //updateRectView(nodeInfo)
                //editTextNodeInfo(nodeInfo)
                editTextOverlayView.updateContentView(nodeInfo)
            }
            else -> {
                when (translationStyle()) {
                    OVERLAY_MODE_GENERAL -> {
                        updateRectView(nodeInfo)
                    }
                    OVERLAY_MODE_SIMPLE -> {
                        removeContentView()
                        removeRectView()
                    }
                    OVERLAY_MODE_SPEECH -> {
                        removeContentView()
                        removeRectView()
                    }
                }
                updateContentView(nodeInfo)
            }
        }
    }

    /**
     * 寻找移动最佳停留的位置
     */
    private fun findDockPosition() {
        val params = layoutParams as WindowManager.LayoutParams
        val x = params.x
        val width = params.width
        val centerX = x + width / 2

        val outSize = Point()
        windowManager.defaultDisplay.getSize(outSize)

        val screenWidth = outSize.x

        val newX = if (centerX > screenWidth / 2) {
            screenWidth - width
        } else {
            0
        }

        params.x = newX

        try {
            windowManager.updateViewLayout(this, params)
        } catch (e: Exception) {
            e.printStackTrace()
        }

        context.floatingSavePosition(newX, params.y)
    }

    /**
     * 更新视图位置
     */
    private fun updateViewLayout() {

        val params = layoutParams as WindowManager.LayoutParams
        params.x = selfX.toInt()
        params.y = selfY.toInt()
        try {
            windowManager.updateViewLayout(this, params)
        } catch (e: Throwable) {
            e.printStackTrace()
        }

    }

    /**
     * 始终保持球在手指左上角
     */
    private fun toLeftBottom() {

        //始终保持球在手指左上角 START

        val wx = selfX + width
        val wy = selfY + height

        selfX -= wx - lastX
        selfY -= wy - lastY
        //始终保持球在手指左上角 END
    }


    /**
     * 识别器相对于屏幕的中心位置
     */
    private fun recognizerLocation(): IntArray {

        recognizerView.getLocationOnScreen(recognizerCenterPoint)

        recognizerCenterPoint[0] = recognizerCenterPoint[0] + recognizerView.width / 2
        recognizerCenterPoint[1] =
            recognizerCenterPoint[1] /*+ recognizerView.height*/ + recognizerView.height / 2

        return recognizerCenterPoint
    }


    /**
     * 启动服务，遍历节点信息
     */
    private fun startAccessService() {
        AccessService.startServiceForDown(context)
    }


    /**
     * 添加矩形背景
     */
    @Synchronized
    private fun addRectView(x: Int, y: Int) {

        val params = overlayParams(rectView.layoutParams as? WindowManager.LayoutParams)
        params.x = x
        params.y = y
        params.width = 0
        params.height = 0
        try {
            if (!rectView.isAttachedToWindow && rectView.parent == null) {
                windowManager.addView(rectView, params)
            } else {
                windowManager.updateViewLayout(rectView, params)
            }
        } catch (e: Exception) {
            e.printStackTrace()
        }

        (rectView as BaseOverlayView).setOnBackClickListener =
            object : BaseOverlayView.BackClickListener {
                override fun onBackClickListener(): Boolean {
                    try {
                        if (rectView.isAttachedToWindow) {
                            windowManager.removeView(rectView)
                            return true
                        } else {
                            return false
                        }
                    } catch (e: Exception) {
                        e.printStackTrace()
                    }
                    return false
                }

            }
    }

    /**
     * 移除矩形背景
     */
    private fun removeRectView() {
        try {
            if (rectView.isAttachedToWindow) {
                windowManager.removeView(rectView)
            }
        } catch (e: Exception) {
            e.printStackTrace()
        }
        rectView.setTag(R.id.id_content_view_node_info, null)
    }


    /**
     * 更新矩形背景
     */
    private fun updateRectView(nodeInfo: AccessibilityNodeInfo) {
        if (!rectView.isAttachedToWindow) {
            return
        }
        nodeInfo.getBoundsInScreen(rectInScreen)
        val params = rectView.layoutParams as WindowManager.LayoutParams
        params.x = rectInScreen.left
        params.y = rectInScreen.top
        params.width = rectInScreen.width()
        params.height = rectInScreen.height()
        //params.windowAnimations = android.R.style.Animation_Activity
        rectView.setBackgroundResource(R.drawable.shape_rect_view_bg)

        if (rectView.visibility != View.VISIBLE) {
            rectView.visibility = View.VISIBLE
        }
        try {
            windowManager.updateViewLayout(rectView, params)
        } catch (e: Exception) {
            e.printStackTrace()
        }

    }

    private fun showEditProgressBar() {
        val progressBar by lazy { rectView.findViewById<ContentLoadingProgressBar>(R.id.loading_progress_bar) }
        progressBar.show()
    }

    private fun hideEditProgressBar() {
        val progressBar by lazy { rectView.findViewById<ContentLoadingProgressBar>(R.id.loading_progress_bar) }
        progressBar.hide()
    }

    /**
     * 通过侦测视图位置(建议中心点位置)，寻找相应节点
     */
    @Synchronized
    private fun findNodeByPosition(
        cx: Int,
        cy: Int,
        nodeSet: LinkedHashSet<AccessibilityNodeInfo>
    ): AccessibilityNodeInfo? {

        synchronized(lock) {
            val nodes = nodeSet.filter { nodeInfo ->
                nodeInfo.getBoundsInScreen(rectInScreen)
                rectInScreen.contains(cx, cy)
            }
            var minRect: Rect? = null
            var nodeInfo: AccessibilityNodeInfo? = null
            for (node in nodes) {

                val rect = Rect()
                node.getBoundsInScreen(rect)

                if (minRect == null || minRect.contains(rect) || rect == minRect) {
                    minRect = rect
                    nodeInfo = node
                }
            }

            return nodeInfo
        }
    }


    /**
     * 添加内容视图
     */
    private fun addContentView(x: Int, y: Int) {
        /*if (readShowTips1()) {
            firstOverlayView.addContentView(x, y)
            return
        }*/
        when (translationStyle(readPrefer = true)) {
            OVERLAY_MODE_GENERAL -> {
                generalOverlayView.addContentView(x, y)
            }
            OVERLAY_MODE_SIMPLE -> {
                simpleOverlayView.addContentView(x, y)
            }
            OVERLAY_MODE_SPEECH -> {
                speechOverlayView.addContentView(x, y)
            }
        }

    }

    /**
     * 更新内容视图
     */
    private fun updateContentView(nodeInfo: AccessibilityNodeInfo) {
        /*if (readShowTips1()) {
            firstOverlayView.updateContentView(nodeInfo)
            return
        }*/
        when (translationStyle()) {
            OVERLAY_MODE_GENERAL -> {
                generalOverlayView.updateContentView(nodeInfo)
            }
            OVERLAY_MODE_SIMPLE -> {
                simpleOverlayView.updateContentView(nodeInfo)
            }
            OVERLAY_MODE_SPEECH -> {
                speechOverlayView.updateContentView(nodeInfo)
            }
        }

    }


    /**
     * 移除内容视图
     */
    private fun removeContentView() {

        /*if (readShowTips1()) {
            firstOverlayView.removeContentView()
            return
        }*/
        when (translationStyle()) {
            OVERLAY_MODE_GENERAL -> {
                generalOverlayView.removeContentView()
            }
            OVERLAY_MODE_SIMPLE -> {
                simpleOverlayView.removeContentView()
            }
            OVERLAY_MODE_SPEECH -> {
                speechOverlayView.removeContentView()
            }
        }

    }

    /**
     * translation style
     */
    private fun translationStyle(readPrefer: Boolean = false): Int {
        if (readPrefer || currentOverlayStyle == -1) {
            //val sharedName = "${context.packageName}_preferences"
            //val prefer = context.getSharedPreferences(sharedName, Context.MODE_PRIVATE)
            /*
            <item>simple</item>
            <item>general</item>
            <item>speech</item>
            */

            val style = readTranslationStyleValue()
            currentOverlayStyle = when (style) {
                "simple" -> OVERLAY_MODE_SIMPLE
                "general" -> OVERLAY_MODE_GENERAL
                else -> OVERLAY_MODE_SPEECH
            }
        }
        return currentOverlayStyle
    }

    /**
     * 0 <item>simple</item>
     * 1 <item>general</item>
     * 2 <item>speech</item>
     */
    private fun readTranslationStyleValue(): String {
        if (ActivityManager.isUserAMonkey()) {
            return "simple"
        }
        val prefer = PreferenceManager.getDefaultSharedPreferences(context)
        return prefer.getString("key_translate_style", "simple") ?: "simple"
        //return KeyValue.get("prefer_trans_result_style") ?: "simple"
    }


    /**
     * 悬浮视图Params
     */
    private fun overlayParams(layoutParams: WindowManager.LayoutParams? = null): WindowManager.LayoutParams {
        val type = if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O)
            WindowManager.LayoutParams.TYPE_APPLICATION_OVERLAY
        else WindowManager.LayoutParams.TYPE_PHONE
        val flags = WindowManager.LayoutParams.FLAG_WATCH_OUTSIDE_TOUCH or
                WindowManager.LayoutParams.FLAG_NOT_TOUCH_MODAL or
                WindowManager.LayoutParams.FLAG_NOT_FOCUSABLE or
                WindowManager.LayoutParams.FLAG_LAYOUT_IN_SCREEN or
                WindowManager.LayoutParams.FLAG_HARDWARE_ACCELERATED
        val params =
            layoutParams ?: WindowManager.LayoutParams(type, flags, PixelFormat.TRANSLUCENT)
        //
        params.x = 0
        params.y = 0
        params.width = 0
        params.height = 0

        params.gravity = Gravity.START or Gravity.TOP

        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.P) {
            params.layoutInDisplayCutoutMode =
                WindowManager.LayoutParams.LAYOUT_IN_DISPLAY_CUTOUT_MODE_SHORT_EDGES
        }

        params.format = PixelFormat.RGBA_8888
        return params
    }


    /**
     * 粘贴板点击事件
     */
    private fun dispatchClip() {

        val clipboardManager =
            context.getSystemService(Context.CLIPBOARD_SERVICE) as ClipboardManager
        val primaryClip = clipboardManager.primaryClip
        val clipText = if (primaryClip?.itemCount ?: 0 > 0) {
            primaryClip?.getItemAt(0)?.text?.toString()
        } else ""

        //android.intent.action.PROCESS_TEXT
        val intent = Intent("android.intent.action.PROCESS_TEXT").apply {
            action = "android.intent.action.PROCESS_TEXT"
            setPackage(context.packageName)
        }


        intent.component = ComponentName(
            context.packageName,
            "com.talpa.translate.ui.translateChooseText.TranslateChooseTextActivity"
        )
        intent.putExtra("action_type", 1)
        intent.putExtra("android.intent.extra.PROCESS_TEXT", clipText)
        intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK)

        if (intent.resolveActivity(context.packageManager) != null) {
            context.startActivity(intent)
        } else {
            Toast.makeText(context.applicationContext, "尚未集成复制翻译", Toast.LENGTH_SHORT)
                .show()
        }

        FloatingStateMachine.sendMessage(WHAT_LIGHT_ENTER)
    }


    override fun onConfigurationChanged(newConfig: Configuration?) {
        super.onConfigurationChanged(newConfig)

        /*if (newConfig != null) {
            orientation(newConfig)
        }*/
        removeContentView()
        removeRectView()
        val point = context.floatingResetDefaultPosition()
        selfX = point.x.toFloat()
        selfY = point.y.toFloat()
        updateViewLayout()


        FloatingManager.onConfigurationChanged(newConfig)
    }

    /**
     * 手势
     */
    private val onGestureListener = object : GestureDetector.SimpleOnGestureListener() {

        override fun onDoubleTapEvent(e: MotionEvent?): Boolean {
            // println("SimpleOnGestureListener#onDoubleTapEvent MotionEvent=${e?.action}")

            when (e?.action) {
                ACTION_UP -> {//双击松开
                    when (FloatingStateMachine.currentState()) {
                        is FloatingStateMachine.FloatingState.StaticState -> {
                            if (!mAccessServiceEnable) {
                                mSubject.onNext(WHAT_ASK_PERMISSION_WHOLE)
                            } else {
                                FloatingStateMachine.sendMessage(FloatingStateMachine.WHAT_HIGHLIGHT_ENTER)
                            }
                        }
                        is FloatingStateMachine.FloatingState.HighLightState -> {
                            FloatingStateMachine.sendMessage(WHAT_LIGHT_ENTER)
                        }
                    }
                }
            }
            return super.onDoubleTapEvent(e)
        }

        override fun onScroll(
            e1: MotionEvent?,
            e2: MotionEvent?,
            distanceX: Float,
            distanceY: Float
        ): Boolean {
            if (BuildConfig.DEBUG) println("SimpleOnGestureListener#onScroll#MotionEvent1=${e1?.action}   MotionEvent2=${e2?.action}\n distanceX=$distanceX  distanceY=$distanceY")
            when (FloatingStateMachine.currentState()) {
                is FloatingStateMachine.FloatingState.StaticState -> {
                    FloatingStateMachine.sendMessage(FloatingStateMachine.WHAT_DETECT_ENTER)
                }
            }

            return super.onScroll(e1, e2, distanceX, distanceY)
        }

        fun onScrollUp(e: MotionEvent?) {

        }

        override fun onSingleTapConfirmed(e: MotionEvent?): Boolean {
            // println("SimpleOnGestureListener#onSingleTapConfirmed MotionEvent=${e?.action}")

            when (FloatingStateMachine.currentState()) {
                is FloatingStateMachine.FloatingState.StaticState.DarkState,
                is FloatingStateMachine.FloatingState.StaticState.LightState -> {
                    FloatingStateMachine.sendMessage(FloatingStateMachine.WHAT_MENU_ENTER)
                }
                is FloatingStateMachine.FloatingState.StaticState.IdleState -> {

                    val touchX = e?.rawX ?: 0f
                    val width = this@FloatingContainer.width
                    val fx = (this@FloatingContainer.layoutParams as WindowManager.LayoutParams).x

                    val offsetWidth = width / 2f

                    if (touchX < offsetWidth) {
                        FloatingStateMachine.sendMessage(FloatingStateMachine.WHAT_MENU_ENTER)
                    } else if (touchX > width && touchX - fx + offsetWidth > width) {
                        FloatingStateMachine.sendMessage(FloatingStateMachine.WHAT_MENU_ENTER)
                    }

                    //FloatingStateMachine.sendMessage(FloatingStateMachine.WHAT_MENU_ENTER)
                }
                is FloatingStateMachine.FloatingState.HighLightState -> {
                    //与onSingleTapConfirmed　事件冲突，需要延迟
                    /*postDelayed({
                        FloatingStateMachine.sendMessage(FloatingStateMachine.WHAT_LIGHT_ENTER)
                    }, 1000)*/
                    FloatingStateMachine.sendMessage(FloatingStateMachine.WHAT_LIGHT_ENTER)
                }
                is FloatingStateMachine.FloatingState.CopyState -> {

                    dispatchClip()
                }
            }

            return super.onSingleTapConfirmed(e)
        }


        override fun onLongPress(e: MotionEvent?) {
            if (BuildConfig.DEBUG) println("SimpleOnGestureListener#onLongPress MotionEvent=${e?.action}")
            super.onLongPress(e)
            FloatingStateMachine.sendMessage(FloatingStateMachine.WHAT_FIND_LOCATION_ENTER)
        }


        override fun onFling(
            e1: MotionEvent?,
            e2: MotionEvent?,
            velocityX: Float,
            velocityY: Float
        ): Boolean {
            if (BuildConfig.DEBUG) println("SimpleOnGestureListener#onFling #MotionEvent1=${e1?.action}   MotionEvent2=${e2?.action}")
            when (e2?.action) {
                ACTION_UP -> {
                    FloatingStateMachine.sendMessage(WHAT_LIGHT_ENTER)
                }
            }
            return super.onFling(e1, e2, velocityX, velocityY)
        }
    }


    private fun px2dp(px: Int): Float = px / context.resources.displayMetrics.density + 0.5f


    /**
     * 打点开始翻译
     */
    private fun logStartTranslate(
        sourceText: String,
        targetLanguageLocaleTag: String,
        packageName: String
    ) {

        val intent = Intent(OverlayImpl.ACTION_EDIT_TEXT_TRANSLATE_START).apply {
            putExtra(OverlayImpl.EXTRA_TEXT, sourceText)
            putExtra(OverlayImpl.EXTRA_TARGET_LANGUAGE, targetLanguageLocaleTag)
            putExtra(OverlayImpl.EXTRA_PACKAGE_NAME, packageName)
        }

        localBroadcastManager.sendBroadcast(intent)
    }


    /**
     * 打点开始翻译
     */
    private fun logSuccessTranslate(
        sourceText: String,
        sourceLanguage: String,
        targetLanguage: String,
        packageName: String,
        isCached: Boolean = false,
        moduleType: String
    ) {

        val intent = Intent(OverlayImpl.ACTION_TEXT_SCREEN_TRANSLATE_SUCCESS).apply {
            putExtra(OverlayImpl.EXTRA_TEXT, sourceText)
            putExtra(OverlayImpl.EXTRA_SOURCE_TARGET_LANGUAGE, "$sourceLanguage&&$targetLanguage")
            putExtra(OverlayImpl.EXTRA_PACKAGE_NAME, packageName)
            putExtra(OverlayImpl.EXTRA_CACHE, isCached)
            putExtra(OverlayImpl.EXTRA_MODULE_TYPE, moduleType)
        }

        localBroadcastManager.sendBroadcast(intent)
    }

    /**
     * 打点开始翻译
     */
    private fun logFailTranslate(
        sourceText: String,
        targetLanguage: String,
        packageName: String
    ) {

        val intent = Intent(OverlayImpl.ACTION_EDIT_TEXT_TRANSLATE_ERROR).apply {
            putExtra(OverlayImpl.EXTRA_TEXT, sourceText)
            putExtra(OverlayImpl.EXTRA_TARGET_LANGUAGE, targetLanguage)
            putExtra(OverlayImpl.EXTRA_PACKAGE_NAME, packageName)
        }

        localBroadcastManager.sendBroadcast(intent)
    }
}