package com.talpa.overlay.state

import android.os.Message
import android.util.Log
import com.talpa.overlay.RxRelay.EVENT_COPY_ENTER
import com.talpa.overlay.RxRelay.EVENT_DARK_ENTER
import com.talpa.overlay.RxRelay.EVENT_DARK_EXIT
import com.talpa.overlay.RxRelay.EVENT_DETECT_ENTER
import com.talpa.overlay.RxRelay.EVENT_DETECT_EXIT
import com.talpa.overlay.RxRelay.EVENT_FIND_LOCATION_ENTER
import com.talpa.overlay.RxRelay.EVENT_HIGHLIGHT_ENTER
import com.talpa.overlay.RxRelay.EVENT_HIGHLIGHT_EXIT
import com.talpa.overlay.RxRelay.EVENT_IDLE_ENTER
import com.talpa.overlay.RxRelay.EVENT_IDLE_EXIT
import com.talpa.overlay.RxRelay.EVENT_INVISIBLE_ENTER
import com.talpa.overlay.RxRelay.EVENT_LIGHT_ENTER
import com.talpa.overlay.RxRelay.EVENT_LIGHT_EXIT
import com.talpa.overlay.RxRelay.EVENT_MENU_ENTER
import com.talpa.overlay.RxRelay.EVENT_MENU_EXIT
import com.talpa.overlay.RxRelay.postByEventBus
import com.talpa.overlay.state.framework.IState
import com.talpa.overlay.state.framework.State
import com.talpa.overlay.state.framework.StateMachine

/**
 * 悬浮视图状态机
 * @author CY 19-8-14
 */

internal object FloatingStateMachine : StateMachine("FloatingMachine") {

    //悬浮球状态事件
    const val WHAT_IDLE_ENTER = 100
    const val WHAT_LIGHT_ENTER = 110
    const val WHAT_DARK_ENTER = 120
    const val WHAT_DETECT_ENTER = 130
    const val WHAT_FIND_LOCATION_ENTER = 140
    const val WHAT_MENU_ENTER = 150
    const val WHAT_HIGHLIGHT_ENTER = 160
    const val WHAT_INVISIBLE_ENTER = 170
    const val WHAT_COPY_ENTER = 180

    private val lightState = FloatingState.StaticState.LightState()
    private val idleState = FloatingState.StaticState.IdleState()
    private val darkState = FloatingState.StaticState.DarkState()
    private val findLocationState = FloatingState.WorkingState.FindLocationState()
    private val detectState = FloatingState.WorkingState.DetectState()
    private val menuState = FloatingState.MenuState()
    private val highLightState = FloatingState.HighLightState()
    private val invisibleState = FloatingState.InvisibleState()
    private val copyState = FloatingState.CopyState()

    var grammarCheckEnable = false

    init {

        isDbg = false

        if (isDbg) log(message = "FloatingStateMachine#init")
        addState(idleState)
        addState(lightState)//,FloatingState.StaticState.IdleState
        addState(darkState)// FloatingState.StaticState.LightState

        addState(findLocationState)//,FloatingState.StaticState.IdleState
        addState(detectState)

        addState(menuState)
        addState(highLightState)
        addState(invisibleState)
        addState(copyState)

        setInitialState(invisibleState)
        //start()
    }

    /**
     *
     */
    private var mCurrentState: IState? = null
    private var mLastState: IState? = null

    /**
     * 当前状态
     */
    fun currentState(): IState? {

        try {
            return currentState
        } catch (e: Exception) {
            e.printStackTrace()
        }

        return null
    }

    /**
     *
     */
    fun lastState() = mLastState

    /**
     * 启动状态机
     */
    fun setupStateMachine() {

        start()
    }

    /**
     * 停止状态机
     */
    fun stopStateMachine() {
        quit()
    }

    /*　State START*/

    /**
     * 悬浮球状态
     */
    sealed class FloatingState : State() {

        override fun enter() {
            super.enter()
            mCurrentState = this
        }

        override fun exit() {
            super.exit()
            mLastState = this
        }

        protected fun postMessage(what: Int) {
            //  post(Message.obtain(null, what))
            postByEventBus(Message.obtain(null, what))
        }

        /**
         * Static State
         */
        sealed class StaticState : FloatingState() {

            override fun processMessage(msg: Message?): Boolean {

                when (msg?.what) {
                    WHAT_INVISIBLE_ENTER -> {
                        transitionTo(invisibleState)
                    }
                    WHAT_FIND_LOCATION_ENTER -> {
                        transitionTo(findLocationState)
                    }
                    WHAT_DETECT_ENTER -> {
                        transitionTo(detectState)
                    }
                    WHAT_HIGHLIGHT_ENTER -> {
                        transitionTo(highLightState)
                    }
                    WHAT_MENU_ENTER -> {
                        transitionTo(menuState)
                    }
                    WHAT_COPY_ENTER -> {
                        transitionTo(copyState)
                    }
                }

                return super.processMessage(msg)
            }


            /**
             * IDLE State
             */
            open class IdleState : StaticState() {

                override fun enter() {
                    super.enter()
                    if (isDbg) {
                        log(message = "IdleState#enter")
                    }

                    postMessage(EVENT_IDLE_ENTER)
                }

                override fun exit() {
                    super.exit()
                    if (isDbg) {
                        log(message = "IdleState#exit")
                    }
                    postMessage(EVENT_IDLE_EXIT)
                }

                override fun processMessage(msg: Message?): Boolean {
                    if (isDbg) {
                        log(message = "IdleState#processMessage")
                    }
                    return super.processMessage(msg)
                }

            }

            /**
             * Light State
             */
            open class LightState : StaticState() {
                override fun enter() {
                    super.enter()
                    if (isDbg) {
                        log(message = "LightState#enter")
                    }
                    postMessage(EVENT_LIGHT_ENTER)

                    // deferMessage(Message.obtain(handler, WHAT_DARK_ENTER))
                }

                override fun processMessage(msg: Message?): Boolean {
                    if (isDbg) {
                        log(message = "LightState#processMessage")
                    }

                    when (msg?.what) {
                        WHAT_DARK_ENTER -> {
                            transitionTo(darkState)
                        }
                    }

                    return super.processMessage(msg)
                }

                override fun exit() {
                    super.exit()
                    if (isDbg) {
                        log(message = "LightState#exit")
                    }

                    postMessage(EVENT_LIGHT_EXIT)
                }
            }


            /**
             * Dark State
             */
            class DarkState : StaticState() {
                override fun enter() {
                    super.enter()
                    if (isDbg) {
                        log(message = "DarkState#enter")
                    }
                    postMessage(EVENT_DARK_ENTER)
                }

                override fun exit() {
                    super.exit()
                    postMessage(EVENT_DARK_EXIT)
                }

                override fun processMessage(msg: Message?): Boolean {
                    if (isDbg) {
                        log(message = "LightState#processMessage")
                    }

                    when (msg?.what) {
                        WHAT_IDLE_ENTER -> {
                            transitionTo(idleState)
                        }
                    }

                    return super.processMessage(msg)
                }
            }

        }



        /**
         * Working State
         */
        sealed class WorkingState : FloatingState() {


            override fun processMessage(msg: Message?): Boolean {
                if (isDbg) {
                    log(message = "WorkingState#processMessage")
                }
                when (msg?.what) {
                    WHAT_LIGHT_ENTER -> {
                        transitionTo(lightState)
                    }
                }

                return super.processMessage(msg)
            }

            /**
             * Detect State
             */
            open class DetectState : WorkingState() {
                override fun enter() {
                    super.enter()
                    if (isDbg) {
                        log(message = "DetectState#enter")
                    }
                    postMessage(EVENT_DETECT_ENTER)
                }

                override fun exit() {
                    super.exit()
                    postMessage(EVENT_DETECT_EXIT)
                }
            }

            /**
             * FindLocation State
             */
            class FindLocationState : WorkingState() {
                override fun enter() {
                    super.enter()
                    if (isDbg) {
                        log(message = "FindLocationState#enter")
                    }
                    postMessage(EVENT_FIND_LOCATION_ENTER)
                }
            }
        }

        /**
         * Menu State
         */
        class MenuState : FloatingState() {
            override fun enter() {
                super.enter()
                if (isDbg) {
                    log(message = "MenuState#enter")
                }
                postMessage(EVENT_MENU_ENTER)
            }

            override fun processMessage(msg: Message?): Boolean {
                if (isDbg) {
                    log(message = "MenuState#processMessage")
                }
                when (msg?.what) {
                    WHAT_LIGHT_ENTER -> {
                        transitionTo(lightState)
                    }
                    WHAT_FIND_LOCATION_ENTER -> {
                        transitionTo(findLocationState)
                    }
                    WHAT_INVISIBLE_ENTER -> {
                        transitionTo(invisibleState)
                    }
                    WHAT_HIGHLIGHT_ENTER -> {
                        transitionTo(highLightState)
                    }
                }

                return super.processMessage(msg)
            }

            override fun exit() {
                super.exit()
                if (isDbg) {
                    log(message = "MenuState#exit")
                }
                postMessage(EVENT_MENU_EXIT)
            }
        }

        /**
         * HighLight State
         */
        class HighLightState : FloatingState() {
            override fun enter() {
                super.enter()
                if (isDbg) {
                    log(message = "HighLightState#enter")
                }
                postMessage(EVENT_HIGHLIGHT_ENTER)
            }

            override fun exit() {
                super.exit()
                postMessage(EVENT_HIGHLIGHT_EXIT)
            }

            override fun processMessage(msg: Message?): Boolean {
                if (isDbg) {
                    log(message = "InvisibleState#processMessage")
                }

                when (msg?.what) {
                    WHAT_LIGHT_ENTER -> {
                        transitionTo(lightState)
                    }
                    WHAT_INVISIBLE_ENTER -> {
                        transitionTo(invisibleState)
                    }
                }
                return super.processMessage(msg)
            }

        }

        /**
         * Invisible State
         */
        class InvisibleState : FloatingState() {
            override fun enter() {
                super.enter()
                if (isDbg) {
                    log(message = "InvisibleState#enter")
                }
                postMessage(EVENT_INVISIBLE_ENTER)
            }

            override fun processMessage(msg: Message?): Boolean {
                if (isDbg) {
                    log(message = "InvisibleState#processMessage")
                }

                when (msg?.what) {
                    WHAT_LIGHT_ENTER -> {
                        transitionTo(lightState)
                    }
                    WHAT_COPY_ENTER -> {
                        transitionTo(copyState)
                    }
                }
                return super.processMessage(msg)
            }

            override fun exit() {
                super.exit()
                if (isDbg) {
                    log(message = "InvisibleState#exit")
                }
            }

        }

        /**
         * 复制状态
         */
        class CopyState : FloatingState() {
            override fun enter() {
                super.enter()
                if (isDbg) {
                    log(message = "InvisibleState#enter")
                }
                postMessage(EVENT_COPY_ENTER)
            }

            override fun processMessage(msg: Message?): Boolean {
                if (isDbg) {
                    log(message = "CopyState#processMessage")
                }

                when (msg?.what) {
                    WHAT_LIGHT_ENTER -> {
                        transitionTo(lightState)
                    }
                }
                return super.processMessage(msg)
            }

            override fun exit() {
                super.exit()
                if (isDbg) {
                    log(message = "CopyState#exit")
                }
            }
        }
    }

    /*　State END*/
}

fun log(tag: String = "Access", message: String) {
    Log.i(tag, message)
}
