package com.talpa.translate.camera


import android.annotation.SuppressLint
import android.app.Activity
import android.content.ActivityNotFoundException
import android.content.Context
import android.content.DialogInterface
import android.content.Intent
import android.graphics.Bitmap
import android.graphics.Typeface
import android.graphics.drawable.Drawable
import android.net.Uri
import android.os.Bundle
import android.text.TextUtils
import android.view.LayoutInflater
import android.view.MotionEvent
import android.view.View
import android.widget.CheckBox
import android.widget.Checkable
import android.widget.ImageView
import android.widget.Toast
import androidx.activity.OnBackPressedCallback
import androidx.activity.result.contract.ActivityResultContract
import androidx.annotation.RestrictTo
import androidx.appcompat.app.AlertDialog
import androidx.core.content.ContextCompat
import androidx.core.graphics.drawable.toBitmap
import androidx.core.os.LocaleListCompat
import androidx.core.view.forEachIndexed
import androidx.fragment.app.Fragment
import androidx.fragment.app.viewModels
import androidx.lifecycle.*
import androidx.lifecycle.Observer
import androidx.viewpager.widget.ViewPager
import com.airbnb.lottie.LottieDrawable
import com.bumptech.glide.Glide
import com.bumptech.glide.load.DataSource
import com.bumptech.glide.load.engine.GlideException
import com.bumptech.glide.request.RequestListener
import com.bumptech.glide.request.target.Target
import com.fasterxml.jackson.databind.ObjectMapper
import com.google.android.material.bottomsheet.BottomSheetDialog
import com.google.android.material.snackbar.Snackbar
import com.gyf.immersionbar.BarHide
import com.gyf.immersionbar.ImmersionBar
import com.gyf.immersionbar.ktx.immersionBar
import com.photo.translation.R
import com.photo.translation.databinding.FragmentCameraBinding
import com.talpa.tengine.lang.LANG
import com.talpa.translate.base.adapter.Feature
import com.talpa.translate.base.adapter.FeatureAdapter
import com.talpa.translate.base.state.PhotoMLState
import com.talpa.translate.base.state.PhotoMLState.Companion.TYPE_OBJECT
import com.talpa.translate.base.state.PhotoMLState.Companion.TYPE_TEXT
import com.talpa.translate.base.utils.*
import com.talpa.translate.base.view.wheel.WheelView
import com.talpa.translate.camera.view.gesture.ViewPageGestureWrapper
import com.talpa.translate.camera.view.preview.ViewTouchListener
import com.talpa.translate.datasource.CameraCompatSource
import com.talpa.translate.datasource.ICameraSource
import com.talpa.translate.imglabel.ImageLabelData
import com.talpa.translate.ocr.CombineViewModel
import com.talpa.translate.ocr.ImageTranslate
import com.talpa.translate.ocr.OcrDispatcher
import com.talpa.translate.ocr.Utils
import com.talpa.translate.ocr.datasource.CompleteResult
import com.talpa.translate.ocr.exception.NoContentException
import com.talpa.translate.ocr.result.CompleteTransfer
import com.talpa.translate.ocr.result.ContrastActivity
import com.talpa.translate.repository.box.collins.SenseNew
import com.talpa.translate.repository.db.TranslateHistory
import com.tapla.textspeech.TextSpeech
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.flow.flow
import kotlinx.coroutines.flow.flowOn
import java.util.*


/**
 * Camera Preview
 *
 * @author CY 2020-02-03
 */

const val COMPLETE_RESULT = "complete_result"

@RestrictTo(RestrictTo.Scope.LIBRARY)
class CameraFragment : Fragment(R.layout.fragment_camera), View.OnClickListener,
    ICameraSource.Status, ViewTouchListener {

    internal lateinit var binding: FragmentCameraBinding

    /**
     * 当前状态
     */
    internal var currentStatus = STATE_CAMERA

    internal var mCompleteResult: CompleteResult? = null
    internal var mItemRecognizeResult: ImageLabelData? = null
    private var mViewPageGestureWrapper: ViewPageGestureWrapper? = null

    internal val translateViewModel: ImageTranslate by lazy {
        ViewModelProvider(
            this,
            ViewModelProvider.AndroidViewModelFactory.getInstance(activity?.application!!)
        ).get(ImageTranslate::class.java)
    }

    internal lateinit var mCameraSource: CameraCompatSource

    companion object {

        /*Camera state*/
        const val STATE_CAMERA = 100

        const val STATE_SELECT_IMAGE = 300

        const val STATE_RESULT_FROM_CAMERA_SUCCESS = 400
        const val STATE_RESULT_FROM_GALLERY_SUCCESS = 401

        const val STATE_RESULT_FROM_CAMERA_FAIL = 402
        const val STATE_RESULT_FROM_GALLERY_FAIL = 403

        const val STATE_TRANSLATING_CAMERA = 500
        const val STATE_TRANSLATING_GALLERY = 501

        /*const val REQUEST_CODE_SELECT_IMAGE = 1000
        const val REQUEST_CODE_SELECT_SOURCE_LANGUAGE = 1100
        const val REQUEST_CODE_SELECT_TARGET_LANGUAGE = 1200*/

        const val EXTRA_DATA = "data"
        const val ACTION_FOR_DICTIONARY_TRANS =
            "com.talpa.translate.ACTION_FOR_DICTIONARY_TRANS_DETAIL"
    }

    internal var mState: PhotoMLState = PhotoTranslateState()
        set(value) {
            field.onStateChange(false)
            value.onStateChange(true)
            field = value
        }
    internal val photoTranslateState = PhotoTranslateState()
    internal val itemRecognizeState = ItemRecognizeState()
    internal var mFeatureAdapter: FeatureAdapter? = null

    private fun cleanUp() {
        mCompleteResult?.renderResult = null
        mCompleteResult = null
        mItemRecognizeResult?.sourceBitmap = null
        mItemRecognizeResult = null
        mPreviewBitmap = null
    }

    abstract inner class AbstractState : PhotoMLState {
        override fun enterCameraState() {
            if (screenShotMode) {
                activity?.finish()
                return
            }
            cleanUp()

            currentStatus = STATE_CAMERA
            binding.btnControl.visibility = View.VISIBLE
            binding.btnControl.setState(binding.btnControl.cameraState)
            binding.btnControl.revertAnim()
            binding.btnControl.isClickable = true

            binding.overlay.visibility = View.GONE
            binding.ivPreview.visibility = View.INVISIBLE

            binding.torchSelector.setOnClickListener(this@CameraFragment)

            binding.torchSelector.setButtonDrawable(R.drawable.selector_torch)
            binding.ivSelectImage.setImageResource(R.drawable.icon_gallery)

            hideProgressBar()

            binding.tvSourceLanguage.isClickable = true
            binding.tvTargetLanguage.isClickable = true
            binding.tabLayout.isEnabled = true

            binding.tvSourceLanguage.setTextColor(
                ContextCompat.getColor(
                    requireContext(),
                    R.color.camera_tool_bg
                )
            )
            binding.tvTargetLanguage.setTextColor(
                ContextCompat.getColor(
                    requireContext(),
                    R.color.camera_tool_bg
                )
            )
            binding.ivExchange.alpha = 1.0f
            binding.ivExchange.isClickable = true
            mCameraSource.torch(mPreTorchState)
            //binding.ivPreview.attacher.resetMatrix()
            //binding.overlay.attacher.resetMatrix()
        }

        protected open fun enterResultState(isSuccess: Boolean) {
            hideProgressBar()
            binding.tabLayout.isEnabled = false
            binding.tvSourceLanguage.isClickable = true
            binding.tvTargetLanguage.isClickable = true
            binding.btnControl.revertAnim()
            binding.btnControl.setState(binding.btnControl.resultState)
            binding.btnControl.isClickable = true
            binding.btnControl.visibility = View.VISIBLE
            binding.languageSelector.background?.alpha = 0xff
            binding.languageSelector.isClickable = true
            binding.tvSourceLanguage.setTextColor(
                ContextCompat.getColor(
                    requireContext(),
                    R.color.camera_tool_bg
                )
            )
            binding.tvTargetLanguage.setTextColor(
                ContextCompat.getColor(
                    requireContext(),
                    R.color.camera_tool_bg
                )
            )
            binding.ivExchange.alpha = 1.0f
            binding.ivExchange.isClickable = true
        }

    }

    inner class PhotoTranslateState : AbstractState() {

        override fun onStateChange(focus: Boolean) {
            if (focus) {
                translateViewModel.sourceLiveData.observe(
                    this@CameraFragment,
                    Observer { language ->
                        binding.tvSourceLanguage.text =
                            Locale.forLanguageTag(language).displayLanguage
                        //exchangeState()
                    })
                translateViewModel.targetLiveData.observe(
                    this@CameraFragment,
                    Observer { language ->
                        binding.tvTargetLanguage.text =
                            Locale.forLanguageTag(language).displayLanguage
                        //exchangeState()
                    })
            } else {
                translateViewModel.sourceLiveData.removeObservers(this@CameraFragment)
                translateViewModel.targetLiveData.removeObservers(this@CameraFragment)
            }
        }

        override fun getType(): Int {
            return TYPE_TEXT
        }

        override fun onPictureTaken(
            bitmap: Bitmap,
            sourceLanguage: String,
            targetLanguage: String,
            rotation: Int
        ) {
            translateImage(
                bitmap,
                sourceLanguage,
                targetLanguage,
                rotation
            )?.observe(viewLifecycleOwner, Observer {
                if (it.isSuccess) {
                    mCompleteResult = it.getOrNull()
                    val completeBitmap = mCompleteResult?.renderResult
                    recognitionSuccess(completeBitmap)
                } else {
                    recognitionFailure(it.exceptionOrNull())
                }
                enterResultState(it.isSuccess)
            })
            logEvent(
                Trans_start_translate,
                hashMapOf(
                    language to "$sourceLanguage-$targetLanguage",
                    moduleType to modulePic
                )
            )
        }

        override fun enterCameraState() {
            super.enterCameraState()
            binding.torchSelector.visibility = View.VISIBLE
            binding.ivSelectImage.visibility = View.VISIBLE
            binding.ivSelectImage.setOnClickListener(this@CameraFragment)
            binding.include.root.visibility = View.GONE
            readLanguage()
            binding.ivExchange.isEnabled = judgeExchangeEnable(
                Locale.forLanguageTag(translateViewModel.getSourceData(this).value ?: return),
                Locale.forLanguageTag(translateViewModel.getTargetData(this).value ?: return),
                getSupportSourceLanguage(),
                getSupportTargetLanguage()
            )
            /*binding.overlay.setOnScaleChangeListener { scaleFactor, focusX, focusY ->
                binding.ivPreview.setScale(binding.overlay.scale, focusX, focusY, false)
                //binding.ivPreview.setDisplayMatrix(binding.overlay.imageMatrix, false)
            }
            binding.ivPreview.setOnScaleChangeListener { scaleFactor, focusX, focusY ->
                binding.overlay.setScale(binding.ivPreview.scale, focusX, focusY, false)
                //binding.overlay.setDisplayMatrix(binding.ivPreview.imageMatrix, false)
            }*/
            binding.overlay.setOnMatrixChangeListener {
                //val displayMatrix = Matrix()
                //binding.overlay.getDisplayMatrix(displayMatrix)
                //binding.ivPreview.setDisplayMatrix(displayMatrix, false)
                binding.ivPreview.imageMatrix = binding.overlay.imageMatrix
            }
            binding.ivPreview.setOnMatrixChangeListener {
                //val displayMatrix = Matrix()
                //binding.ivPreview.getDisplayMatrix(displayMatrix)
                binding.overlay.imageMatrix = binding.ivPreview.imageMatrix
            }
        }

        override fun getSupportSourceLanguage(): LocaleListCompat {
            return if (OcrDispatcher.getGoogleOcrEnable()) {
                getLocaleListCompat(freeOcrLanguages, nonFreeOcrLanguages)
            } else {
                getLocaleListCompat(freeOcrLanguages)
            }
        }

        override fun getSupportTargetLanguage(): LocaleListCompat {
            return getLocaleListCompat(targetSupportLanguages)
        }

        override fun changeLanguageAfter(sourceLocale: CustomLocale, targetLocale: CustomLocale) {
            val context = context ?: return
            saveLanguage(context, PREFER_KEY_SOURCE_LANG, sourceLocale.languageTag)
            saveLanguage(
                context,
                PREFER_KEY_TARGET_LANG, targetLocale.languageTag
            )
            translateViewModel.getSourceData(mState).value = sourceLocale.languageTag
            translateViewModel.getTargetData(mState).value = targetLocale.languageTag

            binding.ivExchange.isEnabled = judgeExchangeEnable(
                Locale.forLanguageTag(translateViewModel.getSourceData(this).value),
                Locale.forLanguageTag(translateViewModel.getTargetData(this).value),
                getSupportSourceLanguage(),
                getSupportTargetLanguage()
            )

            if (currentStatus == STATE_RESULT_FROM_CAMERA_SUCCESS
                || currentStatus == STATE_RESULT_FROM_GALLERY_SUCCESS
                || currentStatus == STATE_RESULT_FROM_CAMERA_FAIL
                || currentStatus == STATE_RESULT_FROM_GALLERY_FAIL
            ) {
                binding.ivPreview.drawable?.toBitmap()?.let {
                    showProgressBar()
                    enterTranslateState()
                    if (currentStatus == STATE_RESULT_FROM_CAMERA_SUCCESS) {
                        currentStatus =
                            STATE_TRANSLATING_CAMERA
                    } else if (currentStatus == STATE_RESULT_FROM_GALLERY_SUCCESS) {
                        currentStatus =
                            STATE_TRANSLATING_GALLERY
                    }
                    mCompleteResult?.let { result ->
                        if (result.sourceLanguage != sourceLocale.languageTag) {
                            translateImage(
                                it,
                                sourceLocale.languageTag,
                                targetLocale.languageTag,
                                rotation = 0
                            )?.observe(viewLifecycleOwner, Observer {
                                if (it.isSuccess) {
                                    mCompleteResult = it.getOrNull()
                                    val completeBitmap = mCompleteResult?.renderResult
                                    recognitionSuccess(completeBitmap)
                                } else {
                                    recognitionFailure(it.exceptionOrNull())
                                }
                                enterResultState(it.isSuccess)
                            })
                        } else {
                            translateViewModel.translate(
                                it,
                                0,
                                translateViewModel.getSourceData(mState).value ?: return@let,
                                translateViewModel.getTargetData(mState).value ?: return@let
                            )
                                .observe(viewLifecycleOwner, Observer { transResult ->
                                    if (transResult.isSuccess) {
                                        mCompleteResult = transResult.getOrNull()
                                        recognitionSuccess(mCompleteResult?.renderResult)
                                    } else {
                                        recognitionFailure(transResult.exceptionOrNull())
                                    }
                                    enterResultState(transResult.isSuccess)
                                })
                        }
                    } ?: kotlin.run {
                        recognitionFailure(NoContentException(context?.getString(R.string.no_content_identified)))
                        enterResultState(false)
                    }

                }
            }
        }

        override fun enterResultState(isSuccess: Boolean) {
            //currentStatus = STATE_RESULT_FROM_CAMERA
            super.enterResultState(isSuccess)
            binding.ivPreview.setOnViewTapListener { _, _, _ ->
                if (currentStatus == STATE_RESULT_FROM_CAMERA_SUCCESS
                    || currentStatus == STATE_RESULT_FROM_GALLERY_SUCCESS
                ) {
                    binding.overlay.visibility = View.VISIBLE
                }
            }
            binding.overlay.setOnViewTapListener { _, _, _ ->
                if (currentStatus == STATE_RESULT_FROM_CAMERA_SUCCESS
                    || currentStatus == STATE_RESULT_FROM_GALLERY_SUCCESS
                ) {
                    binding.overlay.visibility = View.GONE
                }
            }

            if (isSuccess) {
                binding.torchSelector.visibility = View.VISIBLE
                binding.torchSelector.setButtonDrawable(R.drawable.ic_contrast)
                binding.torchSelector.setOnClickListener {
                    val bundle = Bundle()
                    bundle.putBinder(COMPLETE_RESULT, CompleteTransfer(mCompleteResult))
                    //bundle.putBinder()
                    //findNavController().navigate(R.id.contrastFragment)
                    val intent =
                        Intent(context ?: return@setOnClickListener, ContrastActivity::class.java)
                    intent.putExtras(bundle)
                    startActivity(intent)
                    logEvent(PT_camera_contrast)
                }
                binding.ivSelectImage.setImageResource(R.drawable.ic_save_img)
                binding.ivSelectImage.visibility = View.VISIBLE
                binding.ivSelectImage.setOnClickListener {
                    saveToGallery(binding.overlay.visibility == View.VISIBLE)
                }
            }
            binding.include.dict.isClickable = true
            binding.include.sound.isClickable = true
        }
    }

    inner class ItemRecognizeState : AbstractState() {

        override fun onStateChange(focus: Boolean) {
            if (focus) {
                translateViewModel.objectSource.observe(
                    this@CameraFragment,
                    Observer { language ->
                        binding.tvSourceLanguage.text =
                            Locale.forLanguageTag(language).displayLanguage
                        //exchangeState()
                    })
                translateViewModel.objectTarget.observe(
                    this@CameraFragment,
                    Observer { language ->
                        binding.tvTargetLanguage.text =
                            Locale.forLanguageTag(language).displayLanguage
                        //exchangeState()
                    })
                logEvent(LO_Identify_objects)
            } else {
                translateViewModel.objectSource.removeObservers(this@CameraFragment)
                translateViewModel.objectTarget.removeObservers(this@CameraFragment)
            }
        }

        override fun getType(): Int {
            return TYPE_OBJECT
        }

        override fun onPictureTaken(
            bitmap: Bitmap,
            sourceLanguage: String,
            targetLanguage: String,
            rotation: Int
        ) {
            translateViewModel.itemRecognize(
                bitmap,
                rotation,
                sourceLanguage = sourceLanguage,
                targetLanguage = targetLanguage
            ).observe(this@CameraFragment, Observer {
                if (it.isSuccess) {
                    mItemRecognizeResult = it.getOrNull()
                    if (mItemRecognizeResult != null && mItemRecognizeResult!!.source.isNotEmpty() && mItemRecognizeResult!!.target.isNotEmpty()) {
                        labelSuccess(mItemRecognizeResult!!)
                    } else {
                        recognitionFailure(it.exceptionOrNull())
                    }

                } else {
                    recognitionFailure(it.exceptionOrNull())
                }
                enterResultState(it.isSuccess)
            })
        }

        override fun enterResultState(isSuccess: Boolean) {
            super.enterResultState(isSuccess)
            binding.ivPreview.setOnViewTapListener(null)
            binding.overlay.setOnViewTapListener(null)
        }

        private fun showDict(): Boolean {
            return translateViewModel.getTargetData(mState).value == "en"
        }

        private fun labelSuccess(labelData: ImageLabelData) {
            binding.include.root.visibility = View.VISIBLE
            binding.include.word.text = labelData.target[0].data?.translated
            binding.include.dict.tag = labelData.target[0]
            binding.include.dict.visibility = if (showDict()) {
                View.VISIBLE
            } else {
                View.GONE
            }
            val pronunciation = labelData.target[0].data?.pronunciation
            if (TextUtils.isEmpty(pronunciation) || showDict().not()) {
                binding.include.prounce.visibility = View.GONE
            } else {
                binding.include.prounce.visibility = View.VISIBLE
                binding.include.prounce.text = pronunciation
            }
            binding.include.replacement.removeAllViews()
            labelData.source.forEachIndexed { index, senseNew ->
                val item = LayoutInflater.from(context ?: return@forEachIndexed)
                    .inflate(
                        R.layout.image_label_tab_item,
                        binding.include.replacement,
                        false
                    ) as CheckBox
                item.text = senseNew.data?.translated
                item.tag = labelData.target[index]
                if (index == 0) {
                    item.typeface = Typeface.DEFAULT_BOLD
                }
                item.setOnClickListener {
                    val checkBox = it as CheckBox
                    binding.include.replacement.forEachIndexed { index, view ->
                        val checkedView: CheckBox = view as CheckBox
                        if (it != checkedView) {
                            if (TextSpeech.isSpeaking()) {
                                TextSpeech.stopSpeak()
                            }
                            if (checkedView.isChecked) {
                                checkedView.isChecked = false
                            }
                            checkedView.typeface = Typeface.SANS_SERIF
                        }
                    }
                    checkBox.typeface = Typeface.DEFAULT_BOLD
                    //checkBox.isChecked = true
                    val targetSense = checkBox.tag as SenseNew
                    binding.include.dict.tag = targetSense
                    binding.include.word.text = targetSense.data?.translated
                    val pronunciation = targetSense.data?.pronunciation
                    if (TextUtils.isEmpty(pronunciation) || showDict().not()) {
                        binding.include.prounce.visibility = View.GONE
                    } else {
                        binding.include.prounce.visibility = View.VISIBLE
                        binding.include.prounce.text = pronunciation
                    }

                    if (showDict()) {
                        val sb = StringBuilder()
                        val defSize = targetSense.data?.definitions?.size ?: 0
                        targetSense.data?.definitions?.forEachIndexed { index, definition ->
                            sb.append(definition.speechPart)
                            sb.append(".")
                            sb.append(definition.source)
                            if (index < defSize - 1) {
                                sb.append("\n")
                            }
                        }

                        binding.include.explainTv.text = sb.toString()
                        binding.include.explainTv.visibility = View.VISIBLE
                    } else {
                        binding.include.explainTv.visibility = View.GONE
                    }
                    binding.include.dict.visibility = if (showDict()) {
                        View.VISIBLE
                    } else {
                        View.GONE
                    }
                    logEvent(LO_to_action, hashMapOf("type" to "word"))
                }
                binding.include.replacement.addView(item)
            }
            (binding.include.replacement.getChildAt(0) as CheckBox).isChecked = true
            if (TextSpeech.isLanguageAvailable(
                    Locale.forLanguageTag(
                        translateViewModel.getTargetData(
                            mState
                        ).value ?: return
                    )
                )
            ) {
                binding.include.sound.visibility = View.VISIBLE
            } else {
                binding.include.sound.visibility = View.INVISIBLE
            }

            binding.include.dict.setOnClickListener {
                if (TextSpeech.isSpeaking()) {
                    TextSpeech.stopSpeak()
                }
                val intent = Intent()
                intent.action = ACTION_FOR_DICTIONARY_TRANS
                val senseNew = it.tag as SenseNew
                val mapper = ObjectMapper()
                intent.putExtra(
                    EXTRA_DATA, TranslateHistory(
                        text = senseNew.data?.word ?: return@setOnClickListener,
                        sourceLanguageTag = translateViewModel.getSourceData(mState).value
                            ?: return@setOnClickListener,
                        targetLanguageTag = translateViewModel.getTargetData(mState).value
                            ?: return@setOnClickListener,
                        millis = System.currentTimeMillis(),
                        star = false,
                        scene = 0,
                        sceneJson = if (showDict()) {
                            mapper.writeValueAsString(senseNew)
                        } else {
                            ""
                        },
                        history = false,
                        translation = senseNew.data?.translated ?: return@setOnClickListener
                    )
                )
                startActivity(intent)
                logEvent(LO_to_action, hashMapOf("type" to "dict"))
            }

            binding.include.sound.setOnClickListener {
                if (TextSpeech.isSpeaking()) {
                    TextSpeech.stopSpeak()
                }
                TextSpeech.speak(
                    binding.include.word.text.toString(),
                    Locale.forLanguageTag(
                        translateViewModel.getTargetData(mState).value ?: return@setOnClickListener

                    ),
                    object : TextSpeech.SoundProgressListener {
                        override fun forceStop() {
                            val lottie = binding.include.sound
                            lottie.postDelayed(
                                {
                                    lottie.cancelAnimation()
                                    lottie.progress = 0.55f
                                }, 500
                            )
                        }

                        override fun onDone() {
                            val lottie = binding.include.sound
                            lottie.postDelayed(
                                {
                                    lottie.cancelAnimation()
                                    lottie.progress = 0.55f
                                }, 500
                            )
                        }

                        override fun onError() {
                            val lottie = binding.include.sound
                            lottie.postDelayed(
                                {
                                    lottie.cancelAnimation()
                                    lottie.progress = 0.55f
                                }, 500
                            )
                        }

                        override fun onStart() {
                            val lottie = binding.include.sound
                            lottie.post {
                                //lottie.progress = 0f
                                lottie.repeatMode = LottieDrawable.RESTART
                                lottie.repeatCount = LottieDrawable.INFINITE
                                lottie.playAnimation()
                            }
                        }

                        override fun onStop() {
                            val lottie = binding.include.sound
                            lottie.postDelayed(
                                {
                                    lottie.cancelAnimation()
                                    lottie.progress = 0.55f
                                }, 500
                            )
                        }

                    }
                )
                logEvent(LO_to_action, hashMapOf("type" to "speak"))
            }

            if (showDict()) {
                val sb = StringBuilder()
                val defSize = labelData.target[0].data?.definitions?.size ?: 0
                labelData.target[0].data?.definitions?.forEachIndexed { index, definition ->
                    sb.append(definition.speechPart)
                    sb.append(".")
                    sb.append(definition.source)
                    if (index < defSize - 1) {
                        sb.append("\n")
                    }
                }
                binding.include.explainTv.setText(sb.toString())
                binding.include.explainTv.visibility = View.VISIBLE
            } else {
                binding.include.explainTv.visibility = View.GONE
            }
            binding.include.labelContainer.post {
                binding.include.labelContainer.scrollTo(0, 0)
            }

            if (currentStatus == STATE_TRANSLATING_CAMERA) {
                currentStatus =
                    STATE_RESULT_FROM_CAMERA_SUCCESS
            }
            logEvent(
                LO_camera_click_success,
                hashMapOf(
                    source_target_language to ("${translateViewModel.getSourceData(mState).value ?: return}-${
                        translateViewModel.getTargetData(mState).value ?: return
                    }")
                )
            )
        }

        override fun enterCameraState() {
            super.enterCameraState()
            binding.torchSelector.visibility = View.INVISIBLE
            binding.ivSelectImage.visibility = View.INVISIBLE
            mCameraSource.torch(false)
            mPreTorchState = false
            readLanguageLabel()
            binding.include.root.visibility = View.GONE
            translateViewModel.objectSource.observe(viewLifecycleOwner, Observer { language ->
                binding.tvSourceLanguage.text = Locale.forLanguageTag(language).displayLanguage
            })
            translateViewModel.objectTarget.observe(viewLifecycleOwner, Observer { language ->
                binding.tvTargetLanguage.text = Locale.forLanguageTag(language).displayLanguage
                //exchangeState()
            })
        }

        override fun getSupportSourceLanguage(): LocaleListCompat {
            return getLocaleListCompat(targetSupportLanguages)
        }

        override fun getSupportTargetLanguage(): LocaleListCompat {
            return if (OcrDispatcher.getGoogleOcrEnable()) {
                getLocaleListCompat(freeOcrLanguages, nonFreeOcrLanguages)
            } else {
                getLocaleListCompat(freeOcrLanguages)
            }
        }

        override fun changeLanguageAfter(sourceLocale: CustomLocale, targetLocale: CustomLocale) {
            val context = context ?: return
            translateViewModel.getSourceData(mState).value = sourceLocale.languageTag
            translateViewModel.getTargetData(mState).value = targetLocale.languageTag
            saveLanguage(context, PREFER_KEY_SOURCE_LANG_LABEL, sourceLocale.languageTag)
            saveLanguage(
                context,
                PREFER_KEY_TARGET_LANG_LABEL, targetLocale.languageTag
            )
            if (currentStatus == STATE_RESULT_FROM_CAMERA_SUCCESS
                || currentStatus == STATE_RESULT_FROM_GALLERY_SUCCESS
                || currentStatus == STATE_RESULT_FROM_CAMERA_FAIL
                || currentStatus == STATE_RESULT_FROM_GALLERY_FAIL
            ) {
                if (currentStatus == STATE_RESULT_FROM_CAMERA_SUCCESS) {
                    currentStatus =
                        STATE_TRANSLATING_CAMERA
                } else if (currentStatus == STATE_RESULT_FROM_GALLERY_SUCCESS) {
                    currentStatus =
                        STATE_TRANSLATING_GALLERY
                }
                mItemRecognizeResult?.let {
                    showProgressBar()
                    enterTranslateState()
                    translateViewModel.imageLabelChangeLanguage(
                        it,
                        sourceLocale.languageTag,
                        targetLocale.languageTag
                    ).observe(this@CameraFragment, Observer {
                        if (it.isSuccess) {
                            mItemRecognizeResult = it.getOrNull()
                            if (mItemRecognizeResult != null && mItemRecognizeResult!!.source.isNotEmpty() && mItemRecognizeResult!!.target.isNotEmpty()) {
                                labelSuccess(mItemRecognizeResult!!)
                            } else {
                                recognitionFailure(it.exceptionOrNull())
                            }

                        } else {
                            recognitionFailure(it.exceptionOrNull())
                        }
                        enterResultState(it.isSuccess)
                    })
                }
            }
        }
    }


    override fun onViewCreated(view: View, savedInstanceState: Bundle?) {
        super.onViewCreated(view, savedInstanceState)
        //translateViewModel = ImageTranslate(requireContext())
        immersionBar {
            hideBar(BarHide.FLAG_HIDE_BAR)
            fullScreen(true)
        }
        mState = photoTranslateState
        binding = FragmentCameraBinding.bind(view)
        mViewPageGestureWrapper = ViewPageGestureWrapper(requireContext())
            .also { it.setViewPager(binding.feature) }
        mCameraSource = CameraCompatSource()
        //translateViewModel = ImageTranslate(requireContext())
        //mainExecutor = ContextCompat.getMainExecutor(requireContext())
        initPreviewView()
        initView()
        mState.enterCameraState()
        handleIntent()
        mCameraSource.setStatusCallback(this)
        requireActivity().onBackPressedDispatcher.addCallback(
            viewLifecycleOwner,
            object : OnBackPressedCallback(true) {
                override fun handleOnBackPressed() {
                    if (TextSpeech.isSpeaking()) {
                        TextSpeech.stopSpeak()
                    }
                    if (currentStatus == STATE_RESULT_FROM_GALLERY_FAIL
                        || currentStatus == STATE_RESULT_FROM_GALLERY_SUCCESS
                        || currentStatus == STATE_RESULT_FROM_CAMERA_SUCCESS
                        || currentStatus == STATE_RESULT_FROM_CAMERA_FAIL
                    ) {
                        mState.enterCameraState()
                    } else {
                        activity?.finish()
                    }
                }

            })
    }

    override fun cameraInitializeFail(exception: Exception) {
        showErrorAlert(requireContext(), exception)
    }

    override fun onPicturetaked(data: ByteArray, rotation: Int) {
        if (!isDetached && activity?.isDestroyed == false) {
            onImageSaved(data, rotation)
        }
    }

    override fun onPicturetakeFail(exception: java.lang.Exception) {
        showErrorAlert(requireActivity(), exception)
    }

    override fun onStatusChange(state: Int) {
        when (state) {
            ICameraSource.TORCH_OFF -> {
                binding.torchSelector.isChecked = false
            }
            ICameraSource.TORCH_ON -> {
                binding.torchSelector.isChecked = true
            }
        }
    }

    /**
     * Handle Intent
     */
    private fun handleIntent() {
        val intent = requireActivity().intent
        when (intent?.action) {
            Intent.ACTION_SEND -> {
                val uri = intent.getParcelableExtra<Uri?>(Intent.EXTRA_STREAM)

                if (uri != null) {
                    setImageFromGallery(uri)
                }
            }
        }
    }

    override fun onPause() {
        super.onPause()
        mCameraSource.torch(false)
    }

    override fun onDestroy() {
        super.onDestroy()
        cleanUp()
    }

    internal fun enterTranslateState() {
        binding.tvSourceLanguage.isClickable = false
        binding.tvTargetLanguage.isClickable = false
        //mControlView.setState(mControlView.cameraState)
        //mControlView.setState(mControlView.cropState)
        binding.btnControl.startAnim()
        binding.btnControl.isClickable = false
        binding.languageSelector.background?.alpha = 0x66
        binding.languageSelector.isClickable = false
        binding.tvSourceLanguage.setTextColor(
            ContextCompat.getColor(
                requireContext(),
                R.color.camera_tool_bg2
            )
        )
        binding.tvTargetLanguage.setTextColor(
            ContextCompat.getColor(
                requireContext(),
                R.color.camera_tool_bg2
            )
        )
        binding.ivExchange.alpha = 0.5f
        binding.ivExchange.isClickable = false
        binding.overlay.visibility = View.INVISIBLE
        binding.torchSelector.visibility = View.INVISIBLE
        binding.ivSelectImage.visibility = View.INVISIBLE
        binding.include.dict.isClickable = false
        binding.include.sound.isClickable = false
        binding.tabLayout.isEnabled = false
    }

    @SuppressLint("ClickableViewAccessibility")
    private fun initView() {

        mFeatureAdapter = FeatureAdapter(
            context ?: return,
            arrayListOf(
                Feature.PHOTO_TRANSLATE,
                Feature.ITEM_RECOGNIZE
            )
        )
        binding.feature.adapter = mFeatureAdapter
        binding.feature.addOnPageChangeListener(object : ViewPager.OnPageChangeListener {
            override fun onPageScrolled(
                position: Int,
                positionOffset: Float,
                positionOffsetPixels: Int
            ) {

            }

            override fun onPageSelected(position: Int) {
                when (mFeatureAdapter!!.getItem(position)) {
                    Feature.PHOTO_TRANSLATE -> {
                        mState = photoTranslateState
                    }
                    else -> {
                        mState = itemRecognizeState
                    }
                }
                mState.enterCameraState()
            }

            override fun onPageScrollStateChanged(state: Int) {

            }

        })

        binding.tabLayout.setViewPager(binding.feature)

        binding.btnControl.setOnClickListener(this)
        binding.ivSelectImage.setOnClickListener(this)
        binding.ivExchange.setOnClickListener(this)
        binding.languageSelector.setOnClickListener(this)
        binding.ivFinish.setOnClickListener(this)
        binding.tvSourceLanguage.setOnClickListener(this)
        binding.tvTargetLanguage.setOnClickListener(this)
        binding.progress.setOnTouchListener { _, _ -> true }
    }

    internal fun readLanguage() {
        val defaultSourceLan = if (Locale.getDefault().language == Locale.ENGLISH.language) {
            LANG.HI
        } else {
            Locale.ENGLISH.language
        }
        val prefer = context?.getSharedPreferences(PREFER_NAME, Context.MODE_PRIVATE) ?: return
        //val defaultTargetLang = PhotoTranslateController.targetLanguage

        var sourceLang =
            prefer.getString(PREFER_KEY_SOURCE_LANG, defaultSourceLan)
                ?: defaultSourceLan
        if (!ENG_SEQUENCE.contains(sourceLang)) {
            sourceLang = Locale.ENGLISH.language
        }

        val defaultTarget =
            if (Locale.getDefault().language == "fil" || Locale.getDefault().language == LANG.TL) {
                LANG.TL
            } else {
                Locale.getDefault().language
            }
        val targetLang =
            prefer.getString(PREFER_KEY_TARGET_LANG, defaultTarget)
                ?: defaultTarget
        translateViewModel.getSourceData(mState).value = sourceLang
        translateViewModel.getTargetData(mState).value = targetLang
    }

    internal fun readLanguageLabel() {
        val prefer = context?.getSharedPreferences(PREFER_NAME, Context.MODE_PRIVATE) ?: return
        //val defaultTargetLang = PhotoTranslateController.targetLanguage

        var sourceLang =
            prefer.getString(PREFER_KEY_SOURCE_LANG_LABEL, translateViewModel.targetLiveData.value)
                ?: translateViewModel.targetLiveData.value
        if (!ENG_SEQUENCE.contains(sourceLang)) {
            sourceLang = Locale.ENGLISH.language
        }
        val targetLang =
            prefer.getString(PREFER_KEY_TARGET_LANG_LABEL, translateViewModel.sourceLiveData.value)
                ?: translateViewModel.sourceLiveData.value
        translateViewModel.getSourceData(mState).value = sourceLang
        translateViewModel.getTargetData(mState).value = targetLang
    }

    /***************************LANGUAGE START*********************************/

    internal fun judgeExchangeEnable(
        sourceSelected: Locale,
        targetSelected: Locale,
        sourceList: LocaleListCompat,
        targetList: LocaleListCompat
    ): Boolean {
        val hasSource = sourceList.indexOf(targetSelected) != -1
        val hasTarget = targetList.indexOf(sourceSelected) != -1
        return hasSource && hasTarget
    }

    /**
     * Picker for Language
     */
    @SuppressLint("InflateParams")
    private fun languagePicker(context: Context, anchorView: View) {
        val bottomSheetDialog = BottomSheetDialog(requireActivity())

        val contentView =
            LayoutInflater.from(context).inflate(R.layout.layout_language_picker, null)

        //val bmp: Bitmap = BitmapFactory.decodeResource(resources, R.drawable.picker_bg)

        bottomSheetDialog.setContentView(contentView)
        bottomSheetDialog.window?.findViewById<View>(R.id.design_bottom_sheet)
            ?.setBackgroundResource(android.R.color.transparent)

        val sourcePicker = contentView.findViewById<WheelView<CustomLocale>>(R.id.np_source_picker)
        val targetPicker = contentView.findViewById<WheelView<CustomLocale>>(R.id.np_target_picker)
        val submit = contentView.findViewById<View>(R.id.submit_language)
        val exchange = contentView.findViewById<View>(R.id.exchange_language)

        /* ***********Source Picker*************/

        val supportSourceLanguages = mState.getSupportSourceLanguage()

        val supportSourceLanguageSize = supportSourceLanguages.size()
        val sourceValues = Array(supportSourceLanguageSize) {
            CustomLocale(
                supportSourceLanguages[it].language,
                supportSourceLanguages[it].displayLanguage
            )
        }

        val sourceLanguageTag = translateViewModel.getSourceData(mState).value
        val defaultSourceIndex = if (sourceLanguageTag != null) {
            var index = 0
            repeat(sourceValues.size) {
                if (sourceValues[it].languageTag == sourceLanguageTag) {
                    index = it
                    return@repeat
                }
            }
            index
        } else 0

        sourcePicker.setData(sourceValues.toList())
        sourcePicker.setSelectedItemPosition(defaultSourceIndex)

        /* ***********Target Picker*************/

        val supportTargetLanguages = mState.getSupportTargetLanguage()
        val supportTargetLanguageSize = supportTargetLanguages.size()
        val targetValues = Array(supportTargetLanguageSize) {
            CustomLocale(
                supportTargetLanguages[it].language,
                supportTargetLanguages[it].displayLanguage
            )
        }
        //println("supportTargetLanguageSize=$supportTargetLanguageSize")
        val targetLanguageTag = translateViewModel.getTargetData(mState).value
        val defaultTargetIndex = if (targetLanguageTag != null) {
            var index = 0
            for (i in 0..supportTargetLanguageSize) {
                val locale = supportTargetLanguages.get(i) ?: continue
                val tempTag = locale.language
                if (tempTag == targetLanguageTag) {
                    index = i
                    break
                }
            }

            index

        } else 0

        targetPicker.setData(targetValues.toList())
        targetPicker.setSelectedItemPosition(defaultTargetIndex)

        var sourceLocale = sourceValues[sourcePicker.getSelectedItemPosition()]
        var targetLocale = targetValues[targetPicker.getSelectedItemPosition()]

        val sourceSelectedListener = object : WheelView.OnItemSelectedListener<CustomLocale> {
            override fun onItemSelected(
                wheelView: WheelView<CustomLocale>,
                data: CustomLocale,
                position: Int
            ) {
                sourceLocale = wheelView.getItemData(position)
                exchange.isEnabled = judgeExchangeEnable(
                    Locale.forLanguageTag(sourceLocale.languageTag),
                    Locale.forLanguageTag(targetLocale.languageTag),
                    supportSourceLanguages,
                    supportTargetLanguages
                )
            }

        }


        val targetSelectedListener = object : WheelView.OnItemSelectedListener<CustomLocale> {
            override fun onItemSelected(
                wheelView: WheelView<CustomLocale>,
                data: CustomLocale,
                position: Int
            ) {
                targetLocale = wheelView.getItemData(position)
                exchange.isEnabled = judgeExchangeEnable(
                    Locale.forLanguageTag(sourceLocale.languageTag),
                    Locale.forLanguageTag(targetLocale.languageTag),
                    supportSourceLanguages,
                    supportTargetLanguages
                )
            }

        }

        exchange.isEnabled = judgeExchangeEnable(
            Locale.forLanguageTag(sourceLocale.languageTag),
            Locale.forLanguageTag(targetLocale.languageTag),
            supportSourceLanguages,
            supportTargetLanguages
        )
        sourcePicker.setOnItemSelectedListener(sourceSelectedListener)
        targetPicker.setOnItemSelectedListener(targetSelectedListener)

        exchange.setOnClickListener {
            val sourceIndex = sourcePicker.getData().find {
                it.languageTag == targetLocale.languageTag
            }?.let {
                sourcePicker.getData().indexOf(it)
            }
            val targetIndex = targetPicker.getData().find {
                it.languageTag == sourceLocale.languageTag
            }?.let {
                targetPicker.getData().indexOf(it)
            }


            sourcePicker.setSelectedItemPosition(sourceIndex ?: 0)
            targetPicker.setSelectedItemPosition(targetIndex ?: 0)
        }

        submit.setOnClickListener {
            //readLanguage(submit)

            if (bottomSheetDialog.isShowing) {
                bottomSheetDialog.dismiss()
            }

            if (currentStatus == STATE_RESULT_FROM_CAMERA_SUCCESS
                || currentStatus == STATE_RESULT_FROM_GALLERY_SUCCESS
                || currentStatus == STATE_RESULT_FROM_CAMERA_FAIL
                || currentStatus == STATE_RESULT_FROM_GALLERY_FAIL
            ) {
                if (!context.isNetworkConnected()) {
                    Snackbar.make(anchorView, R.string.network_unavailable, Snackbar.LENGTH_SHORT)
                        .show()
                    return@setOnClickListener
                }

            }

            mState.changeLanguageAfter(sourceLocale, targetLocale)

            when (mState.getType()) {
                TYPE_TEXT -> {
                    logEvent(
                        PT_lan_click_success,
                        source_target_language,
                        "${translateViewModel.getSourceData(mState).value ?: return@setOnClickListener}&${
                            translateViewModel.getTargetData(
                                mState
                            ).value ?: return@setOnClickListener
                        }"
                    )
                }
                TYPE_OBJECT -> {
                    logEvent(
                        LO_lan_click_success,
                        source_target_language,
                        "${translateViewModel.getSourceData(mState).value ?: return@setOnClickListener}&${
                            translateViewModel.getTargetData(
                                mState
                            ).value ?: return@setOnClickListener
                        }"
                    )
                }
            }

        }

        bottomSheetDialog.setOnShowListener {
            binding.languageSelector.visibility = View.INVISIBLE
        }
        bottomSheetDialog.setOnDismissListener {
            activity ?: return@setOnDismissListener
            binding.languageSelector.visibility = View.VISIBLE
            ImmersionBar.with(this@CameraFragment)
                .fullScreen(true)
                .init()
        }

        targetPicker.setVibratorEnable(true)
        sourcePicker.setVibratorEnable(true)

        bottomSheetDialog.show()
    }

    /**
     * Exchange Language
     */
    private fun exchangeLanguage(context: Context) {
        val sourceLanguageTag = translateViewModel.getTargetData(mState).value ?: return
        val targetLanguageTag = translateViewModel.getSourceData(mState).value ?: return
        /*saveLanguage(context, PREFER_KEY_SOURCE_LANG, sourceLanguageTag)
        saveLanguage(
            context,
            PREFER_KEY_TARGET_LANG, targetLanguageTag
        )*/
        translateViewModel.getSourceData(mState).value = sourceLanguageTag
        translateViewModel.getTargetData(mState).value = targetLanguageTag
        val sourceLocale = Locale.forLanguageTag(sourceLanguageTag)
        val targetLocale = Locale.forLanguageTag(targetLanguageTag)
        mState.changeLanguageAfter(
            CustomLocale(sourceLanguageTag, sourceLocale.displayLanguage),
            CustomLocale(targetLanguageTag, targetLocale.displayLanguage)
        )
    }

    /***************************LANGUAGE END*********************************/

    /***************************CAMERA START*********************************/

    private fun initPreviewView() {
        mCameraSource.initialize(requireContext(), binding.previewContainer, this)
        mCameraSource.setViewTouchListener(this)
        //mPreviewView = view.findViewById(R.id.preview)
    }

    /**
     * Take Photo
     */
    private fun takePicture() {
        mCameraSource.takePicture()
    }

    internal var mPreviewBitmap: Bitmap? = null

    private fun doOcrTranslate(img: Any, rotation: Int = 0) {

        Glide.with(this)
            .asBitmap()
            .load(img)
            .listener(object : RequestListener<Bitmap> {
                override fun onResourceReady(
                    resource: Bitmap,
                    model: Any?,
                    target: Target<Bitmap>?,
                    dataSource: DataSource?,
                    isFirstResource: Boolean
                ): Boolean {
                    //mCameraSource.pause()
                    mPreviewBitmap = resource
                    binding.ivPreview.visibility = View.VISIBLE
                    val sourceLanguageTag = translateViewModel.getSourceData(mState).value ?: "en"
                    val targetLanguageTag = translateViewModel.getTargetData(mState).value ?: "en"
                    //val sourceLanguage = localeToGoogleLanguage[sourceLanguageTag] ?: "en"
                    //val targetLanguage = localeToGoogleLanguage[targetLanguageTag] ?: "en"
                    showProgressBar()
                    //enterTranslateState()

                    mState.onPictureTaken(resource, sourceLanguageTag, targetLanguageTag, rotation)
                    return false
                }

                override fun onLoadFailed(
                    e: GlideException?,
                    model: Any?,
                    target: Target<Bitmap>?,
                    isFirstResource: Boolean
                ): Boolean {
                    binding.failToast.visibility = View.VISIBLE
                    binding.failToastText.text = getString(R.string.load_image_fail)
                    binding.failToast.postDelayed(
                        { binding.failToast.visibility = View.GONE },
                        2000
                    )
                    mState.enterCameraState()
                    return false
                }

            })
            .into(binding.ivPreview)
    }

    internal var mPreTorchState: Boolean = false

    /**
     * Take Photo Success
     */
    private fun onImageSaved(data: ByteArray, rotation: Int) {
        binding.ivPreview.scaleType = ImageView.ScaleType.CENTER_CROP
        binding.overlay.scaleType = ImageView.ScaleType.CENTER_CROP
        doOcrTranslate(data, rotation)
        mPreTorchState = binding.torchSelector.isChecked
        mCameraSource.torch(switch = false)
    }

    private fun showErrorAlert(context: Context, exc: java.lang.Exception) {
        val onClickListener = DialogInterface.OnClickListener { _, which ->
            activity?.finish()
        }

        AlertDialog.Builder(context)
            .setTitle(android.R.string.dialog_alert_title)
            .setMessage(R.string.camera_unavailable)
            .setPositiveButton(android.R.string.ok, onClickListener)
            .setCancelable(false)
            .create()
            .show()
    }

    /***************************CAMERA END*********************************/

    /**
     * 翻译图片
     */
    internal fun translateImage(
        bitmap: Bitmap,
        sourceLanguage: String,
        targetLanguage: String,
        rotation: Int
    ): LiveData<Result<CompleteResult>>? {
        return translateViewModel.translate(
            bitmap,
            rotation,
            sourceLanguage,
            targetLanguage
        )
    }

    internal fun recognitionFailure(throwable: Throwable? = null) {
        if (currentStatus == STATE_TRANSLATING_CAMERA) {
            currentStatus =
                STATE_RESULT_FROM_CAMERA_FAIL
        } else if (currentStatus == STATE_TRANSLATING_GALLERY) {
            currentStatus =
                STATE_RESULT_FROM_GALLERY_FAIL
        }
        showTranslateError(throwable)
        when (mState.getType()) {
            TYPE_TEXT -> {
                translationFailEvent(
                    PT__camera_click_fail,
                    currentStatus,
                    throwable
                )

            }
            TYPE_OBJECT -> {
                translationFailEvent(LO__camera_click_fail, currentStatus, throwable)
            }
        }

        logEvent(
            Trans_translate_failure,
            hashMapOf(
                language to ("${translateViewModel.getSourceData(mState).value ?: return}-${
                    translateViewModel.getTargetData(
                        mState
                    ).value ?: return
                }"),
                moduleType to modulePic,
                errorMessage to (throwable?.message ?: "unknown error")
            )
        )

    }

    /**
     * 识别翻译成功
     */
    internal fun recognitionSuccess(newBitmap: Bitmap?) {
        //if (isDetached) return
        when (currentStatus) {
            STATE_TRANSLATING_GALLERY -> {
                currentStatus =
                    STATE_RESULT_FROM_GALLERY_SUCCESS
                logEvent(PT__gallery_click_success)
                logEvent(
                    Trans_translate_success,
                    hashMapOf(
                        language to ("${translateViewModel.getSourceData(mState).value ?: return}-${
                            translateViewModel.getTargetData(
                                mState
                            ).value ?: return
                        }"),
                        moduleType to modulePic
                    )
                )
            }
            STATE_TRANSLATING_CAMERA -> {
                currentStatus =
                    STATE_RESULT_FROM_CAMERA_SUCCESS
                logEvent(PT__camera_click_success)
                logEvent(
                    Trans_translate_success,
                    hashMapOf(
                        language to ("${translateViewModel.getSourceData(mState).value ?: return}-${
                            translateViewModel.getTargetData(
                                mState
                            ).value ?: return
                        }"),
                        moduleType to modulePic
                    )
                )
            }
            STATE_RESULT_FROM_GALLERY_FAIL -> {
                currentStatus =
                    STATE_RESULT_FROM_GALLERY_SUCCESS
            }
            STATE_RESULT_FROM_CAMERA_FAIL -> {
                currentStatus =
                    STATE_RESULT_FROM_CAMERA_SUCCESS
            }
        }

        val matrix = binding.overlay.imageMatrix
        Glide.with(this)
            .load(newBitmap)
            .listener(object : RequestListener<Drawable> {
                override fun onResourceReady(
                    resource: Drawable?,
                    model: Any?,
                    target: Target<Drawable>?,
                    dataSource: DataSource?,
                    isFirstResource: Boolean
                ): Boolean {
                    binding.overlay.visibility = View.VISIBLE
                    binding.overlay.setImageDrawable(resource)
                    binding.overlay.setDisplayMatrix(matrix)
                    return true
                }

                override fun onLoadFailed(
                    e: GlideException?,
                    model: Any?,
                    target: Target<Drawable>?,
                    isFirstResource: Boolean
                ): Boolean {
                    return false
                }

            })
            .into(binding.overlay)
        //mPreviewResult?.setImageBitmap(newBitmap)

    }


    private val combineViewModel by viewModels<CombineViewModel> {
        object : ViewModelProvider.Factory {
            override fun <T : ViewModel?> create(modelClass: Class<T>): T {
                return CombineViewModel() as T
            }

        }
    }

    private fun saveToGallery(combine: Boolean) {
        logEvent(PT_save_to_gallery)
        val overlay = mCompleteResult?.renderResult
        val preview = mPreviewBitmap
        if (combine) {
            if (overlay?.isRecycled?.not() == true && preview?.isRecycled?.not() == true) {
                combineViewModel.combine(preview, overlay).switchMap {
                    flow {
                        emit(
                            Utils.saveBitmap(
                                requireContext(),
                                it,
                                "HD-${System.currentTimeMillis()}.jpeg"
                            )
                        )
                        //it.recycle()
                    }.asLiveData(Dispatchers.IO)
                }.observe(this, Observer {
                    //binding.ivSelectImage.visibility = View.INVISIBLE
                    Toast.makeText(
                        context ?: return@Observer,
                        getString(R.string.save_toast_string),
                        Toast.LENGTH_SHORT
                    ).show()
                })
            }
        } else {
            if (preview?.isRecycled?.not() == true) {
                flow {
                    emit(
                        Utils.saveBitmap(
                            requireContext(),
                            preview,
                            "HD-${System.currentTimeMillis()}.jpeg"
                        )
                    )
                }.flowOn(Dispatchers.IO)
                    .asLiveData().observe(this, Observer {
                        Toast.makeText(
                            context ?: return@Observer,
                            getString(R.string.save_toast_string),
                            Toast.LENGTH_SHORT
                        ).show()
                    })
            }
        }
    }


    /**
     * Show Progress
     */
    internal fun showProgressBar() {
        binding.progress.visibility = View.VISIBLE

    }

    /**
     * Hide Progress
     */
    internal fun hideProgressBar() {
        binding.progress.visibility = View.GONE
    }

    private fun showTranslateError(throwable: Throwable? = null) {
        binding.failToast.visibility = View.VISIBLE
        binding.failToastText.text = throwable?.message ?: ""
        binding.failToast.postDelayed({ binding.failToast.visibility = View.GONE }, 2000)
    }

    /***************************CROPPER WITH TRANSLATE END*********************************/

    /***************************SELECT START*********************************/
    /**
     * 设置图片
     */
    private fun setImageFromUri(uri: Uri) {
        currentStatus =
            STATE_TRANSLATING_GALLERY
        binding.ivPreview.scaleType = ImageView.ScaleType.CENTER_INSIDE
        binding.overlay.scaleType = ImageView.ScaleType.CENTER_INSIDE
        enterTranslateState()
        doOcrTranslate(uri)
    }

    fun setImageFromShot(uri: Uri) {
        setImageFromUri(uri)
        logEvent(PT_screen_shot_iden_click)
    }

    fun setImageFromGallery(uri: Uri) {
        setImageFromUri(uri)
        logEvent(PT_gallery_iden_click)
    }

    internal var screenShotMode = false

    fun setScreenShotMode(mode: Boolean) {
        screenShotMode = mode
    }

    /***************************SELECT END*********************************/

    private val gallerySelector = registerForActivityResult(
        object : ActivityResultContract<String, Uri>() {
            override fun createIntent(context: Context, input: String): Intent {
                return Intent(Intent.ACTION_PICK)
                    .setType(input)
            }

            override fun parseResult(resultCode: Int, intent: Intent?): Uri? {
                return if (intent == null || resultCode != Activity.RESULT_OK) null else intent.data
            }

        }
    ) { uri ->
        setImageFromGallery(uri ?: return@registerForActivityResult)
    }

    override fun onClick(v: View) {
        if (TextSpeech.isSpeaking()) {
            TextSpeech.stopSpeak()
        }
        when (v.id) {
            R.id.btn_control -> {
                if (AntiShakeUtils.isInvalidClick(v)) {
                    return
                }

                when (currentStatus) {
                    STATE_CAMERA -> {
                        if (!v.context.isNetworkConnected()) {
                            Snackbar.make(v, R.string.network_unavailable, Snackbar.LENGTH_SHORT)
                                .show()
                            return
                        }
                        currentStatus =
                            STATE_TRANSLATING_CAMERA
                        enterTranslateState()
                        takePicture()
                        when (mState.getType()) {
                            TYPE_TEXT -> {
                                logEvent(PT_camera_click)
                            }
                            TYPE_OBJECT -> {
                                logEvent(LO_camera_click)
                            }
                        }
                    }
                    STATE_RESULT_FROM_CAMERA_FAIL,
                    STATE_RESULT_FROM_CAMERA_SUCCESS,
                    STATE_RESULT_FROM_GALLERY_SUCCESS,
                    STATE_RESULT_FROM_GALLERY_FAIL -> {
                        mState.enterCameraState()
                        when (mState.getType()) {
                            TYPE_TEXT -> {
                                logEvent(PT_exit)
                                logEvent(PT_camera_refresh)
                            }
                            TYPE_OBJECT -> {
                                logEvent(LO_camera_refresh)
                            }
                        }
                    }
                }
            }
            R.id.iv_select_image -> {
                if (AntiShakeUtils.isInvalidClick(v)) {
                    return
                }
                if (!v.context.isNetworkConnected()) {
                    Snackbar.make(v, R.string.network_unavailable, Snackbar.LENGTH_SHORT)
                        .show()
                    return
                }

                try {
                    gallerySelector.launch("image/*")
                    logEvent(PT_gallery_click)
                } catch (e: ActivityNotFoundException) {
                    binding.failToast.visibility = View.VISIBLE
                    binding.failToastText.text = context?.getString(R.string.gallery_not_found)
                    binding.failToast.postDelayed(
                        { binding.failToast.visibility = View.GONE },
                        2000
                    )
                }
                mCameraSource.torch(false)
                mPreTorchState = false
            }
            R.id.torch_selector -> {
                val isChecked = (v as? Checkable)?.isChecked
                mCameraSource.torch(isChecked == true)
                mPreTorchState = isChecked == true
                //mCamera?.cameraControl?.enableTorch(isChecked == true)
                logEvent(PT_flash_click)
            }
            R.id.iv_finish -> {
                //requireActivity().finish()
                if (AntiShakeUtils.isInvalidClick(v)) {
                    return
                }
                if (currentStatus == STATE_RESULT_FROM_GALLERY_FAIL
                    || currentStatus == STATE_RESULT_FROM_GALLERY_SUCCESS
                    || currentStatus == STATE_RESULT_FROM_CAMERA_SUCCESS
                    || currentStatus == STATE_RESULT_FROM_CAMERA_FAIL
                ) {
                    mState.enterCameraState()
                } else {
                    activity?.finish()
                }
            }
            R.id.tv_source_language, R.id.tv_target_language, R.id.language_selector -> {
                /*startActivityForSelectLanguage(
                    v.context,
                    REQUEST_CODE_SELECT_SOURCE_LANGUAGE
                )*/
                if (AntiShakeUtils.isInvalidClick(v)) {
                    return
                }
                languagePicker(v.context, v)
                //pickerClickedEvent(state = currentStatus)
            }
            R.id.iv_exchange -> {
                if (AntiShakeUtils.isInvalidClick(v)) {
                    return
                }
                exchangeLanguage(v.context)
                logEvent(PT_lan_click_switch)
            }

        }
    }

    override fun onTouchEvent(event: MotionEvent) {
        mViewPageGestureWrapper?.onTouchEvent(event)
    }

    override fun onInterceptTouchEvent(event: MotionEvent) {
        mViewPageGestureWrapper?.onInterceptTouchEvent(event)
    }
}