package com.talpa.translate.datasource

import android.content.Context
import android.graphics.ImageFormat.YUV_420_888
import androidx.lifecycle.LifecycleOwner
import com.talpa.translate.camera.view.CameraListener
import com.talpa.translate.camera.view.CameraView
import com.talpa.translate.camera.view.PictureResult
import com.talpa.translate.camera.view.controls.*
import com.talpa.translate.camera.view.gesture.Gesture
import com.talpa.translate.camera.view.gesture.GestureAction
import com.talpa.translate.camera.view.markers.DefaultAutoFocusMarker
import com.talpa.translate.camera.view.preview.ViewTouchListener
import com.talpa.translate.ocr.exception.NoContentException

/**
 * Create by chenjunsheng on 2020/5/9
 */
open class CameraCompatSource :
    ICameraSource {

    protected lateinit var mCamera: CameraView
    private var mStatus: ICameraSource.Status? = null

    override fun initialize(context: Context, camera: CameraView, lifecycleOwner: LifecycleOwner) {
        mCamera = camera
            .apply {
                engine = Engine.CAMERA2
                setExperimental(true)
                preview = Preview.GL_SURFACE
                playSounds = false
                grid = Grid.OFF
                flash = Flash.OFF
                audio = Audio.OFF
                facing = Facing.BACK
                mapGesture(Gesture.TAP, GestureAction.AUTO_FOCUS)
                mapGesture(Gesture.LONG_TAP, GestureAction.NONE)
                mapGesture(Gesture.PINCH, GestureAction.ZOOM)
                mode = Mode.PICTURE
                setAutoFocusMarker(DefaultAutoFocusMarker())
                useDeviceOrientation = false
                frameProcessingFormat = YUV_420_888
            }
        //parent.addView(mCamera, FrameLayout.LayoutParams(ViewGroup.LayoutParams.MATCH_PARENT, ViewGroup.LayoutParams.MATCH_PARENT))
        mCamera.setLifecycleOwner(lifecycleOwner)
        mCamera.addCameraListener(L())
    }

    fun setViewTouchListener(listener: ViewTouchListener) {
        mCamera.setViewTouchListener(listener)
    }

    override fun setStatusCallback(status: ICameraSource.Status) {
        mStatus = status
    }

    override fun open() {
        if (!mCamera.isOpened) {
            mCamera.open()
        }
    }

    override fun pause() {
        if (mCamera.isOpened) {
            mCamera.close()
        }
    }

    override fun release() {
        mCamera.destroy()
    }

    override fun takePicture() {
        mCamera.takePictureSnapshot()
    }

    override fun torch(switch: Boolean) {
        mCamera.set(
            if (switch) {
                Flash.TORCH
            } else {
                Flash.OFF
            }
        )
        mStatus?.onStatusChange(
            if (switch) {
                ICameraSource.TORCH_ON
            } else {
                ICameraSource.TORCH_OFF
            }
        )
    }

    private var mOrientation = 0

    inner class L : CameraListener() {
        override fun onPictureTaken(result: PictureResult) {
            if (result.data != null) {
                mStatus?.onPicturetaked(result.data, mOrientation)
            } else {
                mStatus?.onPicturetakeFail(NoContentException())
            }
        }

        override fun onOrientationChanged(orientation: Int) {
            super.onOrientationChanged(orientation)
            mOrientation = orientation
        }
    }

}