package com.talpa.translate.network

import android.annotation.SuppressLint
import android.content.Context
import android.util.Log
import com.photo.translation.R
import com.talpa.PhotoTranslateController
import com.talpa.tengine.Trans
import com.talpa.tengine.lang.LANG
import com.talpa.tengine.lang.languageTagToLangMap
import com.talpa.translate.base.utils.crypt
import com.talpa.translate.network.service.ApiService
import com.talpa.translate.ocr.exception.NoContentException
import com.talpa.translator.link.translate
import kotlinx.coroutines.flow.asFlow
import kotlinx.coroutines.flow.flatMapMerge
import kotlinx.coroutines.flow.flowOf
import kotlinx.coroutines.flow.toList
import okhttp3.OkHttpClient
import retrofit2.Retrofit
import retrofit2.converter.gson.GsonConverterFactory
import java.util.concurrent.TimeUnit

/**
 * Hi Translator Sync
 *
 * @author CY 20-3-5
 */
class HiTranslator private constructor(val context: Context) {
    companion object {

        //private const val DEBUG_BASE_URL = "http://tapi.translasion.com/"
        private const val RELEASE_BASE_URL = "https://api.translasion.com/"
        private const val CONNECT_TIME_OUT = 6000
        private const val READ_TIME_OUT = 6000
        private const val REQUEST_METHOD = "POST"
        private const val CONTENT_TYPE = "application/json"
        private const val APP_KEY = "app_key"
        private const val PREFER_TRANSLATOR = "prefer_translator_data"
        const val SERVER_SECRET = "1e98215f-929b-7cd9-9c20-4c523d4b493b"
        const val SERVER_KEY = "e292400433a4"

        @SuppressLint("StaticFieldLeak")
        @Volatile
        private var mInstance: HiTranslator? = null

        fun getInstance(context: Context): HiTranslator {
            if (mInstance == null) {
                synchronized(HiTranslator::class.java) {
                    if (mInstance == null) {
                        mInstance = HiTranslator(context.applicationContext)
                    }
                }
            }
            return mInstance!!
        }
    }

    private val apiService: ApiService

    init {
        val okHttpClient = OkHttpClient.Builder()
            .connectTimeout(CONNECT_TIME_OUT.toLong(), TimeUnit.MILLISECONDS)
            .readTimeout(READ_TIME_OUT.toLong(), TimeUnit.MILLISECONDS)
            .addInterceptor {
                val newRequest = it.request().newBuilder()
                    .addHeader(
                        "content-type",
                        CONTENT_TYPE
                    )
                    .build()
                val response = it.proceed(newRequest)
                response
            }
            .build()

        apiService = Retrofit.Builder()
            .baseUrl(RELEASE_BASE_URL)
            .client(okHttpClient)
            .addConverterFactory(GsonConverterFactory.create())
            //.addCallAdapterFactory(CoroutineCallAdapterFactory.invoke())
            .build().create(ApiService::class.java)
    }

    suspend fun postTranslateSingle(from: String, to: String, texts: List<String>): TransResponse {
        val langMap = languageTagToLangMap()
        val requests = texts.map {
            Trans(langMap[from] ?: LANG.EN, langMap[to] ?: LANG.EN, it)
        }
        val responses = requests.asFlow().flatMapMerge {
            flowOf(context.translate(it).blockingSingle())
        }.toList()
        val translations = arrayListOf<String>()
        repeat(responses.size) { index ->
            translations.add(
                responses.find { trans ->
                    trans.hashCode() == requests[index].hashCode()
                }?.result?.translation
                    ?: /*requests[index].text*/context.getString(R.string.translate_fail)
            ) //翻译失败的话显示错误信息
        }
        val result = Result(translations)
        if (responses.isEmpty()) {
            return TransResponse(0, null, Result(arrayListOf()))
        } else {
            return TransResponse(
                responses.first().result?.code ?: 0,
                responses.first().result?.errorMessage, result
            )
        }
    }

    suspend fun postTranslate(from: String, to: String, texts: List<String>): TransResponse? {
        if (from == to) {
            return TransResponse(1000, "success", Result(texts))
        }
        val serverAppKey =
            PhotoTranslateController.serverAppKey ?: throw NoContentException("app key not set")
        val serverSecret =
            PhotoTranslateController.serverAppSecret
                ?: throw NoContentException("app secret not set")
        val timeStamp = System.currentTimeMillis()
        val random = ((Math.random() * 9 + 1) * 1000).toInt()
        val randomKey = "${serverAppKey}&$to&$timeStamp&${serverSecret}&$random"

        val requestbody =
            TransRequest(
                app_key = serverAppKey,
                from = from,
                to = to,
                texts = texts,
                sig = crypt(randomKey),
                nonce = random.toString(),
                timestamp = timeStamp
            )

        return try {
            apiService.translate(requestbody)
        } catch (e: Exception) {
            e.printStackTrace()
            null
        }

        /*val translations = arrayListOf<String>()
        repeat(responses.size) { index ->
            translations.add(responses.find { trans ->
                trans.hashCode() == requests[index].hashCode()
            }?.result?.translation ?: requests[index].text) //翻译失败的话显示原文
        }
        val result = Result(translations)
        if (responses.isEmpty()) {
            return TransResponse(0, null, Result(arrayListOf()))
        } else {
            return TransResponse(responses.first().result?.code ?: 0,
                responses.first().result?.errorMessage, result)
        }*/
    }
}