package com.talpa.translate.ocr

import android.graphics.*
import androidx.annotation.Keep
import kotlinx.coroutines.*
import kotlin.math.abs
import kotlin.math.max

/**
 * Create by chenjunsheng on 2020/7/28
 */
@Keep
class PhotoAnalyzer private constructor(private val mNativePtr: Long) {

    companion object {
        init {
            System.loadLibrary("imgprocessor")
            System.loadLibrary("opencv_java4")
        }

        const val X_OFFSET = 16
        const val Y_OFFSET = 12

        fun getBase64String(bitmap: Bitmap): String? {
            return getBase64Native(bitmap)
        }

        fun compressBitmap(bitmap: Bitmap): Bitmap {
            return nativeCompressBitmap(bitmap)
        }

        fun getBoundingRect(width: Int, height: Int, points: Array<Point>): Rect {
            return nativeGetBoundingRect(width, height, points)
        }

        fun findRevertTextColor(bitmap: Bitmap, blockCorners: Array<Point>): Boolean {
            return nativeFindRevertTextColor(bitmap, blockCorners)
        }

        /**
         * 仅用于屏幕截图文本区域
         */
        fun findTextAreaFast(bitmap: Bitmap): Array<Rect> {
            val rects = nativeFindTextAreaFast(bitmap).also {
                it.forEach {
                    it.inset(-X_OFFSET, -Y_OFFSET)
                }
            }.onEach {
                if (it.top < 0) it.top = 0
                if (it.left < 0) it.left = 0
                if (it.right > bitmap.width) it.right = bitmap.width
                if (it.bottom > bitmap.height) it.bottom = bitmap.height
            }.filter {
                it.width() > 0 && it.height() > 0

            }

            fun isRectangleOverlap(current: Rect, other: Rect): Boolean {
                return current.left < other.right
                        && other.left < current.right
                        && current.top < other.bottom
                        && other.top < current.bottom
            }

            val rectList = rects.toMutableList()

            do {
                var removeIndex = -1
                out@ for (i in 0 until rectList.size) {
                    for (k in 0 until rectList.size - 1 - i) {
                        val first = rectList[i]
                        val second = rectList[i + 1 + k]
                        if (isRectangleOverlap(current = first, other = second)) {
                            first.union(second)
                            removeIndex = i + 1 + k
                            break@out
                        }
                    }
                }

                if (removeIndex != -1) {
                    rectList.removeAt(removeIndex)
                }

            } while (removeIndex != -1)



            return rectList.toTypedArray()
        }

        @JvmStatic
        private external fun nativeFindTextAreaFast(bitmap: Bitmap): Array<Rect>

        @JvmStatic
        private external fun nativeFindRevertTextColor(
            bitmap: Bitmap,
            blockCorners: Array<Point>
        ): Boolean

        @JvmStatic
        private external fun nativeGetBoundingRect(
            width: Int,
            height: Int,
            points: Array<Point>
        ): Rect

        @JvmStatic
        private external fun nativeCompressBitmap(bitmap: Bitmap): Bitmap

        @JvmStatic
        private external fun getBase64Native(bitmap: Bitmap): String?

        @JvmStatic
        private external fun nativeCreate(bitmap: Bitmap): PhotoAnalyzer

        @JvmStatic
        private external fun nativeRecycle(ptr: Long)

        @JvmStatic
        private external fun nativeGenBitmap(
            ptr: Long, roi: Rect, blockCorner: Array<Point>,
            linePoints: Array<Array<Point>>
        ): Bitmap?


        @JvmStatic
        private external fun testnativeGenBitmap(
            ptr: Long,
            area: Rect,
            points: Array<Array<Point>>
        ): Bitmap?

        fun createAnalyzer(source: Bitmap): PhotoAnalyzer {
            val analyzer = nativeCreate(bitmap = source)
            analyzer.mWidth = source.width
            analyzer.mHeight = source.height
            return analyzer
        }
    }

    private var mHeight: Int = 0
    private var mWidth: Int = 0

    private var mRecycled = false

    fun recycle() {
        if (!mRecycled) {
            nativeRecycle(mNativePtr)
            mRecycled = true
        }
    }

    data class AnalyzerStub(
        var bitmap: Bitmap?, val rect: Rect,
        val blockCorners: List<Point>, val linePoints: List<Array<Point>>
    )

    fun generateBgBitmap(stubs: Array<AnalyzerStub>): Bitmap {
        if (mRecycled) {
            throw IllegalStateException("analyzer has been recycled")
        }

        /*if (false) {
            val paint = Paint()
            paint.setColor(Color.RED)
            val bitmap = Bitmap.createBitmap(mWidth, mHeight, Bitmap.Config.ARGB_8888, true)
            val canvas = Canvas(bitmap)
            stubs.forEach {
                it.linePoints.forEach {
                    canvas.drawLine(
                        it[0].x.toFloat(), it[0].y.toFloat(),
                        it[1].x.toFloat(), it[1].y.toFloat(), paint
                    )
                    canvas.drawLine(
                        it[1].x.toFloat(), it[1].y.toFloat(),
                        it[2].x.toFloat(), it[2].y.toFloat(), paint
                    )
                    canvas.drawLine(
                        it[2].x.toFloat(), it[2].y.toFloat(),
                        it[3].x.toFloat(), it[3].y.toFloat(), paint
                    )
                    canvas.drawLine(
                        it[3].x.toFloat(), it[3].y.toFloat(),
                        it[0].x.toFloat(), it[0].y.toFloat(), paint
                    )
                }
            }

            stubs.forEach {
                val bitmap = nativeGenBitmap(
                    mNativePtr, it.rect,
                    it.blockCorners.toTypedArray(), it.linePoints.toTypedArray()
                )
                Log.d("cjslog", "return bitmap")
            }
            return bitmap
        }*/

        //val bitmapDeffer = arrayListOf<Deferred<AnalyzerStub?>>()

        /*val list = arrayListOf<Array<Point>>()

        stubs.forEach {
            it.linePoints.forEach {
                list.add(it)
            }
        }*/

        return runBlocking {
            val bitmapDeffer = stubs.map { stub ->
                async(Dispatchers.Default) {
                    val lines = arrayListOf<Array<Point>>()
                    stub.linePoints.mapTo(lines) { arrayOfPoints ->
                        val points = arrayOfPoints.copyOf()
                        points.onEach {
                            it.x = max(abs(it.x - stub.rect.left), 0)
                            it.y = max(abs(it.y - stub.rect.top), 0)
                        }
                        points
                    }

                    val blockCorners = arrayListOf<Point>()
                    stub.blockCorners.mapTo(blockCorners) {
                        Point(
                            max(abs(it.x - stub.rect.left), 0),
                            max(abs(it.y - stub.rect.top), 0)
                        )
                    }

                    stub.bitmap = nativeGenBitmap(
                        mNativePtr, stub.rect,
                        blockCorners.toTypedArray(), lines.toTypedArray()
                    )
                    //AnalyzerStub(nativeGenBitmap(mNativePtr, stub.rect, stub.points), rect)
                    stub
                }

            }.awaitAll()
            val bitmap: Bitmap = Bitmap.createBitmap(mWidth, mHeight, Bitmap.Config.ARGB_8888)
            val canvas = Canvas(bitmap)
            bitmapDeffer.forEach { analyzerStub ->
                val resultBlock = analyzerStub.bitmap ?: return@forEach
                val area = analyzerStub.rect
                canvas.drawBitmap(resultBlock, null, area, null)
                analyzerStub.bitmap = null
            }
            bitmap
        }
    }

/*    fun testgenerateBgBitmap(stubs: Array<AnalyzerStub>): Bitmap? {
        if (mRecycled) {
            throw IllegalStateException("analyzer has been recycled")
        }

        if (false) {
            val paint = Paint()
            paint.setColor(Color.RED)
            val bitmap = Bitmap.createBitmap(mWidth, mHeight, Bitmap.Config.ARGB_8888, true)
            val canvas = Canvas(bitmap)
            stubs.forEach {
                it.linePoints.forEach {
                    canvas.drawLine(
                        it[0].x.toFloat(), it[0].y.toFloat(),
                        it[1].x.toFloat(), it[1].y.toFloat(), paint
                    )
                    canvas.drawLine(
                        it[1].x.toFloat(), it[1].y.toFloat(),
                        it[2].x.toFloat(), it[2].y.toFloat(), paint
                    )
                    canvas.drawLine(
                        it[2].x.toFloat(), it[2].y.toFloat(),
                        it[3].x.toFloat(), it[3].y.toFloat(), paint
                    )
                    canvas.drawLine(
                        it[3].x.toFloat(), it[3].y.toFloat(),
                        it[0].x.toFloat(), it[0].y.toFloat(), paint
                    )
                }
            }

            stubs.forEach {
                val bitmap = nativeGenBitmap(
                    mNativePtr, it.rect,
                    it.blockCorners.toTypedArray(), it.linePoints.toTypedArray()
                )
                Log.d("cjslog", "return bitmap")
            }
            return bitmap
        }

        val bitmapDeffer = arrayListOf<Deferred<AnalyzerStub?>>()

        val list = arrayListOf<Array<Point>>()

        stubs.forEach {
            it.linePoints.forEach {
                list.add(it)
            }
        }



        return testnativeGenBitmap(
            mNativePtr, Rect(0, 0, mWidth, mHeight),
            list.toTypedArray()
        )
    }*/

}