package com.talpa.translate.ocr.result

import android.content.Context
import android.os.Bundle
import android.util.Log
import android.widget.Toast
import androidx.activity.compose.setContent
import androidx.activity.viewModels
import androidx.appcompat.app.AppCompatActivity
import androidx.compose.runtime.*
import androidx.compose.ui.ExperimentalComposeUiApi
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.unit.ExperimentalUnitApi
import androidx.core.content.ContextCompat
import androidx.core.view.WindowCompat
import androidx.lifecycle.MutableLiveData
import androidx.lifecycle.ViewModelProvider
import com.google.accompanist.systemuicontroller.rememberSystemUiController
import com.photo.translation.R
import com.talpa.translate.base.utils.*
import com.talpa.translate.camera.COMPLETE_RESULT
import com.talpa.translate.ocr.datasource.CompleteResult
import java.util.*

/**
 * Create by chenjunsheng on 2020/11/26
 */
class ContrastEditActivity : AppCompatActivity() {

    private val sourceLiveData: MutableLiveData<String> = MutableLiveData<String>("en")
    private val targetLiveData: MutableLiveData<String> = MutableLiveData<String>("en")
    private val translateViewModel: TranslateViewModel by viewModels {
        ViewModelProvider.AndroidViewModelFactory.getInstance(application)
    }
    private var completeResult: CompleteResult? = null

    @ExperimentalComposeUiApi
    @ExperimentalUnitApi
    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        val completeTransfer = intent.extras?.getBinder(COMPLETE_RESULT) as? CompleteTransfer
        completeResult = completeTransfer?.completeResult

        WindowCompat.setDecorFitsSystemWindows(window, true)

        readLanguage()
        var sourceText = ""
        var targetInit = ""
        completeResult?.let {
            val ocrSb = StringBuilder()
            val originSize = it.ocrResult?.blocks?.size ?: 0
            val transSize = it.transResponse?.result?.texts?.size ?: 0
            it.ocrResult?.blocks?.forEachIndexed { index, block ->
                ocrSb.append(block.text)
                if (index < originSize - 1) {
                    ocrSb.append("\n")
                }
            }
            sourceText = ocrSb.toString()

            val resSb = StringBuilder()
            it.transResponse?.result?.texts?.forEachIndexed { index, s ->
                resSb.append(s)
                if (index < transSize - 1) {
                    resSb.append("\n")
                }
            }
            targetInit = resSb.toString()
        }
        setContent {
            var loading by remember { mutableStateOf(false) }
            var targetText by remember { mutableStateOf(targetInit) }

            val systemUiController = rememberSystemUiController()
            SideEffect {
                // Update all of the system bar colors to be transparent, and use
                // dark icons if we're in light theme
                systemUiController.setSystemBarsColor(
                    color = Color(ContextCompat.getColor(this, R.color.actionbar_background_color)),
                )
                // setStatusBarsColor() and setNavigationBarsColor() also exist
            }
            GetContrastEditScreen(
                loading,
                sourceText,
                targetText,
                onNavIconPressed = { finish() },
                menuClick = {
                    when (it) {
                        is ContrastEditEvent.DoneEvent -> {
                            loading = true
                            val from = sourceLiveData.value ?: return@GetContrastEditScreen
                            val to = targetLiveData.value ?: return@GetContrastEditScreen

                            val texts = it.text.split(Regex("\\n"))
                            logEvent(PT_edit_start_translate, hashMapOf("language" to "${sourceLiveData.value}-${targetLiveData.value}"))
                            translateViewModel.startTranslate(from, to, texts)
                                .observe(this, {
                                    loading = false

                                    if (!isNetworkConnected()) {
                                        Toast.makeText(
                                            this,
                                            R.string.network_unavailable,
                                            Toast.LENGTH_SHORT
                                        ).show()
                                    }
                                    if (it.isSuccess) {
                                        val response = it.getOrNull()
                                        if (response != null) {
                                            val resSb = StringBuilder()
                                            response.result?.texts?.let {
                                                it.forEachIndexed { index, s ->
                                                    resSb.append(s)
                                                    if (index < it.size - 1) {
                                                        resSb.append("\n")
                                                    }
                                                }
                                            }
                                            Log.d("cjslog", "set target text:${resSb.toString()}")
                                            targetText = resSb.toString()
                                            logEvent(
                                                PT_edit_translate_success,
                                                hashMapOf("language" to "${sourceLiveData.value}-${targetLiveData.value}")
                                            )
                                        } else {
                                            Toast.makeText(
                                                this,
                                                R.string.translate_fail,
                                                Toast.LENGTH_SHORT
                                            ).show()
                                            logEvent(
                                                PT_edit_translate_failure,
                                                hashMapOf("language" to "${sourceLiveData.value}-${targetLiveData.value}")
                                            )

                                        }
                                    } else {
                                        Toast.makeText(
                                            this,
                                            R.string.translate_fail,
                                            Toast.LENGTH_SHORT
                                        )
                                            .show()
                                        logEvent(
                                            PT_edit_translate_failure,
                                            hashMapOf("language" to "${sourceLiveData.value}-${targetLiveData.value}")
                                        )
                                    }
                                })
                        }
                        is ContrastEditEvent.EditEvent -> {
                            logEvent(PT_edit_edit)
                        }
                        is ContrastEditEvent.CopyEvent -> {
                            clipboard(label = "Contrast", text = it.text)
                            Toast.makeText(this, R.string.copied_toast_contrast, Toast.LENGTH_SHORT).show()
                            logEvent(PT_edit_copy, hashMapOf("type" to it.type.toString()))
                        }
                    }

                }
            )
        }
    }

    private fun readLanguage() {
        val prefer = getSharedPreferences(PREFER_NAME, Context.MODE_PRIVATE) ?: return
        //val defaultTargetLang = PhotoTranslateController.targetLanguage

        var sourceLang =
            prefer.getString(PREFER_KEY_SOURCE_LANG, Locale.ENGLISH.language)
                ?: Locale.ENGLISH.language
        if (!ENG_SEQUENCE.contains(sourceLang)) {
            sourceLang = Locale.ENGLISH.language
        }
        val targetLang =
            prefer.getString(PREFER_KEY_TARGET_LANG, Locale.getDefault().language)
                ?: Locale.getDefault().language
        sourceLiveData.value = sourceLang
        targetLiveData.value = targetLang
    }

}

sealed class ContrastEditEvent {
    class DoneEvent(var text: String) : ContrastEditEvent()
    class EditEvent() : ContrastEditEvent()
    class CopyEvent(var text: String, var type: Int) : ContrastEditEvent()
}