package com.talpa.translate.ocr.result

import androidx.compose.foundation.*
import androidx.compose.foundation.interaction.MutableInteractionSource
import androidx.compose.foundation.layout.*
import androidx.compose.foundation.text.BasicTextField
import androidx.compose.foundation.text.KeyboardActions
import androidx.compose.foundation.text.KeyboardOptions
import androidx.compose.material.CircularProgressIndicator
import androidx.compose.material.MaterialTheme
import androidx.compose.material.Scaffold
import androidx.compose.material.Text
import androidx.compose.material.ripple.LocalRippleTheme
import androidx.compose.material.ripple.rememberRipple
import androidx.compose.runtime.*
import androidx.compose.ui.Alignment
import androidx.compose.ui.ExperimentalComposeUiApi
import androidx.compose.ui.Modifier
import androidx.compose.ui.focus.FocusRequester
import androidx.compose.ui.focus.focusRequester
import androidx.compose.ui.focus.onFocusChanged
import androidx.compose.ui.graphics.ColorFilter
import androidx.compose.ui.graphics.SolidColor
import androidx.compose.ui.platform.LocalFocusManager
import androidx.compose.ui.platform.LocalSoftwareKeyboardController
import androidx.compose.ui.res.colorResource
import androidx.compose.ui.res.painterResource
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.text.TextRange
import androidx.compose.ui.text.TextStyle
import androidx.compose.ui.text.font.FontWeight
import androidx.compose.ui.text.input.ImeAction
import androidx.compose.ui.text.input.TextFieldValue
import androidx.compose.ui.unit.ExperimentalUnitApi
import androidx.compose.ui.unit.dp
import androidx.compose.ui.unit.sp
import com.photo.translation.R
import com.talpa.translate.base.compose.HiAppbar
import com.talpa.translate.base.compose.RippleCustomTheme

/**
 * Create by chenjunsheng on 2021/8/13
 */

@ExperimentalComposeUiApi
@ExperimentalUnitApi
@Composable
fun GetContrastEditScreen(
    loading: Boolean,
    sourceText: String,
    targetText: String,
    onNavIconPressed: () -> Unit = { },
    menuClick: (ContrastEditEvent) -> Unit = { },
) {
    var textFieldValue by remember {
        mutableStateOf(TextFieldValue(sourceText, TextRange(sourceText.length)))
    }
    var textFieldFocus by remember { mutableStateOf(true) }
    val buttonEnable by remember { mutableStateOf(textFieldValue.text.isNotEmpty()) }
    val translateEnable = buttonEnable && !loading
    val keyboardController = LocalSoftwareKeyboardController.current
    val focusManager = LocalFocusManager.current
    val scroll = rememberScrollState()
    CompositionLocalProvider(
        LocalRippleTheme provides RippleCustomTheme,
    ) {
        val ripple = rememberRipple(
            bounded = false,
            radius = 20.dp,
            color = LocalRippleTheme.current.defaultColor()
        )
        val focusRequester = remember { FocusRequester() }
        Scaffold(
            topBar = {
                HiAppbar(
                    title = stringResource(id = R.string.edit_title),
                    onNavIconPressed = onNavIconPressed,
                    actions = {
                        Image(
                            painter = if (textFieldFocus) {
                                painterResource(id = R.drawable.ic_done)
                            } else {
                                painterResource(id = R.drawable.ic_contrast_edit)
                            },
                            contentDescription = "Translate",
                            modifier = Modifier
                                .padding(end = 12.dp)
                                .clickable(
                                    indication = ripple,
                                    interactionSource = remember { MutableInteractionSource() },
                                    onClick = {
                                        if (textFieldFocus) {
                                            focusManager.clearFocus()
                                            keyboardController?.hide()
                                            menuClick(ContrastEditEvent.DoneEvent(textFieldValue.text))
                                        } else {
                                            focusRequester.requestFocus()
                                            keyboardController?.show()
                                            menuClick(ContrastEditEvent.EditEvent())
                                        }

                                    },
                                    enabled = translateEnable
                                ),
                            colorFilter = ColorFilter.tint(
                                color = if (translateEnable) {
                                    colorResource(id = R.color.actionbar_icon_fillcolor)
                                } else {
                                    colorResource(id = R.color.contrast_done_unenable_color)
                                }
                            )
                        )

                    })
            },
            backgroundColor = colorResource(id = R.color.contrast_background_color)
        ) {

            Box {

                Column(
                    modifier = Modifier
                        .fillMaxSize()
                ) {
                    Box(
                        modifier = Modifier
                            .background(color = colorResource(id = R.color.actionbar_background_color))
                            .padding(horizontal = 16.dp)
                            .fillMaxWidth()
                            .height(56.dp)
                    ) {
                        Text(
                            text = stringResource(id = R.string.contrast_source_title),
                            fontSize = 16.sp,
                            color = colorResource(id = R.color.contrast_text_color),
                            fontWeight = FontWeight.Bold,
                            modifier = Modifier.align(Alignment.CenterStart)
                        )
                        Image(
                            painter = painterResource(id = R.drawable.ic_copy_contrast),
                            contentDescription = "Copy",
                            modifier = Modifier
                                .align(Alignment.CenterEnd)
                                .clickable(
                                    enabled = translateEnable,
                                    indication = ripple,
                                    interactionSource = remember { MutableInteractionSource() },
                                    onClick = {
                                        menuClick(
                                            ContrastEditEvent.CopyEvent(
                                                textFieldValue.text,
                                                1
                                            )
                                        )
                                    }
                                )
                        )
                    }
                    BasicTextField(
                        value = textFieldValue,
                        onValueChange = { textField ->
                            textFieldValue = textField
                        },
                        textStyle = TextStyle(
                            color = colorResource(id = R.color.contrast_text_color),
                            fontSize = 14.sp
                        ),
                        modifier = Modifier
                            .background(color = colorResource(id = R.color.contrast_background_color))
                            .padding(horizontal = 16.dp, vertical = 4.dp)
                            .requiredHeightIn(min = 56.dp)
                            .fillMaxWidth()
                            .weight(1f, false)
                            //.focusable()
                            .focusRequester(focusRequester)
                            .onFocusChanged {
                                textFieldFocus = it.isFocused
                            },
                        keyboardOptions = KeyboardOptions(imeAction = ImeAction.Done),
                        keyboardActions = KeyboardActions(onDone = {
                            //keyboardController?.hide()
                            if (translateEnable) {
                                //focusRequester.freeFocus()
                                focusManager.clearFocus()
                                keyboardController?.hide()
                                menuClick(ContrastEditEvent.DoneEvent(textFieldValue.text))
                            }
                        }),
                        cursorBrush = SolidColor(MaterialTheme.colors.primary)
                    )
                    DisposableEffect(Unit) {
                        focusRequester.requestFocus()
                        keyboardController?.show()
                        onDispose { }
                    }
                    Box(
                        modifier = Modifier
                            .background(color = colorResource(id = R.color.actionbar_background_color))
                            .padding(horizontal = 16.dp)
                            .fillMaxWidth()
                            .height(56.dp)
                    ) {
                        Text(
                            text = stringResource(id = R.string.contrast_target_title),
                            fontSize = 16.sp,
                            color = colorResource(id = R.color.contrast_text_color),
                            fontWeight = FontWeight.Bold,
                            modifier = Modifier.align(Alignment.CenterStart)
                        )
                        Image(
                            painter = painterResource(id = R.drawable.ic_copy_contrast),
                            contentDescription = "Copy",
                            modifier = Modifier
                                .align(Alignment.CenterEnd)
                                .clickable(
                                    indication = ripple,
                                    interactionSource = MutableInteractionSource(),
                                    onClick = {
                                        menuClick(
                                            ContrastEditEvent.CopyEvent(
                                                textFieldValue.text,
                                                2
                                            )
                                        )
                                    }
                                )
                        )
                    }

                    Text(
                        text = targetText,
                        modifier = Modifier
                            .background(color = colorResource(id = R.color.actionbar_background_color))
                            .padding(16.dp)
                            .requiredHeightIn(min = 56.dp)
                            .fillMaxWidth()
                            .weight(1f, false)
                            .verticalScroll(scroll)
                    )
                }

                if (loading) {
                    Spacer(modifier = Modifier.height(56.dp))
                    CircularProgressIndicator(
                        modifier = Modifier
                            .padding(top = 16.dp)
                            .size(24.dp)
                            .align(Alignment.TopCenter),
                        color = MaterialTheme.colors.primary
                    )
                }
            }


        }
    }
}