package com.talpa.translate.ocr.result

import androidx.compose.foundation.Image
import androidx.compose.foundation.clickable
import androidx.compose.foundation.interaction.MutableInteractionSource
import androidx.compose.foundation.layout.*
import androidx.compose.foundation.lazy.LazyColumn
import androidx.compose.material.Scaffold
import androidx.compose.material.Text
import androidx.compose.material.TextButton
import androidx.compose.material.ripple.LocalRippleTheme
import androidx.compose.material.ripple.rememberRipple
import androidx.compose.runtime.Composable
import androidx.compose.runtime.CompositionLocalProvider
import androidx.compose.runtime.remember
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.res.colorResource
import androidx.compose.ui.res.painterResource
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.text.font.FontFamily
import androidx.compose.ui.text.font.FontStyle
import androidx.compose.ui.text.font.FontWeight
import androidx.compose.ui.unit.ExperimentalUnitApi
import androidx.compose.ui.unit.dp
import androidx.compose.ui.unit.sp
import com.photo.translation.R
import com.talpa.translate.base.compose.HiAppbar
import com.talpa.translate.base.compose.RippleCustomTheme
import com.talpa.translate.ocr.datasource.CompleteResult
import kotlin.math.min

/**
 * Create by chenjunsheng on 2021/8/13
 */

@ExperimentalUnitApi
@Composable
fun GetContrastScreen(
    completeResult: CompleteResult?,
    onNavIconPressed: () -> Unit = { },
    menuClick: () -> Unit = { },
    itemClick: (String, String) -> Unit = { _, _ -> }
) {
    CompositionLocalProvider(LocalRippleTheme provides RippleCustomTheme) {
        val ripple = rememberRipple(
            bounded = false,
            radius = 20.dp,
            color = LocalRippleTheme.current.defaultColor()
        )
        Scaffold(
            topBar = {
                HiAppbar(
                    title = stringResource(id = R.string.contrast_title),
                    onNavIconPressed = onNavIconPressed,
                    actions = {
                        Text(
                            text = stringResource(id = R.string.edit_title),
                            fontSize = 14.sp,
                            color = colorResource(id = R.color.contrast_text_color),
                            fontFamily = FontFamily.SansSerif,
                            fontWeight = FontWeight.SemiBold,
                            modifier = Modifier
                                .padding(end = 8.dp)
                                .clickable(
                                    indication = ripple,
                                    interactionSource = MutableInteractionSource(),
                                    onClick = { menuClick() })
                                .padding(4.dp)
                        )

                    })
            },
            backgroundColor = colorResource(id = R.color.actionbar_background_color)
        ) {
            LazyColumn(modifier = Modifier.fillMaxSize()) {
                val blockList = completeResult?.ocrResult?.blocks
                val responseList = completeResult?.transResponse?.result?.texts
                items(min(blockList?.size ?: 0, responseList?.size ?: 0)) { index ->
                    Column(
                        modifier = Modifier
                            .wrapContentHeight()
                            .fillMaxWidth()
                            .padding(horizontal = 16.dp, vertical = 16.dp)
                    ) {
                        val ocrText = blockList?.get(index)?.text
                        val translation = responseList?.get(index)
                        if (!ocrText.isNullOrEmpty() && !translation.isNullOrEmpty()) {
                            Text(
                                text = ocrText,
                                color = colorResource(id = R.color.contrast_text_color),
                                fontSize = 14.sp,
                                modifier = Modifier
                                    .wrapContentSize()
                            )
                            Spacer(modifier = Modifier.height(3.dp))
                            Text(
                                text = translation,
                                color = colorResource(id = R.color.contrast_text_color2),
                                fontSize = 14.sp,
                                modifier = Modifier
                                    .wrapContentSize()
                            )
                            Spacer(modifier = Modifier.height(4.dp))
                            Box(
                                modifier = Modifier
                                    .fillMaxWidth()
                                    .wrapContentHeight()
                            ) {
                                Image(
                                    painter = painterResource(id = R.drawable.ic_copy_contrast),
                                    contentDescription = "Copy",
                                    modifier = Modifier
                                        .align(Alignment.CenterEnd)
                                        .clickable(
                                            interactionSource = MutableInteractionSource(),
                                            indication = ripple,
                                            onClick = {
                                                itemClick(ocrText, translation)
                                            }
                                        )
                                )
                            }
                        }

                    }
                }
            }
        }
    }

}