package com.talpa.translate.ocr

import android.content.Context
import android.graphics.Bitmap
import android.graphics.Rect
import android.util.Log
import com.google.api.client.http.HttpRequest
import com.google.api.client.http.HttpRequestInitializer
import com.google.api.client.http.javanet.NetHttpTransport
import com.google.api.client.json.JsonFactory
import com.google.api.client.json.gson.GsonFactory
import com.google.api.services.vision.v1.Vision
import com.google.api.services.vision.v1.VisionRequest
import com.google.api.services.vision.v1.VisionRequestInitializer
import com.google.api.services.vision.v1.model.*
import com.photo.translation.BuildConfig
import com.photo.translation.R
import com.talpa.translate.base.common.FrameMetadata
import com.talpa.translate.base.utils.*
import com.talpa.translate.ocr.datasource.TransParams
import com.talpa.translate.ocr.exception.NoContentException
import com.talpa.translate.ocr.result.Block
import com.talpa.translate.ocr.result.OcrResult
import java.io.IOException
import java.lang.IllegalArgumentException
import kotlin.math.abs
import kotlin.math.max
import kotlin.math.min

/**
 * Create by chenjunsheng on 2020/7/29
 */
class GoogleRecognizer(var context: Context) : Recognizer {

    companion object {
        const val ANDROID_PACKAGE_HEADER = "X-Android-Package"
        const val ANDROID_CERT_HEADER = "X-Android-Cert"
        const val TAG = "Recognizer"
        const val TEST_PACKAGE_NAME = "com.talpa.translate"
        const val TEST_SHA1 = "ebd663baf5c6e7e4139675407b8216914bca417a"
    }

    private lateinit var mTransParams: TransParams
    private var mHistory: OcrResult? = null

    override fun setup(transParams: TransParams) {
        mTransParams = transParams
    }

    override suspend fun doOcr(bitmap: Bitmap): OcrResult {
        Log.d(TAG, "start google ocr")
        logEvent(PT_ocr_with_google_vision)
        val httpTransport = NetHttpTransport()
        val jsonFactory: JsonFactory = GsonFactory.getDefaultInstance()
        val requestInitializer: VisionRequestInitializer =
            object : VisionRequestInitializer(BuildConfig.CLOUD_VISION_API_KEY) {
                /**
                 * We override this so we can inject important identifying fields into the HTTP
                 * headers. This enables use of a restricted cloud platform API key.
                 */
                @Throws(IOException::class)
                override fun initializeVisionRequest(visionRequest: VisionRequest<*>) {
                    super.initializeVisionRequest(visionRequest)
                    if (BuildConfig.DEBUG) {
                        val packageName: String = TEST_PACKAGE_NAME
                        visionRequest.requestHeaders[ANDROID_PACKAGE_HEADER] = packageName
                        //val sig: String? = getSignature(context.packageManager, packageName)
                        visionRequest.requestHeaders[ANDROID_CERT_HEADER] = TEST_SHA1
                    } else {
                        val packageName: String = context.packageName
                        visionRequest.requestHeaders[ANDROID_PACKAGE_HEADER] = packageName
                        val sig: String? = getSignature(context.packageManager, packageName)
                        visionRequest.requestHeaders[ANDROID_CERT_HEADER] = sig
                    }
                }
            }

        val builder = Vision.Builder(httpTransport, jsonFactory, object : HttpRequestInitializer {
            override fun initialize(request: HttpRequest?) {
                request?.setReadTimeout(8 * 1000)
                request?.setConnectTimeout(8 * 1000)
                request?.setWriteTimeout(8 * 1000)
            }

        })
        builder.setVisionRequestInitializer(requestInitializer)
        val vision = builder.build()
        val batchAnnotateImagesRequest = BatchAnnotateImagesRequest().apply {
            requests = arrayListOf(AnnotateImageRequest()
                .also
                { imageRequest ->

                    // Add the image
                    val base64EncodedImage = Image()
                    // Convert the bitmap to a JPEG
                    // Just in case it's a format that Android understands but Cloud Vision
                    base64EncodedImage.content = PhotoAnalyzer.getBase64String(bitmap)
                    imageRequest.image = base64EncodedImage

                    // add the features we want
                    imageRequest.features = arrayListOf(Feature().also { feature ->
                        feature.type = "TEXT_DETECTION"
                    })
                })
        }

        val annotateRequest =
            vision.images().annotate(batchAnnotateImagesRequest)
        // Due to a bug: requests to Vision API containing large images fail when GZipped.
        annotateRequest.disableGZipContent = true

        val response: BatchAnnotateImagesResponse = try {
            annotateRequest.execute()
        } catch (e: IOException) {
            throw NoContentException(context.getString(R.string.network_unavailable))
        }

        val blocks = arrayListOf<Block>()
        response.responses?.forEach { res ->

            res.fullTextAnnotation ?: return OcrResult(arrayListOf())
            if (res.fullTextAnnotation.isEmpty())
                return OcrResult(arrayListOf())

            val totalStr = res.fullTextAnnotation.text
            val sb = StringBuilder()
            res.fullTextAnnotation.pages.forEach {
                it.blocks.forEach {
                    sb.clear()
                    val paragraph = it.paragraphs.firstOrNull()
                    val x0 = paragraph?.boundingBox?.vertices?.get(0)?.x ?: 0
                    val y0 = paragraph?.boundingBox?.vertices?.get(0)?.y ?: 0
                    val x1 = paragraph?.boundingBox?.vertices?.get(2)?.x ?: 0
                    val y1 = paragraph?.boundingBox?.vertices?.get(2)?.y ?: 0
                    paragraph?.words?.forEach {
                        it.symbols.forEach {
                            sb.append(it.text)
                        }
                        paragraph?.property?.detectedLanguages?.firstOrNull()?.let {
                            if (firebaseSourceSupportLanguages.contains(it.languageCode, true)) {
                                sb.append(" ")
                            }
                        }
                        //sb.append(" ")
                    }
                    val block = Block(sb.trimEnd().toString(), Rect(x0, y0, x1, y1))
                    blocks.add(block)
                }
            }


/*            res.textAnnotations ?: return OcrResult(emptyList())
            if (res.textAnnotations.isEmpty()) {
                return OcrResult(emptyList())
            }
            var blockText: List<String>? = null
            val sb = StringBuilder()
            var skipToIndex = 0
            var blockIndex = 0
            res.textAnnotations.filter {
                it.description.isNotEmpty()
            }.forEachIndexed { index, entityAnnotation ->
                sb.clear()
                if (index < skipToIndex) {
                    return@forEachIndexed
                }
                if (index == 0) {
                    blockText = entityAnnotation.description.lines().filter {
                        it.isNotEmpty()
                    }
                } else {
                    if (!blockText.isNullOrEmpty()) {
                        sb.append(entityAnnotation.description)
                        val rect = Rect()
                        if (sb.toString().length == (blockText!![blockIndex]).length) {
                            val x0 = entityAnnotation.boundingPoly?.vertices?.get(0)?.x ?: 0
                            val y0 = entityAnnotation.boundingPoly?.vertices?.get(0)?.y ?: 0
                            val x1 = entityAnnotation.boundingPoly?.vertices?.get(2)?.x ?: 0
                            val y1 = entityAnnotation.boundingPoly?.vertices?.get(2)?.y ?: 0
                            rect.set(x0, y0, x1, y1)
                            val metadata = mTransParams.metaData

                            if (rect.right >= metadata.width) {
                                rect.right = metadata.width - 1
                            }
                            if (rect.bottom >= metadata.height) {
                                rect.bottom = metadata.height - 1
                            }
                            if (rect.left <= 0) {
                                rect.left = 0
                            }
                            if (rect.top <= 0) {
                                rect.top = 0
                            }
                            blockIndex ++
                            if (rect.left + rect.width() > metadata.width) {
                                return@forEachIndexed
                            }
                            if (rect.top + rect.height() > metadata.height) {
                                return@forEachIndexed
                            }
                            if (rect.width() <= 0 || rect.height() <= 0) {
                                return@forEachIndexed
                            }
                            blocks.add(Block(sb.toString(), rect))

                        } else if (sb.toString().length < (blockText!![blockIndex]).length) {
                            val startIndex = index
                            val startAnnotation = entityAnnotation
                            var endAnnotation = startAnnotation
                            val blockStr = blockText!![blockIndex]

                            sb.clear()
                            kotlin.run {
                                var x0: Int = startAnnotation.boundingPoly?.vertices?.get(0)?.x ?: 0
                                var y0: Int = startAnnotation.boundingPoly?.vertices?.get(0)?.y ?: 0
                                var x1: Int = startAnnotation.boundingPoly?.vertices?.get(2)?.x ?: 0
                                var y1: Int = startAnnotation.boundingPoly?.vertices?.get(2)?.y ?: 0
                                res.textAnnotations.subList(startIndex, res.textAnnotations.size - 1)
                                    .forEachIndexed { index1, entityAnnotation ->
                                        if (sb.toString().length < blockStr.length) {
                                            //x0 = entityAnnotation.boundingPoly?.vertices?.get(0)?.x ?: x0
                                            //y0 = entityAnnotation.boundingPoly?.vertices?.get(0)?.y ?: y0
                                            x1 = entityAnnotation.boundingPoly?.vertices?.get(0)?.x ?: x1
                                            y1 = entityAnnotation.boundingPoly?.vertices?.get(2)?.y ?: y1
                                            sb.append(entityAnnotation.description)
                                            endAnnotation = entityAnnotation
                                            blockStr.let { s ->
                                                if (s.length >= sb.length) {
                                                    s.substring(sb.length).forEach {
                                                        if (it.isWhitespace()) {
                                                            sb.append(it)
                                                        } else {
                                                            return@let
                                                        }
                                                    }
                                                }
                                            }
                                        } else {
                                            skipToIndex = res.textAnnotations.indexOf(endAnnotation) + 1
                                            rect.set(x0, y0, x1, y1)
                                            val metadata = mTransParams.metaData
                                            if (rect.right >= metadata.width) {
                                                rect.right = metadata.width - 1
                                            }
                                            if (rect.bottom >= metadata.height) {
                                                rect.bottom = metadata.height - 1
                                            }
                                            if (rect.left <= 0) {
                                                rect.left = 0
                                            }
                                            if (rect.top <= 0) {
                                                rect.top = 0
                                            }
                                            blockIndex ++
                                            if (rect.left + rect.width() > metadata.width) {
                                                return@run
                                            }
                                            if (rect.top + rect.height() > metadata.height) {
                                                return@run
                                            }
                                            if (rect.width() <= 0 || rect.height() <= 0) {
                                                return@run
                                            }
                                            blocks.add(Block(sb.toString(), rect))

                                            return@run
                                        }
                                    }
                            }


                        } else {
                            return@forEachIndexed
                        }
                    }
                }
            }*/

        }

        /*val resultBlocks = arrayListOf<Block>()
        val containsIndexs = arrayListOf<Int>()
        response.responses?.forEach {
            it.fullTextAnnotation.pages?.firstOrNull()?.blocks?.forEach {
                val x0 = it.boundingBox?.vertices?.get(0)?.x ?: 0
                val y0 = it.boundingBox?.vertices?.get(0)?.y ?: 0
                val x1 = it.boundingBox?.vertices?.get(2)?.x ?: 0
                val y1 = it.boundingBox?.vertices?.get(2)?.y ?: 0
                val pageRect = Rect(x0, y0, x1, y1)

                containsIndexs.clear()
                blocks.forEachIndexed { index, block ->
                    if (pageRect.contains(block.rect)) {
                        containsIndexs.add(index)
                    }
                }
                val sb = StringBuilder()
                containsIndexs.forEach {
                    sb.append(blocks[it].text)
                    sb.append(" ")
                }

                val resultBlock = Block(sb.toString().trim(), pageRect)
                resultBlocks.add(resultBlock)
                containsIndexs.reversed().forEach {
                    blocks.removeAt(it)
                }

            }
        }*/

        return OcrResult(blocks.map {
            val metadata = mTransParams.metaData
            if (it.rect.right >= metadata.width) {
                it.rect.right = metadata.width - 1
            }
            if (it.rect.bottom >= metadata.height) {
                it.rect.bottom = metadata.height - 1
            }
            if (it.rect.left <= 0) {
                it.rect.left = 0
            }
            if (it.rect.top <= 0) {
                it.rect.top = 0
            }
            it
        }.filter {
            val metadata = mTransParams.metaData
            if (it.rect.left + it.rect.width() > metadata.width) {
                return@filter false
            }
            if (it.rect.top + it.rect.height() > metadata.height) {
                return@filter false
            }
            if (it.rect.width() <= 0 || it.rect.height() <= 0) {
                return@filter false
            }
            return@filter true
        }.toMutableList()).also { mHistory = it }
    }

    override fun getFrameMetadata(): FrameMetadata? {
        return mTransParams.metaData
    }

    override fun getHistory(): OcrResult? {
        return mHistory
    }

    override fun getType(): Int {
        return TYPE_GOOGLE_VISION
    }
}