package com.talpa.translate.ocr

import android.app.Application
import android.content.Context
import android.graphics.Bitmap
import android.util.Log
import androidx.lifecycle.*
import com.google.mlkit.vision.label.ImageLabel
import com.photo.translation.R
import com.talpa.translate.base.common.FrameMetadata
import com.talpa.translate.base.state.PhotoMLState
import com.talpa.translate.factory.TranslatorFactory
import com.talpa.translate.imglabel.ImageLabelData
import com.talpa.translate.imglabel.ImageLabelWorker
import com.talpa.translate.network.HiTranslator
import com.talpa.translate.ocr.datasource.CompleteResult
import com.talpa.translate.ocr.datasource.TransParams
import com.talpa.translate.ocr.exception.NoContentException
import com.talpa.translate.ocr.result.OcrResult
import com.talpa.translate.render.Render
import com.talpa.translate.repository.box.collins.SenseNew
import com.talpa.translate.repository.net.dictionary.DictionaryRepo
import kotlinx.coroutines.*
import kotlinx.coroutines.flow.*
import java.lang.IllegalArgumentException
import java.util.*
import kotlin.math.min


/**
 * Photo Translation
 *
 * @author CY 2020-02-05
 */
class ImageTranslate(application: Application) : AndroidViewModel(application) {

    companion object {
        const val TAG = "ocr_translate"
    }

    val sourceLiveData: MutableLiveData<String> = MutableLiveData("en")
    val targetLiveData: MutableLiveData<String> = MutableLiveData("en")
    val objectSource: MutableLiveData<String> = MutableLiveData("en")
    val objectTarget: MutableLiveData<String> = MutableLiveData("en")

    fun getSourceData(state: PhotoMLState): MutableLiveData<String> {
        if (state.getType() == PhotoMLState.TYPE_TEXT) {
            return sourceLiveData
        } else {
            return objectSource
        }
    }

    fun getTargetData(state: PhotoMLState): MutableLiveData<String> {
        if (state.getType() == PhotoMLState.TYPE_TEXT) {
            return targetLiveData
        } else {
            return objectTarget
        }
    }

    private lateinit var mTranslator: Recognizer
    private val mRender: Render

    init {
        val analyzer: IAnalyzer = PhotoAnalyzer(getApplication<Application>())
        mRender = TranslatorFactory.getRender(getApplication<Application>(), analyzer)
    }

    fun imageLabelChangeLanguage(
        imageLabel: ImageLabelData,
        sourceLanguage: String,
        targetLanguage: String
    ): LiveData<Result<ImageLabelData>> {
        val dictRepo = DictionaryRepo()
        return flow {
            emit(
                Result.success(
                    withContext(Dispatchers.IO) {
                        val count = min(imageLabel.source.size, imageLabel.target.size)
                        val sourceSenses = arrayListOf<SenseNew>()
                        val targetSenses = arrayListOf<SenseNew>()
                        repeat(count) { index ->
                            val sourceList = async {
                                dictRepo.search(
                                    imageLabel.source[index].data?.word
                                        ?: throw IllegalArgumentException(
                                            getApplication<Application>().getString(
                                                R.string.translate_fail
                                            )
                                        ),
                                    imageLabel.source[index].targetLanTag
                                        ?: throw IllegalArgumentException(
                                            getApplication<Application>().getString(
                                                R.string.translate_fail
                                            )
                                        ),
                                    sourceLanguage,
                                    getApplication()
                                ).also {
                                    it.sourceLanTag = imageLabel.source[index].targetLanTag
                                    it.targetLanTag = sourceLanguage
                                }
                            }
                            val targetList = async {
                                dictRepo.search(
                                    imageLabel.target[index].data?.word
                                        ?: throw IllegalArgumentException(
                                            getApplication<Application>().getString(
                                                R.string.translate_fail
                                            )
                                        ),
                                    imageLabel.target[index].targetLanTag
                                        ?: throw IllegalArgumentException(
                                            getApplication<Application>().getString(
                                                R.string.translate_fail
                                            )
                                        ),
                                    targetLanguage,
                                    getApplication()
                                ).also {
                                    it.sourceLanTag = imageLabel.target[index].targetLanTag
                                    it.targetLanTag = targetLanguage
                                }
                            }
                            sourceSenses.add(sourceList.await())
                            targetSenses.add(targetList.await())
                        }
                        ImageLabelData(imageLabel.sourceBitmap, sourceSenses, targetSenses)
                    }
                )

            )

        }.catch {
            Log.d(TAG, "label imageLabelChangeLanguage error", it)
            emit(Result.failure(it))
        }.flowOn(Dispatchers.IO)
            .asLiveData()
    }

    /**
     * 物体识别
     */
    fun itemRecognize(
        bitmap: Bitmap,
        metadata: FrameMetadata,
        sourceLanguage: String,
        targetLanguage: String
    ): LiveData<Result<ImageLabelData>> {
        val dictRepo = DictionaryRepo()
        return flow {
            val itemRecognizer =
                ImageLabelWorker(
                    getApplication<Application>(),
                    bitmap,
                    metadata,
                    sourceLanguage,
                    targetLanguage
                )
            val labels = itemRecognizer.doItemRecognize()
            emit(labels)
        }.flatMapConcat { labels ->

            flow {

                emit(
                    Result.success(
                        withContext(Dispatchers.IO) {
                            val sourceSenses = arrayListOf<SenseNew>()
                            val targetSenses = arrayListOf<SenseNew>()
                            labels.forEach {

                                val sourceList = async {
                                    dictRepo.search(
                                        it.text,
                                        Locale.ENGLISH.language,
                                        sourceLanguage,
                                        getApplication()
                                    ).also {
                                        it.sourceLanTag = Locale.ENGLISH.language
                                        it.targetLanTag = sourceLanguage
                                    }
                                }
                                val targetList = async {
                                    dictRepo.search(
                                        it.text,
                                        Locale.ENGLISH.language,
                                        targetLanguage,
                                        getApplication()
                                    ).also {
                                        it.sourceLanTag = Locale.ENGLISH.language
                                        it.targetLanTag = targetLanguage
                                    }
                                }
                                sourceSenses.add(sourceList.await())
                                targetSenses.add(targetList.await())

                            }

                            ImageLabelData(bitmap, sourceSenses, targetSenses)
                        }
                    )

                )
            }
        }
            .catch {
                Log.d(TAG, "label itemRecognize error", it)
                emit(Result.failure(it))
            }
            .flowOn(Dispatchers.IO)
            .asLiveData()
    }

    fun doOcrOnly(bitmap: Bitmap, metadata: FrameMetadata): LiveData<Result<OcrResult>> {
        return flow {
            val ocrResult = mTranslator.doOcr(bitmap)
            emit(Result.success(ocrResult))
        }.onStart {
            mTranslator = OcrDispatcher.dispatchOcrTranslator(
                getApplication<Application>(),
                Locale.ENGLISH.language
            )
            //mRender.initialize(bitmap)
            mTranslator.setup(TransParams(metadata, Locale.ENGLISH.language))
        }.catch {
            //mRender.release()
            Log.d(TAG, "text doOcrOnly error", it)
            emit(Result.failure(it))
        }.flowOn(Dispatchers.IO)
            .asLiveData()
    }

    /**
     * 翻译
     */
    fun translate(
        bitmap: Bitmap,
        metadata: FrameMetadata,
        sourceLanguage: String,
        targetLanguage: String
    ): LiveData<Result<CompleteResult>> {
        val completeResult = CompleteResult(sourceLanguage, targetLanguage)
        return flow {
            val ocrResult = mTranslator.doOcr(bitmap) //Step1:执行OCR
            completeResult.ocrResult = ocrResult
            emit(completeResult)
        }.flatMapConcat { completeResult ->
            flow {
                val ocrResult = completeResult.ocrResult
                    ?: throw NoContentException(getApplication<Application>().getString(R.string.no_content_identified))
                val texts = arrayListOf<String>()
                ocrResult.blocks.forEach { texts.add(it.text) }
                if (texts.isEmpty()) {
                    throw NoContentException(getApplication<Application>().getString(R.string.no_content_identified))
                }
                val transResult = HiTranslator.getInstance(getApplication<Application>())
                    .postTranslate(
                        from = sourceLanguage,
                        to = targetLanguage,
                        texts = texts
                    )  //Step2:OCR结果提交翻译
                val textsResponse = transResult?.result?.texts
                    ?: throw NoContentException(getApplication<Application>().getString(R.string.translate_fail))
                if (textsResponse.isEmpty()) throw NoContentException(
                    getApplication<Application>().getString(
                        R.string.no_trans_for_ocr
                    )
                )
                completeResult.transResponse = transResult
                emit(completeResult)
            }.zip(flow {
                val renderSource = mRender.analyzeOcrResult(bitmap, mTranslator)
                emit(renderSource)
            }) { completeResult, renderSource ->
                val list = completeResult.transResponse?.result?.texts ?: throw NoContentException(
                    getApplication<Application>().getString(R.string.translate_fail)
                )
                val result = mRender.renderOverlay(renderSource, list, mTranslator)
                result
                    ?: throw NoContentException(getApplication<Application>().getString(R.string.generate_bitmap_fail))
                completeResult.renderResult = result
                Result.success(completeResult)
            }
        }
            .onStart {
                mTranslator = OcrDispatcher.dispatchOcrTranslator(
                    getApplication<Application>(),
                    sourceLanguage
                )
                //mRender.initialize(bitmap)
                mTranslator.setup(TransParams(metadata, sourceLanguage))
            }.catch {
                //mRender.release()
                Log.d(TAG, "text translate error", it)
                emit(Result.failure(it))
            }.flowOn(Dispatchers.IO)
            .asLiveData()
    }

    fun changeTargetLanguage(
        bitmap: Bitmap,
        sourceLanguage: String,
        targetLanguage: String
    ): LiveData<Result<CompleteResult>> {
        val completeResult = CompleteResult(sourceLanguage, targetLanguage)
        return flow {
            val history = mTranslator.getHistory() ?: throw NoContentException(
                getApplication<Application>().getString(R.string.no_trans_for_ocr)
            )
            completeResult.ocrResult = history
            emit(completeResult)
        }.flatMapConcat { completeResult ->
            flow {
                val history = completeResult.ocrResult
                val texts = arrayListOf<String>()
                history?.blocks?.forEach { block ->
                    texts.add(block.text)
                }
                val transResult = HiTranslator.getInstance(getApplication<Application>())
                    .postTranslate(from = sourceLanguage, to = targetLanguage, texts = texts)

                val textsResponse = transResult?.result?.texts ?: throw NoContentException(
                    getApplication<Application>().getString(R.string.translate_fail)
                )
                if (textsResponse.isEmpty()) throw NoContentException(
                    getApplication<Application>().getString(R.string.no_trans_for_ocr)
                )
                completeResult.transResponse = transResult
                emit(completeResult)
            }.zip(
                flow {
                    val renderSource = mRender.analyzeOcrResult(bitmap, mTranslator)
                    emit(renderSource)
                }
            ) { completeResult, renderSource ->
                val list = completeResult.transResponse?.result?.texts ?: throw NoContentException(
                    getApplication<Application>().getString(R.string.translate_fail)
                )
                val result =
                    mRender.renderOverlay(renderSource, list, mTranslator)
                        ?: throw NoContentException(
                            getApplication<Application>().getString(R.string.generate_bitmap_fail)
                        )
                completeResult.renderResult = result
                return@zip Result.success(completeResult)
            }
        }
            .catch {
                Log.d(TAG, "text changeTargetLanguage error", it)
                emit(Result.failure(it))
            }.flowOn(Dispatchers.IO)
            .asLiveData()
    }
}