package com.talpa.translate.camera

import android.app.Activity
import android.content.Context
import android.content.DialogInterface
import android.content.Intent
import android.graphics.Bitmap
import android.os.Bundle
import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import android.widget.Checkable
import android.widget.TextView
import android.widget.Toast
import androidx.appcompat.app.AlertDialog
import androidx.constraintlayout.motion.widget.MotionLayout
import androidx.core.view.doOnPreDraw
import androidx.fragment.app.Fragment
import androidx.lifecycle.Observer
import androidx.lifecycle.ViewModelProvider
import androidx.recyclerview.widget.RecyclerView
import com.bumptech.glide.Glide
import com.bumptech.glide.load.DataSource
import com.bumptech.glide.load.engine.GlideException
import com.bumptech.glide.request.RequestListener
import com.bumptech.glide.request.target.Target
import com.photo.translation.R
import com.photo.translation.databinding.CameraPickerFragmentBinding
import com.talpa.translate.activity.OCR_RESULT
import com.talpa.translate.base.utils.*
import com.talpa.translate.base.view.drag.DragSelectTouchListener
import com.talpa.translate.base.view.drag.DragSelectionProcessor
import com.talpa.translate.datasource.CameraCompatSource
import com.talpa.translate.datasource.CameraPickerSource
import com.talpa.translate.datasource.ICameraSource
import com.talpa.translate.ocr.ImageTranslate
import com.talpa.translate.ocr.result.Block
import com.talpa.translate.ocr.result.OcrResult
import com.talpa.translate.ocr.result.OcrTransferWrapper
import java.util.*

/**
 * Create by chenjunsheng on 2021/3/18
 */
class CameraPickerFragment : Fragment(R.layout.camera_picker_fragment), View.OnClickListener,
    ICameraSource.Status {

    private lateinit var binding: CameraPickerFragmentBinding
    internal lateinit var mCameraSource: CameraCompatSource
    internal val translateViewModel: ImageTranslate by lazy {
        ViewModelProvider(
            this,
            ViewModelProvider.AndroidViewModelFactory.getInstance(activity?.application!!)
        ).get(ImageTranslate::class.java)
    }

    override fun onViewCreated(view: View, savedInstanceState: Bundle?) {
        super.onViewCreated(view, savedInstanceState)
        binding = CameraPickerFragmentBinding.bind(view)
        mCameraSource = CameraPickerSource()
        mCameraSource.initialize(requireContext(), binding.previewContainer, this)
        mCameraSource.setStatusCallback(this)
        binding.btnControl.setOnClickListener(this)
        binding.ivFinish.setOnClickListener(this)
        binding.ocrResult.submit.setOnClickListener(this)
        binding.torchSelector.setOnClickListener(this)
    }

    override fun onClick(v: View) {
        when (v.id) {
            R.id.btn_control -> {
                logEvent(PT_camera_picker_click)
                mCameraSource.takePicture()
            }
            R.id.iv_finish -> {
                logEvent(PT_camera_picker_back_click)
                activity?.finish()
            }
            R.id.submit -> {
                val adapter = binding.ocrResult.recyclerView.adapter as PickerParagraphAdapter
                val blocks = adapter.getSelection().map {
                    adapter.getItem(it)
                }.toTypedArray()
                logEvent(PT_camera_picker_submit)
                val bundle = Bundle()
                bundle.putBinder(OCR_RESULT, OcrTransferWrapper(blocks))
                activity?.setResult(Activity.RESULT_OK, Intent().putExtra(OCR_RESULT, bundle))
                activity?.finish()
            }
            R.id.torch_selector -> {
                val isChecked = (v as? Checkable)?.isChecked
                logEvent(PT_camera_picker_torch)
                mCameraSource.torch(isChecked == true)
            }
        }
    }

    override fun onPause() {
        super.onPause()
        mCameraSource.torch(false)
    }

    override fun cameraInitializeFail(exception: Exception) {
        showErrorAlert(requireContext(), exception)
    }

    override fun onPicturetaked(data: ByteArray, rotation: Int) {
        mCameraSource.torch(switch = false)
        Glide.with(requireContext())
            .asBitmap()
            .load(data)
            .listener(object : RequestListener<Bitmap> {
                override fun onLoadFailed(
                    e: GlideException?,
                    model: Any?,
                    target: Target<Bitmap>?,
                    isFirstResource: Boolean
                ): Boolean {
                    return false
                }

                override fun onResourceReady(
                    resource: Bitmap,
                    model: Any?,
                    target: Target<Bitmap>?,
                    dataSource: DataSource?,
                    isFirstResource: Boolean
                ): Boolean {
                    binding.loading.root.visibility = View.VISIBLE
                    binding.ocrResult.root.visibility = View.VISIBLE
                    binding.bottomControl.visibility = View.INVISIBLE
                    translateViewModel.doOcrOnly(resource, rotation)
                        .observe(this@CameraPickerFragment,
                            Observer {
                                binding.loading.root.visibility = View.GONE
                                if (it.isSuccess) {
                                    val ocrResult = it.getOrNull()
                                    ocrResult?.let {
                                        showChooseDialog(it)
                                    } ?: kotlin.run {
                                        Toast.makeText(
                                            context,
                                            R.string.no_content_identified,
                                            Toast.LENGTH_SHORT
                                        ).show()
                                        enterCameraState()
                                    }
                                } else {
                                    Toast.makeText(
                                        context,
                                        R.string.no_content_identified,
                                        Toast.LENGTH_SHORT
                                    ).show()
                                    enterCameraState()
                                }
                            })
                    return false
                }

            })
            .into(binding.ocrResult.ivPreview)
    }

    internal fun enterCameraState() {
        binding.ocrResult.root.visibility = View.INVISIBLE
        binding.bottomControl.visibility = View.VISIBLE
        binding.ocrResult.recyclerView.adapter = null
    }

    private fun showChooseDialog(ocrResult: OcrResult) {
        val adapter = PickerParagraphAdapter(ocrResult)
        // 2) Add the DragSelectListener
        val dragSelectionProcessor = DragSelectionProcessor(object :
            DragSelectionProcessor.ISelectionHandler {

            override fun getSelection(): MutableSet<Int> {
                return adapter.getSelection()
            }

            override fun isSelected(index: Int): Boolean {
                return adapter.getSelection().contains(index)
            }

            override fun updateSelection(
                start: Int,
                end: Int,
                isSelected: Boolean,
                calledFromOnStart: Boolean
            ) {
                adapter.selectRange(start, end, isSelected)
                binding.ocrResult.submit.isEnabled = adapter.getSelection().isNotEmpty()

            }
        })
            .withMode(DragSelectionProcessor.Mode.Simple)
        var dragSelectTouchListener = DragSelectTouchListener()
            .withSelectListener(dragSelectionProcessor)
        adapter.setClickListener(object : PickerParagraphAdapter.ItemClickListener {
            override fun onItemClick(view: View, position: Int) {
                adapter.toggleSelection(position)
                binding.ocrResult.submit.isEnabled = adapter.getSelection().isNotEmpty()
            }

            override fun onItemLongClick(view: View, position: Int): Boolean {
                // if one item is long pressed, we start the drag selection like following:
                // we just call this function and pass in the position of the first selected item
                // the selection processor does take care to update the positions selection mode correctly
                // and will correctly transform the touch events so that they can be directly applied to your adapter!!!
                dragSelectTouchListener.startDragSelection(position)
                return true
            }
        })
        binding.ocrResult.recyclerView.addOnItemTouchListener(dragSelectTouchListener)
        binding.ocrResult.recyclerView.adapter = adapter
        binding.ocrResult.recyclerView.doOnPreDraw {
            adapter.selectAll()
            binding.ocrResult.submit.isEnabled = adapter.getSelection().isNotEmpty()
        }
        binding.ocrResult.root.setTransitionListener(null)
        binding.ocrResult.root.transitionToEnd()
        binding.ocrResult.toolbar.setNavigationOnClickListener {
            binding.ocrResult.root.setTransitionListener(object : MotionLayout.TransitionListener {
                override fun onTransitionStarted(p0: MotionLayout?, p1: Int, p2: Int) {

                }

                override fun onTransitionChange(p0: MotionLayout?, p1: Int, p2: Int, p3: Float) {
                }

                override fun onTransitionCompleted(p0: MotionLayout?, p1: Int) {
                    enterCameraState()
                }

                override fun onTransitionTrigger(
                    p0: MotionLayout?,
                    p1: Int,
                    p2: Boolean,
                    p3: Float
                ) {
                }

            })
            binding.ocrResult.root.transitionToStart()
        }
    }

    override fun onPicturetakeFail(exception: Exception) {

    }

    override fun onStatusChange(state: Int) {

    }

    private fun showErrorAlert(context: Context, exc: Exception) {
        val onClickListener = DialogInterface.OnClickListener { dialog, which ->
            activity?.finish()
        }

        AlertDialog.Builder(context)
            .setTitle(android.R.string.dialog_alert_title)
            .setMessage(R.string.camera_unavailable)
            .setPositiveButton(android.R.string.ok, onClickListener)
            .setCancelable(false)
            .create()
            .show()
    }
}

class PickerParagraphAdapter(private val ocrResult: OcrResult) :
    RecyclerView.Adapter<PickerParagraphAdapter.ParagraphViewHolder>() {

    private val mSelected: HashSet<Int> = HashSet()
    private var mClickListener: ItemClickListener? = null

    override fun onCreateViewHolder(parent: ViewGroup, viewType: Int): ParagraphViewHolder {
        return ParagraphViewHolder(
            LayoutInflater.from(parent.context).inflate(R.layout.paragraph_text_item, parent, false)
        )
    }

    override fun onBindViewHolder(holder: ParagraphViewHolder, position: Int) {
        val item = ocrResult.blocks[position]
        holder.paragraphText.setText(item.text)
        if (mSelected.contains(position)) {
            holder.itemView.setBackgroundResource(R.drawable.paragraph_selected)
        } else {
            holder.itemView.setBackgroundResource(R.drawable.paragraph_unselect)
        }
    }

    override fun getItemCount(): Int {
        return ocrResult.blocks.size
    }

    fun getSelection(): HashSet<Int> {
        return mSelected
    }

    // ----------------------
    // Click Listener
    // ----------------------
    fun setClickListener(itemClickListener: ItemClickListener) {
        mClickListener = itemClickListener
    }

    // ----------------------
    // Selection
    // ----------------------
    fun toggleSelection(pos: Int) {
        if (mSelected.contains(pos)) mSelected.remove(pos) else mSelected.add(pos)
        notifyItemChanged(pos)
    }

    fun select(pos: Int, selected: Boolean) {
        if (selected) mSelected.add(pos) else mSelected.remove(pos)
        notifyItemChanged(pos)
    }

    fun deselectAll() {
        // this is not beautiful...
        mSelected.clear()
        notifyDataSetChanged()
    }

    fun selectAll() {
        for (i in 0 until ocrResult.blocks.size) mSelected.add(i)
        notifyDataSetChanged()
    }

    fun selectRange(start: Int, end: Int, selected: Boolean) {
        for (i in start..end) {
            if (selected) mSelected.add(i) else mSelected.remove(i)
        }
        notifyItemRangeChanged(start, end - start + 1)
    }

    fun getItem(position: Int): Block {
        return ocrResult.blocks[position]
    }

    interface ItemClickListener {
        fun onItemClick(view: View, position: Int)
        fun onItemLongClick(view: View, position: Int): Boolean
    }

    inner class ParagraphViewHolder(itemView: View) : RecyclerView.ViewHolder(itemView),
        View.OnClickListener, View.OnLongClickListener {
        val paragraphText = itemView.findViewById<TextView>(R.id.paragraph_text)

        init {
            paragraphText.setOnClickListener(this)
            paragraphText.setOnLongClickListener(this)
        }

        override fun onClick(v: View) {
            mClickListener?.onItemClick(v, adapterPosition)
        }

        override fun onLongClick(v: View): Boolean {
            return mClickListener?.onItemLongClick(v, adapterPosition) ?: false
        }
    }
}

