package com.talpa.translate.ocr

import android.content.Context
import android.graphics.Bitmap
import android.util.Log
import androidx.lifecycle.LiveData
import androidx.lifecycle.ViewModel
import androidx.lifecycle.asLiveData
import com.photo.translation.R
import com.talpa.translate.base.common.FrameMetadata
import com.talpa.translate.factory.TranslatorFactory
import com.talpa.translate.network.HiTranslator
import com.talpa.translate.ocr.datasource.CompleteResult
import com.talpa.translate.ocr.datasource.TransParams
import com.talpa.translate.ocr.exception.NoContentException
import com.talpa.translate.render.Render
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.flow.*


/**
 * Photo Translation
 *
 * @author CY 2020-02-05
 */
class ImageTranslate constructor(private val context: Context) : ViewModel() {

    companion object {
        const val TAG = "ocr_translate"
    }

    private lateinit var mTranslator: Recognizer
    private val mRender: Render

    init {
        val analyzer: IAnalyzer = PhotoAnalyzer(context)
        mRender = TranslatorFactory.getRender(context, analyzer)
    }

    /**
     * 翻译
     */
    fun translate(
            bitmap: Bitmap,
            metadata: FrameMetadata,
            sourceLanguage: String,
            targetLanguage: String
    ): LiveData<Result<CompleteResult>> {
        val completeResult = CompleteResult()
        return flow {
            val ocrResult = mTranslator.doOcr(bitmap) //Step1:执行OCR
            completeResult.ocrResult = ocrResult
            emit(completeResult)
        }.flatMapConcat { completeResult ->
            flow {
                val ocrResult = completeResult.ocrResult
                        ?: throw NoContentException(context.getString(R.string.no_trans_for_ocr))
                Log.d("cjslog", "start translate:${System.currentTimeMillis()}")
                val texts = arrayListOf<String>()
                ocrResult.blocks.forEach { texts.add(it.text) }
                if (texts.isEmpty()) {
                    throw NoContentException(context.getString(R.string.no_trans_for_ocr))
                }
                val transResult = HiTranslator.getInstance(context)
                        .postTranslate(
                                from = sourceLanguage,
                                to = targetLanguage,
                                texts = texts
                        )  //Step2:OCR结果提交翻译
                val textsResponse = transResult.result?.texts
                        ?: throw NoContentException(context.getString(R.string.translate_fail))
                if (textsResponse.isEmpty()) throw NoContentException(context.getString(R.string.no_trans_for_ocr))
                completeResult.transResponse = transResult
                emit(completeResult)
            }.zip(flow {
                Log.d("cjslog", "start render:${System.currentTimeMillis()}")
                val renderSource = mRender.analyzeOcrResult(bitmap, mTranslator)
                emit(renderSource)
            }) { completeResult, renderSource ->
                val list = completeResult.transResponse?.result?.texts ?: throw NoContentException(
                        context.getString(R.string.translate_fail)
                )
                val result = mRender.renderOverlay(renderSource, list, mTranslator)
                result ?: throw NoContentException(context.getString(R.string.generate_bitmap_fail))
                completeResult.renderResult = result
                Result.success(completeResult)
            }
        }
                .onStart {
                    mTranslator = OcrDispatcher.dispatchOcrTranslator(context, sourceLanguage)
                    //mRender.initialize(bitmap)
                    mTranslator.setup(TransParams(metadata, sourceLanguage))
                }.catch {
                    Log.d("cjslog", "translate error", it)
                    //mRender.release()
                    emit(Result.failure(it))
                }.flowOn(Dispatchers.IO)
                .asLiveData()
    }

    fun changeLanguage(
            bitmap: Bitmap,
            sourceLanguage: String,
            targetLanguage: String
    ): LiveData<Result<CompleteResult>> {
        val completeResult = CompleteResult()
        return flow {
            val history = mTranslator.getHistory() ?: throw NoContentException(
                    context.getString(R.string.no_trans_for_ocr)
            )
            completeResult.ocrResult = history
            emit(completeResult)
        }.flatMapConcat { completeResult ->
            flow {
                val history = completeResult.ocrResult
                val texts = arrayListOf<String>()
                history?.blocks?.forEach { block ->
                    texts.add(block.text)
                }
                val transResult = HiTranslator.getInstance(context)
                        .postTranslate(from = sourceLanguage, to = targetLanguage, texts = texts)

                val textsResponse = transResult.result?.texts ?: throw NoContentException(
                        context.getString(R.string.translate_fail)
                )
                if (textsResponse.isEmpty()) throw NoContentException(
                        context.getString(R.string.no_trans_for_ocr)
                )
                completeResult.transResponse = transResult
                emit(completeResult)
            }.zip(
                    flow {
                        val renderSource = mRender.analyzeOcrResult(bitmap, mTranslator)
                        emit(renderSource)
                    }
            ) { completeResult, renderSource ->
                val list = completeResult.transResponse?.result?.texts ?: throw NoContentException(
                        context.getString(R.string.translate_fail)
                )
                val result =
                        mRender.renderOverlay(renderSource, list, mTranslator)
                                ?: throw NoContentException(
                                        context.getString(R.string.generate_bitmap_fail)
                                )
                completeResult.renderResult = result
                return@zip Result.success(completeResult)
            }
        }
                .catch { emit(Result.failure(it)) }.flowOn(Dispatchers.IO)
                .asLiveData()
    }


}