package com.talpa.translate.record

import android.content.ComponentName
import android.content.Context
import android.content.Intent
import android.content.ServiceConnection
import android.graphics.Bitmap
import android.graphics.Rect
import android.os.IBinder
import android.view.MotionEvent
import android.view.VelocityTracker
import androidx.annotation.RestrictTo
import com.talpa.PhotoTranslateController
import com.talpa.translate.activity.ScreenRecordActivity
import com.talpa.translate.base.common.FrameMetadata
import com.talpa.translate.camera.view.controls.Facing
import com.talpa.translate.camera.view.controls.PictureFormat
import com.talpa.translate.network.HiTranslator
import com.talpa.translate.ocr.OcrDispatcher
import com.talpa.translate.ocr.datasource.TransParams
import com.talpa.translate.ocr.result.OcrResult
import com.talpa.translate.service.RecordService
import kotlinx.coroutines.*
import java.util.*
import kotlin.math.abs
import kotlin.math.max
import kotlin.math.min
import kotlin.math.sqrt


/**
 * Create by chenjunsheng on 2020/8/5
 */
class CaptureManager(context: Context) : ServiceConnection, ICaptureController {

    companion object {

        const val RECT_GAP = 12
        const val MIN_DISTANCE = 15

        const val STATE_INITIALIZE = 0
        const val STATE_ASKING_PERMISSION = 1
        const val STATE_PERMISSION_GRANTED = 2
        const val STATE_SERVICE_PREPARED = 3
        const val STATE_CAPTURING_PIC = 4
        const val STATE_CAPTURED_PIC = 5
        const val STATE_RECOGNIZE_START = 6
        const val STATE_RECOGNIZING = 7
        const val STATE_TRANSLATING = 8
        const val STATE_RECOGNIZED_FINISH = 9
        const val STATE_TRANSLATE_FINISH = 10
        const val STATE_ACTION_CLEAR = 11

        private const val VELOCITY_TIME_MILLIS = 300

        /**
         * 最小速率
         */
        private const val MIN_VELOCITY = 80
    }

    private var mResultListener: PhotoTranslateController.TextRecognizeListener? = null
    private var mRecordService: IRecordService? = null
    private val mContext = context.applicationContext
    private var mCurrentState = STATE_INITIALIZE

    override fun setTextRecognizeListener(listener: PhotoTranslateController.TextRecognizeListener) {
        mResultListener = listener
    }

    @RestrictTo(RestrictTo.Scope.LIBRARY)
    override fun setCurrentState(state: Int) {
        mCurrentState = state
    }

    override fun captureFrameSync(context: Context): CaptureResult {
        if (mRecordService == null) {
            ScreenRecordActivity.start(context)
            //RequestPermissionActivity.start(context)
            return CaptureResult(START_CAPTURE_PERMISSION_REQUEST)
        } else {
            val result = try {
                mRecordService!!.captureFrameSync()
            } catch (e: Exception) {
                e.printStackTrace()
                null
            } ?: return CaptureResult(PROCESSOR_BITMAP_FAIL)
            fastAnalyzeArea(result)
            return result
        }
    }

    private fun isRectangleOverlap(current: Rect, other: Rect): Boolean {
        return current.left < other.right
                && other.left < current.right
                && current.top < other.bottom
                && other.top < current.bottom
    }

    private fun fastAnalyzeArea(result: CaptureResult) {
        if (result.bitmap == null) {
            return
        }
        val list = result.ocrResult?.blocks?.toMutableList() ?: return
        if (list.isEmpty()) return
        list.forEach { block ->
            val rect = block.rect
            //1.先向外扩张
            rect.inset(-RECT_GAP, -RECT_GAP)
            //2.检查是否越界
            rect.left = max(0, rect.left)
            rect.top = max(0, rect.top)
            rect.right = min(result.bitmap!!.width, rect.right)
            rect.bottom = min(result.bitmap!!.height, rect.bottom)
        }
        //3.检查是否重叠
        var i = 0
        while (i < list.size) {
            var j = i + 1
            while (j < list.size) {
                val rect = list[i].rect
                if (isRectangleOverlap(current = rect, other = rect)) {
                    rect.union(rect)
                    list.removeAt(j)
                    j--
                }
                j++
            }
            i++
        }
    }

    @RestrictTo(RestrictTo.Scope.LIBRARY_GROUP)
    override fun startRecordService(context: Context, resultCode: Int, data: Intent) {
        val serviceIntent = Intent(context, RecordService::class.java)
        serviceIntent.putExtra(RecordService.PROJECT_INTENT, data)
        serviceIntent.putExtra(RecordService.PROJECT_CODE, resultCode)
        context.applicationContext.bindService(serviceIntent, this, Context.BIND_AUTO_CREATE)
    }

    override fun destroyRecognizeService() {
        mRecordService?.destroyService()
        mRecordService = null
    }

    override fun onServiceDisconnected(name: ComponentName?) {
        //mRecordServiceMessenger = null
        mRecordService = null
    }

    override fun onServiceConnected(name: ComponentName?, service: IBinder) {
        mRecordService = IRecordService.Stub.asInterface(service)
            .also {
                try {
                    it.registerListener(mRemoteCallback)
                } catch (e: Exception) {
                    e.printStackTrace()
                }
            }
        //mRecordServiceMessenger = Messenger(service)
        //mRecordServiceMessenger?.send(getMessage(mClientMessenger, SEND_CLIENT_MESSENGER))
        setCurrentState(STATE_SERVICE_PREPARED)
    }

    private var mLastX = 0f
    private var mLastY = 0f
    private var mCaptureJob: Job? = null
    private var mRecognizeJob: Job? = null
    private var mLastResult: CaptureTransResult? = null
    private var velocityTracker: VelocityTracker? = null

    override fun delegateRecTouchEvent(e: MotionEvent) {
        if (velocityTracker == null) {
            velocityTracker = VelocityTracker.obtain()
        }
        velocityTracker!!.addMovement(e)

        when (e.action) {
            MotionEvent.ACTION_DOWN -> {
                mLastX = e.rawX
                mLastY = e.rawY
                if (mRecordService != null) {
                    mRecognizeJob?.cancel()
                    mCaptureJob?.cancel()
                    setCurrentState(STATE_SERVICE_PREPARED)
                }
            }
            MotionEvent.ACTION_MOVE -> {
                when (mCurrentState) {
                    STATE_INITIALIZE -> {
                        val dx = abs(e.rawX - mLastX)
                        val dy = abs(e.rawY - mLastY)
                        if (sqrt(dx * dx + dy * dy) > MIN_DISTANCE) {
                            captureFrameSync(mContext)
                            setCurrentState(STATE_ASKING_PERMISSION)
                        }
                    }
                    STATE_SERVICE_PREPARED -> {
                        val dx = abs(e.rawX - mLastX)
                        val dy = abs(e.rawY - mLastY)
                        if (sqrt(dx * dx + dy * dy) > MIN_DISTANCE) {
                            setCurrentState(STATE_CAPTURING_PIC)
                            mCaptureJob?.cancel()
                            mCaptureJob = CoroutineScope(Dispatchers.IO).launch {
                                val captureResult = captureFrameSync(mContext)
                                mLastResult = CaptureTransResult(captureResult)
                                    .also {
                                        it.transMap = hashMapOf()
                                    }

                                withContext(Dispatchers.Main) {
                                    //val intent = Intent(mContext, Test2Activity::class.java)
                                    //intent.putExtra("bitmap", mLastResult!!.bitmap!!)
                                    //mLastBitmap = mLastResult!!.bitmap
                                    //intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK)
                                    //mContext.startActivity(intent)
                                    setCurrentState(STATE_CAPTURED_PIC)
                                }
                            }
                        }
                    }
                    STATE_CAPTURED_PIC -> {
                        velocityTracker!!.computeCurrentVelocity(VELOCITY_TIME_MILLIS)
                        val xVelocity = velocityTracker!!.xVelocity
                        val yVelocity = velocityTracker!!.yVelocity
                        if (xVelocity < MIN_VELOCITY && yVelocity < MIN_VELOCITY) {
                            mLastResult?.captureResult?.let { result ->
                                var index = -1
                                result.ocrResult?.blocks?.let letInner@{ OcrResult ->
                                    OcrResult.forEachIndexed { i, block ->
                                        if (block.rect.contains(e.rawX.toInt(), e.rawY.toInt())) {
                                            index = i
                                            return@letInner
                                        }
                                    }
                                }
                                if (index != -1) {
                                    mLastResult!!.workingIndex = index
                                    setCurrentState(STATE_RECOGNIZE_START)
                                    val recognizeRect =
                                        result.ocrResult?.blocks?.get(index)?.rect ?: return@let
                                    mResultListener?.onAreaConfirmed(recognizeRect)
                                    mRecognizeJob?.cancel()
                                    mRecognizeJob = CoroutineScope(Dispatchers.Main).launch {
                                        setCurrentState(STATE_RECOGNIZING)
                                        val targetLanguage =
                                            mResultListener?.getTransTargetLanguage()
                                                ?: Locale.ENGLISH.language
                                        val sourceLanguage =
                                            mResultListener?.getTransSourceLanguage()
                                                ?: Locale.ENGLISH.language
                                        /*val ocrResult = withContext(Dispatchers.IO) {
                                            doTextRecognize(index, result, sourceLanguage)
                                        }*/
                                        val ocrResult = result.ocrResult?.blocks?.getOrNull(index)
                                        setCurrentState(STATE_RECOGNIZED_FINISH)
                                        if (ocrResult != null) {

                                            setCurrentState(STATE_TRANSLATING)

                                            val transResult = withContext(Dispatchers.IO) {
                                                HiTranslator.getInstance(mContext)
                                                    .postTranslateSingle(
                                                        sourceLanguage,
                                                        targetLanguage, arrayListOf(ocrResult.text)
                                                    )
                                            }

                                            mLastResult!!.workingIndex = -1
                                            mLastResult!!.transMap!![index] = transResult
                                            mResultListener?.onRecognized(index, mLastResult!!)
                                            if (mCurrentState == STATE_ACTION_CLEAR) {
                                                setCurrentState(STATE_SERVICE_PREPARED)
                                                mLastResult = null
                                            } else {
                                                setCurrentState(STATE_TRANSLATE_FINISH)
                                            }
                                        } else {
                                            if (mCurrentState == STATE_ACTION_CLEAR) {
                                                setCurrentState(STATE_SERVICE_PREPARED)
                                                mLastResult = null
                                            } else {
                                                setCurrentState(STATE_TRANSLATE_FINISH)
                                            }
                                            mResultListener?.onRecognizedFail(recognizeRect)
                                        }
                                    }
                                }
                            }
                        }
                    }
                    STATE_RECOGNIZED_FINISH, STATE_TRANSLATING, STATE_TRANSLATE_FINISH -> {
                        mLastResult?.let { transResult ->
                            velocityTracker!!.computeCurrentVelocity(VELOCITY_TIME_MILLIS)
                            val xVelocity = velocityTracker!!.xVelocity
                            val yVelocity = velocityTracker!!.yVelocity
                            if (xVelocity < MIN_VELOCITY && yVelocity < MIN_VELOCITY) {
                                transResult.captureResult.let { captureResult ->
                                    var index = -1
                                    captureResult.ocrResult?.blocks?.let letInner@{ blocks ->
                                        blocks.forEachIndexed { i, block ->
                                            if (block.rect.contains(
                                                    e.rawX.toInt(),
                                                    e.rawY.toInt()
                                                )
                                            ) {
                                                index = i
                                                return@letInner
                                            }
                                        }
                                    }



                                    if (index != -1 && transResult.workingIndex != index) {

                                        val recognizeRect =
                                            captureResult.ocrResult?.blocks?.get(index)?.rect
                                                ?: return@let

                                        if (transResult.transMap?.get(index) != null) {
                                            transResult.workingIndex = index
                                            mResultListener?.onAreaConfirmed(recognizeRect)
                                            mResultListener?.onRecognized(
                                                index,
                                                transResult
                                            )
                                            return@let
                                        }

                                        transResult.workingIndex = index
                                        mResultListener?.onAreaConfirmed(recognizeRect)
                                        //setCurrentState(STATE_TRANSLATING)
                                        val targetLanguage =
                                            mResultListener?.getTransTargetLanguage()
                                                ?: Locale.ENGLISH.language
                                        val sourceLanguage =
                                            mResultListener?.getTransSourceLanguage()
                                                ?: Locale.ENGLISH.language
                                        transResult.captureResult.ocrResult?.let { ocrR ->
                                            mRecognizeJob?.cancel()
                                            mRecognizeJob =
                                                CoroutineScope(Dispatchers.Main).launch {
                                                    setCurrentState(STATE_TRANSLATING)

                                                    val transR =
                                                        withContext(Dispatchers.IO) {
                                                            HiTranslator.getInstance(mContext)
                                                                .postTranslateSingle(
                                                                    sourceLanguage,
                                                                    targetLanguage,
                                                                    arrayListOf(ocrR.blocks[index].text)
                                                                )
                                                        }
                                                    setCurrentState(STATE_TRANSLATE_FINISH)
                                                    //transResult.workingIndex = -1
                                                    transResult.transMap!![index] = transR
                                                    mResultListener?.onRecognized(
                                                        index,
                                                        transResult
                                                    )
                                                }
                                        }
                                    }
                                }

                            }
                        }
                    }
                }
            }
            MotionEvent.ACTION_UP -> {
                if (mRecordService != null) {
                    when (mCurrentState) {
                        STATE_TRANSLATE_FINISH -> {
                            setCurrentState(STATE_SERVICE_PREPARED)
                            mLastResult = null
                        }
                        STATE_TRANSLATING, STATE_RECOGNIZING, STATE_CAPTURING_PIC -> {
                            setCurrentState(STATE_ACTION_CLEAR)
                        }
                    }

                } else {
                    setCurrentState(STATE_INITIALIZE)
                }

                velocityTracker?.recycle()
                velocityTracker = null
            }
        }
    }

    private val mRemoteCallback = object : IRecordListener.Stub() {

        override fun onRecordServiceReady() {
            mResultListener?.onRecognizeServiceReady()
        }

    }
}