package com.talpa.translate.record

import android.graphics.Bitmap
import android.graphics.PixelFormat
import android.hardware.display.DisplayManager
import android.hardware.display.VirtualDisplay
import android.media.ImageReader
import android.media.projection.MediaProjection
import android.os.Handler
import android.os.HandlerThread
import android.util.Log
import kotlinx.coroutines.suspendCancellableCoroutine
import java.lang.UnsupportedOperationException
import kotlin.coroutines.resume


/**
 * Create by chenjunsheng on 2020/8/4
 */
class ScreenRecorder(
    private var mDpi: Int,
    private var mWidth: Int,
    private var mHeight: Int
) : IScreenRecorder {

    companion object {
        private const val TAG = "ScreenRecorder"
    }

    private var mMediaProjection: MediaProjection? = null
    private var mHandlerThread: HandlerThread
    private var mHandler: Handler

    private var mVirtualDisplay: VirtualDisplay? = null
    private var mImageReader: ImageReader? = null

    init {
        mHandlerThread = HandlerThread(TAG).also { it.start() }
        mHandler = Handler(mHandlerThread.looper)
    }

    override fun destroy() {
        mHandlerThread.quitSafely()
    }

    override fun stopRecord() {
        try {
            mVirtualDisplay?.release()
            mMediaProjection?.stop()
            mMediaProjection = null
            mVirtualDisplay = null
        } catch (e: Exception) {
            e.printStackTrace()
        }

    }


    override suspend fun acquireLatestImage(projection: MediaProjection): Bitmap {
        return suspendCancellableCoroutine { continuation ->
            mImageReader = ImageReader.newInstance(mWidth, mHeight, PixelFormat.RGBA_8888, 1)
            mMediaProjection = projection
            mVirtualDisplay = projection.createVirtualDisplay(
                "$TAG-display",
                mWidth, mHeight, mDpi, DisplayManager.VIRTUAL_DISPLAY_FLAG_PUBLIC,
                mImageReader!!.surface, null, null
            )
            mImageReader!!.setOnImageAvailableListener({ reader ->
                continuation.resume(reader.use { closeableReader ->

                    val image = try {
                        closeableReader.acquireLatestImage()
                    } catch (e: IllegalStateException) {
                        e.printStackTrace()
                        null
                    } catch (e: UnsupportedOperationException) {
                        e.printStackTrace()
                        null
                    } ?: return@setOnImageAvailableListener

                    val width = image.width
                    val height = image.height
                    val planes = image.planes
                    val buffer = planes[0].buffer
                    //每个像素的间距
                    val pixelStride = planes[0].pixelStride
                    //总的间距
                    val rowStride = planes[0].rowStride
                    val rowPadding = rowStride - pixelStride * width

                    val bitmap = Bitmap.createBitmap(
                        width + rowPadding / pixelStride, height,
                        Bitmap.Config.ARGB_8888
                    ) //even though ARGB8888 will consume more memory,it has better compatibility on device.

                    bitmap.copyPixelsFromBuffer(buffer)
                    bitmap
                })
                stopRecord()

            }, mHandler)

        }
    }
}