package com.talpa.translate.ocr

import android.content.Context
import android.graphics.Bitmap
import android.util.Log
import androidx.lifecycle.LiveData
import androidx.lifecycle.ViewModel
import androidx.lifecycle.asLiveData
import com.google.mlkit.vision.label.ImageLabel
import com.photo.translation.R
import com.talpa.translate.base.common.FrameMetadata
import com.talpa.translate.factory.TranslatorFactory
import com.talpa.translate.imglabel.ImageLabelData
import com.talpa.translate.imglabel.ImageLabelWorker
import com.talpa.translate.network.HiTranslator
import com.talpa.translate.ocr.datasource.CompleteResult
import com.talpa.translate.ocr.datasource.TransParams
import com.talpa.translate.ocr.exception.NoContentException
import com.talpa.translate.render.Render
import com.talpa.translate.repository.box.collins.SenseNew
import com.talpa.translate.repository.net.dictionary.DictionaryRepo
import kotlinx.coroutines.*
import kotlinx.coroutines.flow.*
import java.util.*
import kotlin.coroutines.CoroutineContext


/**
 * Photo Translation
 *
 * @author CY 2020-02-05
 */
class ImageTranslate constructor(private val context: Context) : ViewModel() {

    companion object {
        const val TAG = "ocr_translate"
    }

    private lateinit var mTranslator: Recognizer
    private val mRender: Render

    init {
        val analyzer: IAnalyzer = PhotoAnalyzer(context)
        mRender = TranslatorFactory.getRender(context, analyzer)
    }

    /**
     * 物体识别
     */
    fun itemRecognize(
        bitmap: Bitmap,
        metadata: FrameMetadata,
        sourceLanguage: String,
        targetLanguage: String
    ): LiveData<Result<ImageLabelData>> {
        val dictRepo = DictionaryRepo()
        return flow {
            val itemRecognizer =
                ImageLabelWorker(context, bitmap, metadata, sourceLanguage, targetLanguage)
            val labels = itemRecognizer.doItemRecognize()
            emit(labels)
        }.flatMapConcat { labels ->

            flow {

                emit(
                    Result.success(
                        withContext(Dispatchers.IO) {
                            val sourceSenses = arrayListOf<SenseNew>()
                            val targetSenses = arrayListOf<SenseNew>()
                            labels.forEach {

                                val sourceList = async {
                                    dictRepo.search(
                                        it.text,
                                        Locale.ENGLISH.language,
                                        sourceLanguage
                                    )
                                }
                                val targetList = async {
                                    dictRepo.search(
                                        it.text,
                                        Locale.ENGLISH.language,
                                        targetLanguage
                                    )
                                }
                                sourceSenses.add(sourceList.await())
                                targetSenses.add(targetList.await())

                            }

                            ImageLabelData(bitmap, sourceSenses, targetSenses)
                        }
                    )

                )
            }
        }
            .catch {
                emit(Result.failure(it))
            }
            .flowOn(Dispatchers.IO)
            .asLiveData()
    }

    /**
     * 翻译
     */
    fun translate(
        bitmap: Bitmap,
        metadata: FrameMetadata,
        sourceLanguage: String,
        targetLanguage: String
    ): LiveData<Result<CompleteResult>> {
        val completeResult = CompleteResult(sourceLanguage, targetLanguage)
        return flow {
            val ocrResult = mTranslator.doOcr(bitmap) //Step1:执行OCR
            completeResult.ocrResult = ocrResult
            emit(completeResult)
        }.flatMapConcat { completeResult ->
            flow {
                val ocrResult = completeResult.ocrResult
                    ?: throw NoContentException(context.getString(R.string.no_content_identified))
                val texts = arrayListOf<String>()
                ocrResult.blocks.forEach { texts.add(it.text) }
                if (texts.isEmpty()) {
                    throw NoContentException(context.getString(R.string.no_content_identified))
                }
                val transResult = HiTranslator.getInstance(context)
                    .postTranslate(
                        from = sourceLanguage,
                        to = targetLanguage,
                        texts = texts
                    )  //Step2:OCR结果提交翻译
                val textsResponse = transResult?.result?.texts
                    ?: throw NoContentException(context.getString(R.string.translate_fail))
                if (textsResponse.isEmpty()) throw NoContentException(context.getString(R.string.no_trans_for_ocr))
                completeResult.transResponse = transResult
                emit(completeResult)
            }.zip(flow {
                val renderSource = mRender.analyzeOcrResult(bitmap, mTranslator)
                emit(renderSource)
            }) { completeResult, renderSource ->
                val list = completeResult.transResponse?.result?.texts ?: throw NoContentException(
                    context.getString(R.string.translate_fail)
                )
                val result = mRender.renderOverlay(renderSource, list, mTranslator)
                result ?: throw NoContentException(context.getString(R.string.generate_bitmap_fail))
                completeResult.renderResult = result
                Result.success(completeResult)
            }
        }
            .onStart {
                mTranslator = OcrDispatcher.dispatchOcrTranslator(context, sourceLanguage)
                //mRender.initialize(bitmap)
                mTranslator.setup(TransParams(metadata, sourceLanguage))
            }.catch {
                //mRender.release()
                emit(Result.failure(it))
            }.flowOn(Dispatchers.IO)
            .asLiveData()
    }

    fun changeTargetLanguage(
        bitmap: Bitmap,
        sourceLanguage: String,
        targetLanguage: String
    ): LiveData<Result<CompleteResult>> {
        val completeResult = CompleteResult(sourceLanguage, targetLanguage)
        return flow {
            val history = mTranslator.getHistory() ?: throw NoContentException(
                context.getString(R.string.no_trans_for_ocr)
            )
            completeResult.ocrResult = history
            emit(completeResult)
        }.flatMapConcat { completeResult ->
            flow {
                val history = completeResult.ocrResult
                val texts = arrayListOf<String>()
                history?.blocks?.forEach { block ->
                    texts.add(block.text)
                }
                val transResult = HiTranslator.getInstance(context)
                    .postTranslate(from = sourceLanguage, to = targetLanguage, texts = texts)

                val textsResponse = transResult?.result?.texts ?: throw NoContentException(
                    context.getString(R.string.translate_fail)
                )
                if (textsResponse.isEmpty()) throw NoContentException(
                    context.getString(R.string.no_trans_for_ocr)
                )
                completeResult.transResponse = transResult
                emit(completeResult)
            }.zip(
                flow {
                    val renderSource = mRender.analyzeOcrResult(bitmap, mTranslator)
                    emit(renderSource)
                }
            ) { completeResult, renderSource ->
                val list = completeResult.transResponse?.result?.texts ?: throw NoContentException(
                    context.getString(R.string.translate_fail)
                )
                val result =
                    mRender.renderOverlay(renderSource, list, mTranslator)
                        ?: throw NoContentException(
                            context.getString(R.string.generate_bitmap_fail)
                        )
                completeResult.renderResult = result
                return@zip Result.success(completeResult)
            }
        }
            .catch { emit(Result.failure(it)) }.flowOn(Dispatchers.IO)
            .asLiveData()
    }
}