package com.tapla.textspeech

import android.app.Application
import android.content.Context
import android.media.AudioManager
import android.media.AudioManager.AUDIOFOCUS_LOSS
import android.speech.tts.TextToSpeech
import android.speech.tts.TextToSpeech.*
import android.speech.tts.UtteranceProgressListener
import android.util.Log
import java.util.*

/**
 * 系统播报
 * @author cy 19-8-19
 */
object TextSpeech : TextToSpeech.OnInitListener, AudioManager.OnAudioFocusChangeListener {

    private var textToSpeech: TextToSpeech? = null

    private lateinit var mApplication: Application

    override fun onInit(status: Int) {
        if (BuildConfig.DEBUG) Log.d(TextSpeech::class.java.name, "status=$status")
        if (status != SUCCESS) {
            textToSpeech = null
        }

    }


    fun init(application: Application) {
        mApplication = application
        try {
            speech()
            textToSpeech?.setOnUtteranceProgressListener(OnUtteranceProgressListener(this))
        } catch (e: Exception) {
            e.printStackTrace()
        }


    }

    private fun speech() = textToSpeech ?: TextToSpeech(mApplication, this)
        .apply {
            textToSpeech = this
            setSpeechRate(0.8f)
        }


    /**
     * 播报
     */
    fun speak(text: CharSequence, locale: Locale, listener: SoundProgressListener? = null): Boolean {
        val textToSpeech: TextToSpeech
        soundProgressListener = listener
        try {
            textToSpeech = speech()
        } catch (e: Exception) {
            e.printStackTrace()
            return false
        }

        val code = textToSpeech.setLanguage(locale)
        return if (matchCode(code)) {
            val status =
                textToSpeech.speak(text, QUEUE_FLUSH, null, System.currentTimeMillis().toString())
            val isSuccess = status == SUCCESS

            if (isSuccess) {
                requestAudioFocus(application = mApplication)
            }
            isSuccess
        } else {
            TextSpeech.textToSpeech = null
            false
        }
    }

    /**
     * 停止播报
     */
    fun stopSpeak() {
        soundProgressListener = null
        val textToSpeech = speech()
        if (textToSpeech.isSpeaking) {
            speech().stop()
        }

    }

    /**
     * 正在播报
     */
    fun isSpeaking() = speech().isSpeaking

    /**
     * 语言是否有效
     */
    fun isLanguageAvailable(locale: Locale): Boolean {
        val textToSpeech = speech()
        val code = textToSpeech.isLanguageAvailable(locale)
        return matchCode(code)
    }

    private fun matchCode(code: Int): Boolean {
        return code == LANG_AVAILABLE || code == LANG_COUNTRY_VAR_AVAILABLE || code == LANG_COUNTRY_AVAILABLE
    }

    private fun requestAudioFocus(application: Application) {

//        val focusRequest = AudioFocusRequest.Builder(AudioManager.AUDIOFOCUS_GAIN_TRANSIENT).build()
        val audioManager = application.getSystemService(Context.AUDIO_SERVICE) as AudioManager
//        audioManager.requestAudioFocus(focusRequest)
        audioManager.requestAudioFocus(
            this,
            AudioManager.STREAM_MUSIC,
            AudioManager.AUDIOFOCUS_GAIN_TRANSIENT
        )
    }

    override fun onAudioFocusChange(focusChange: Int) {
        if (focusChange == AUDIOFOCUS_LOSS) {
            stopSpeak()
        }
    }

    private var soundProgressListener: SoundProgressListener? = null

    interface SoundProgressListener {
        fun onDone()
        fun onError()
        fun onStart()
        fun onStop()
    }

    private class OnUtteranceProgressListener(val textSpeech: TextSpeech) :
        UtteranceProgressListener() {
        override fun onDone(utteranceId: String?) {
            val audioManager = mApplication.getSystemService(Context.AUDIO_SERVICE) as AudioManager
            audioManager.abandonAudioFocus(textSpeech)
            soundProgressListener?.onDone()
        }

        override fun onError(utteranceId: String?) {
            soundProgressListener?.onError()
        }

        override fun onStop(utteranceId: String?, interrupted: Boolean) {
            super.onStop(utteranceId, interrupted)
            soundProgressListener?.onStop()
        }

        override fun onStart(utteranceId: String?) {
            soundProgressListener?.onStart()
        }

    }
}