package com.talpa.datareport

import android.content.Context
import android.content.pm.PackageManager
import android.os.Build
import android.util.Log
import androidx.annotation.Keep
import androidx.work.CoroutineWorker
import androidx.work.WorkerParameters
import java.lang.Exception
import java.util.*
import kotlin.math.abs
import kotlin.math.max

/**
 * Create by chenjunsheng on 2021/7/23
 */
@Keep
class DataReportRequest(appContext: Context, workerParams: WorkerParameters) :
    CoroutineWorker(appContext, workerParams) {

    /**
     * get App versionName
     * @param context
     * @return
     */
    fun getVersionName(context: Context): String {
        val packageManager = context.getPackageManager()
        val versionName = try {
            val packageInfo = packageManager.getPackageInfo(context.getPackageName(), 0)
            packageInfo.versionName;
        } catch (e: PackageManager.NameNotFoundException) {
            e.printStackTrace()
            "null"
        }
        return versionName
    }

    override suspend fun doWork(): Result {
        val reportTask = ReportDatabase.getInstance(applicationContext)
            .reportDao()
            .queryUploadTrans()

        if (reportTask.isNullOrEmpty()) return Result.success()

        val requestBody = UploadBody(
            appName = applicationContext.packageName,
            country = Locale.getDefault().country,
            osVersion = Build.VERSION.RELEASE,
            versionName = getVersionName(applicationContext),
            data = reportTask.map {
                ReportDetail(
                    sourceLanguage = it.from,
                    targetLanguage = it.to,
                    strLength = it.strLength,
                    translationTime = it.startTime,
                    engine = it.source,
                    timeConsuming = max(abs(it.startTime - it.finishTime), 1L),
                    status = it.code
                )
            }
        )
        val response = try {
            ReportNetwork.mReportApi?.report(requestBody)
        } catch (e: Exception) {
            e.printStackTrace()
            null
        }
        response?.let {
            ReportDatabase.getInstance(applicationContext)
                .reportDao()
                .deleteFinishTrans(reportTask)
        }
        return Result.success()
    }
}