package com.talpa.translate

import androidx.annotation.Keep
import com.talpa.tengine.lang.LANG
import org.json.JSONArray
import org.json.JSONObject
import java.io.InputStream
import java.io.OutputStream
import java.net.HttpURLConnection
import java.net.URL
import java.security.MessageDigest

/**
 * Hi Translator Sync
 *
 * @author CY 20-07-16
 */
@Keep
class MultiTextTranslator private constructor(
    private val sourceLanguage: String?,
    private val targetLanguage: String,
    private val texts: ArrayList<String>,
    private val appKey: String,
    private val secret: String
) {


    class Builder {//private val context: Context? = null

        private var sourceLanguage: String? = null
        private var targetLanguage: String? = null
        private var texts: ArrayList<String>? = null
        private var appKey: String? = null
        private var secret: String? = null

        fun setSourceLanguage(sourceLanguage: String?) = apply {
            this.sourceLanguage = sourceLanguage
        }

        fun setTargetLanguage(targetLanguage: String) = apply {
            this.targetLanguage = targetLanguage
        }

        fun setTexts(texts: ArrayList<String>) = apply {
            this.texts = texts
        }

        fun setAppKey(appKey: String) = apply {
            this.appKey = appKey
        }

        fun setSecret(secret: String) = apply {
            this.secret = secret
        }

        fun build(): MultiTextTranslator {

            // apply(builder)
            targetLanguage ?: throw NullPointerException("targetLanguage can`t null.")
            texts ?: throw NullPointerException("text can`t null.")
            appKey ?: throw NullPointerException("key can`t null.")
            secret ?: throw NullPointerException("secret can`t null.")

            return MultiTextTranslator(
                sourceLanguage = sourceLanguage,
                targetLanguage = targetLanguage!!,
                texts = texts!!,
                appKey = appKey!!,
                secret = secret!!
            )
        }
    }

    data class Result constructor(
        val sourceLanguage: String?,
        val targetLanguage: String,
        val texts: ArrayList<String>,
        val translations: List<String> = emptyList(),
        val valueCode: Int = 0,
        val valueMessage: String? = null,
        val isCache: Boolean = false,
        val isSuccess: Boolean = valueCode == 1000
    )

    /**
     * Translate
     */
    fun execute(): Result {

        /*val translation = readTranslation(
            sourceLanguage = sourceLanguage,
            targetLanguage = targetLanguage,
            text = text
        )*/

        /* if (!TextUtils.isEmpty(translation)) {
             return Result(
                 sourceLanguage,
                 targetLanguage,
                 text,
                 translation,
                 valueCode = 1000,
                 isCache = true
             )
         }*/

        val result = connect()

        /*if (!TextUtils.isEmpty(result.translation)) {
            storeTranslation(
                sourceLanguage = sourceLanguage,
                targetLanguage = targetLanguage,
                text = text,
                translation = result.translation!!
            )
        }*/

        return result
    }

    private fun connect(): Result {
        val url = URL(BASE_URL)
        val connection = url.openConnection() as HttpURLConnection
        connection.requestMethod =
            REQUEST_METHOD
        connection.setRequestProperty(
            "content-type",
            CONTENT_TYPE
        )
        connection.connectTimeout =
            CONNECT_TIME_OUT
        connection.readTimeout =
            READ_TIME_OUT
        connection.defaultUseCaches = true
        connection.doOutput = true
        val outputStream = connection.outputStream
        writeOutputStream(outputStream)
        outputStream.close()
        val inputStream = connection.inputStream
        val result = readInputStream(inputStream)
        inputStream.close()
        connection.disconnect()

        return result
    }

    private fun writeOutputStream(outputStream: OutputStream) {

//        {
//            "app_key": "10025"
//            "to":"en",
//            "text":"你好啊",
//            "from":"",
//            "timestamp": 1231234213,
//            "nonce": "acbf"
//            "sig":"feafefaefaefef"
//        }

        val timestamp = System.currentTimeMillis() / 1000
        val nonce = generateNonce()
        val sig = generateSign(
            appKey = appKey,
            targetLanguage = targetLanguage,
            timestamp = timestamp,
            secret = secret,
            nonce = nonce
        )

        val textsArray = JSONArray(texts)

        val jsonObj = JSONObject()
        jsonObj.put("app_key", appKey)
        jsonObj.put("to", targetLanguage)
        jsonObj.put("texts", textsArray)
        jsonObj.put("timestamp", timestamp)
        jsonObj.put("nonce", nonce)
        jsonObj.put("sig", sig)
        if (sourceLanguage != null) {
            jsonObj.put("from", sourceLanguage)
        }

        val json = jsonObj.toString()
        outputStream.write(json.toByteArray())
        outputStream.flush()
    }

    private fun generateNonce(): String {

        val charArray = arrayOf(
            'a',
            'b',
            'c',
            'd',
            'e',
            'f',
            'g',
            'h',
            'i',
            'j',
            'k',
            'l',
            'm',
            'n',
            'o',
            'p',
            'q',
            'r',
            's',
            't',
            'u',
            'v',
            'w',
            'x',
            'y',
            'z'
        )
        val index1 = (charArray.indices).random()
        val index2 = (charArray.indices).random()
        val index3 = (charArray.indices).random()
        val index4 = (charArray.indices).random()

        return "${charArray[index1]}${charArray[index2]}${charArray[index3]}${charArray[index4]}"
    }

    private fun generateSign(
        appKey: String,
        targetLanguage: String,
        timestamp: Long,
        secret: String,
        nonce: String
    ): String {
        //"%s&%s&%s&%s&%s", appId, to, timestamp, secret, nonce
        val value = "$appKey&$targetLanguage&$timestamp&$secret&$nonce"
        return value.md5()
    }

    private fun String.md5(): String {

        val digest = MessageDigest.getInstance("MD5")
        val data = digest.digest(this.toByteArray())

        val charArray = Hex.encodeHex(data)

        return String(charArray)
        //DatatypeConverter.printHexBinary( MessageDigest.getInstance("MD5").digest("a".getBytes("UTF-8")))
        //val md5 = ByteString.of(*this.toByteArray()).md5()
        //val utf8 = md5.utf8()
        //val base64 = md5.base64()
        //val hex = md5.hex()
        //val string = md5.string(UTF_8)
        //val sha1 = md5.sha1()
        //val sha256 = md5.sha256()

        //println("MD5==utf8=$utf8  base64=$base64  hex=$hex  md5String=$string  sha1=$sha1   sha256=$sha256")
    }


    private fun readInputStream(inputStream: InputStream): Result {

        val bytes = inputStream.readBytes()

        val content = String(bytes)

        return parserJson(content)
    }

    /**
     * Parser Content
     *
     * {"code":1000,"message":"success","result":{"text":"你好吗"}}
     *
     */
    private fun parserJson(json: String): Result {

        val jsonObject = JSONObject(json)

        val keyCode = "code"
        val keyMessage = "message"
        val keyResult = "result"
        val keyText = "texts"

        var valueCode: Int = 0
        var valueMessage: String? = null
        var valueTexts: JSONArray? = null

        if (jsonObject.has(keyCode)) {
            valueCode = jsonObject.getInt(keyCode)
        }
        if (jsonObject.has(keyMessage)) {
            valueMessage = jsonObject.getString(keyMessage)
        }
        if (jsonObject.has(keyResult)) {
            val valueResult = jsonObject.getJSONObject(keyResult)
            valueTexts = valueResult.getJSONArray(keyText)
        }

        val translations = arrayListOf<String>()
        val length = valueTexts?.length() ?: 0
        for (index in 0 until length) {
            val value = valueTexts?.getString(index) ?: continue
            translations.add(value)
        }

        return Result(
            sourceLanguage = sourceLanguage,
            targetLanguage = targetLanguage,
            texts = texts,
            translations = translations,
            valueCode = valueCode,
            valueMessage = valueMessage
        )
    }


    companion object {
        //        http://tapi.translasion.com/ 测试
//        http://api.translasion.com/ 正式
//        private const val BASE_URL = "https://tapi.translasion.com/v2/translate"
//        http://tapi.translasion.com/v2/translate
        private const val BASE_URL = "https://api.translasion.com/v2/translate"
        private const val CONNECT_TIME_OUT = 6000
        private const val READ_TIME_OUT = 8000
        private const val REQUEST_METHOD = "POST"
        private const val CONTENT_TYPE = "application/json"

        val supportLanguageMap = hashMapOf<String, String>().apply {
            this[LANG.AUTO] = "auto"
            this[LANG.AF] = "af"//南非荷兰语
            this[LANG.SQ] = "sq"//阿尔巴尼亚语
            this[LANG.AM] = "am"//阿姆哈拉语
            this[LANG.AR] = "ar"//阿拉伯语
            this[LANG.HY] = "hy"//亚美尼亚语
            this[LANG.AZ] = "az"//阿塞拜疆语
            this[LANG.EU] = "eu"//巴斯克语
            this[LANG.BE] = "be"//白俄罗斯语
            this[LANG.BN] = "bn"//孟加拉语
            this[LANG.BS] = "bs"//波斯尼亚语
            this[LANG.BG] = "bg"//保加利亚语
            this[LANG.CA] = "ca"//加泰罗尼亚语
            this[LANG.CEB] = "ceb"//宿务语
            this[LANG.ZH_CN] = "zh"//中文
            this[LANG.ZH_TW] = "zh-TW"//中文
            this[LANG.CO] = "co"//科西嘉语
            this[LANG.HR] = "hr"//克罗地亚语
            this[LANG.CS] = "cs"//捷克语
            this[LANG.DA] = "da"//丹麦语
            this[LANG.NL] = "nl"//荷兰语
            this[LANG.EN] = "en"//英语
            this[LANG.EO] = "eo"//世界语
            this[LANG.ET] = "et"//爱沙尼亚语
            this[LANG.FI] = "fi"//芬兰语
            this[LANG.FR] = "fr"//法语
            this[LANG.FY] = "fy"//西弗里西亚语
            this[LANG.GL] = "gl"//加利西亚语
            this[LANG.KA] = "ka"//格鲁吉亚语
            this[LANG.DE] = "de"//德语
            this[LANG.EL] = "el"//希腊语
            this[LANG.GU] = "gu"//古吉拉特语
            this[LANG.HT] = "ht"//海地克里奥尔语
            this[LANG.HA] = "ha"//豪萨语
            this[LANG.HAW] = "haw"//夏威夷语
            this[LANG.HE] = "he"//希伯来语
            this[LANG.HI] = "hi"//印地语
            this[LANG.HMN] = "hmn"//苗语
            this[LANG.HU] = "hu"//匈牙利语
            this[LANG.IS] = "is"//冰岛语
            this[LANG.IG] = "ig"//伊博语
            this[LANG.ID] = "id"//印度尼西亚语
            this[LANG.GA] = "ga"//爱尔兰语
            this[LANG.IT] = "it"//意大利语
            this[LANG.JA] = "ja"//日语
            this[LANG.JV] = "jv"//爪哇语
            this[LANG.KN] = "kn"//卡纳达语
            this[LANG.KK] = "kk"//哈萨克语
            this[LANG.KM] = "km"//高棉语
            this[LANG.RW] = "rw"//卢旺达语
            this[LANG.KO] = "ko"//韩语
            this[LANG.KU] = "ku"//库尔德语
            this[LANG.KY] = "ky"//柯尔克孜语
            this[LANG.LO] = "lo"//老挝语
            this[LANG.LA] = "la"//拉丁语
            this[LANG.LV] = "lv"//拉脱维亚语
            this[LANG.LT] = "lt"//立陶宛语
            this[LANG.LB] = "lb"//卢森堡语
            this[LANG.MK] = "mk"//马其顿语
            this[LANG.MG] = "mg"//马拉加斯语
            this[LANG.MS] = "ms"//马来语
            this[LANG.ML] = "ml"//马拉雅拉姆语
            this[LANG.MT] = "mt"//马耳他语
            this[LANG.MI] = "mi"//毛利语
            this[LANG.MR] = "mr"//马拉地语
            this[LANG.MN] = "mn"//蒙古语
            this[LANG.MY] = "my"//缅甸语
            this[LANG.NE] = "ne"//尼泊尔语
            this[LANG.NO] = "no"//挪威语
            this[LANG.NY] = "ny"//齐切瓦语
            this[LANG.OR] = "or"//奥里亚语
            this[LANG.PS] = "ps"//普什图语
            this[LANG.FA] = "fa"//波斯语
            this[LANG.PL] = "pl"//波兰语
            this[LANG.PT] = "pt"//葡萄牙语
            this[LANG.PA] = "pa"//旁遮普语
            this[LANG.RO] = "ro"//罗马尼亚语
            this[LANG.RU] = "ru"//俄语
            this[LANG.SM] = "sm"//萨摩亚语
            this[LANG.GD] = "gd"//苏格兰盖尔语
            this[LANG.SR] = "sr"//塞尔维亚语
            this[LANG.ST] = "st"//南索托语
            this[LANG.SN] = "sn"//绍纳语
            this[LANG.SD] = "sd"//信德语
            this[LANG.SI] = "si"//僧伽罗语
            this[LANG.SK] = "sk"//斯洛伐克语
            this[LANG.SL] = "sl"//斯洛文尼亚语
            this[LANG.SO] = "so"//索马里语
            this[LANG.ES] = "es"//西班牙语
            this[LANG.SU] = "su"//巽他语
            this[LANG.SW] = "sw"//斯瓦希里语
            this[LANG.SV] = "sv"//瑞典语
            this[LANG.TL] = "tl"//他加禄语
            this[LANG.TG] = "tg"//塔吉克语
            this[LANG.TA] = "ta"//泰米尔语
            this[LANG.TT] = "tt"//鞑靼语
            this[LANG.TE] = "te"//泰卢固语
            this[LANG.TH] = "th"//泰语
            this[LANG.TR] = "tr"//土耳其语
            this[LANG.TK] = "tk"//土库曼语
            this[LANG.UK] = "uk"//乌克兰语
            this[LANG.UR] = "ur"//乌尔都语
            this[LANG.UG] = "ug"//维吾尔语
            this[LANG.UZ] = "uz"//乌兹别克语
            this[LANG.VI] = "vi"//越南语
            this[LANG.CY] = "cy"//威尔士语
            this[LANG.XH] = "xh"//科萨语
            this[LANG.YI] = "yi"//意第绪语
            this[LANG.YO] = "yo"//约鲁巴语
            this[LANG.ZU] = "zu"//祖玛语
        }
    }
}