package com.tencent.imsdk.common;

import android.os.Build;
import android.os.SystemClock;
import android.text.TextUtils;
import android.util.Log;
import com.tencent.imsdk.BaseConstants;
import java.io.BufferedInputStream;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.Authenticator;
import java.net.HttpURLConnection;
import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.net.PasswordAuthentication;
import java.net.Proxy;
import java.net.Socket;
import java.net.URL;
import java.net.UnknownHostException;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.Executor;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import javax.net.ssl.HostnameVerifier;
import javax.net.ssl.HttpsURLConnection;
import javax.net.ssl.SNIHostName;
import javax.net.ssl.SSLParameters;
import javax.net.ssl.SSLSession;
import javax.net.ssl.SSLSocket;
import javax.net.ssl.SSLSocketFactory;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

public class HttpClient {
    private static final String TAG = HttpClient.class.getSimpleName();

    public static final int HTTP_ACTION_REQUEST = 0;
    public static final int HTTP_ACTION_RESPONSE = 1;

    // 获取当前设备的CPU数
    private static final int CPU_COUNT = Runtime.getRuntime().availableProcessors();
    // 核心池大小设为CPU数加1
    private static final int CORE_POOL_SIZE = CPU_COUNT + 1;
    // 设置线程池的最大大小
    private static final int MAX_POOL_SIZE = 2 * CPU_COUNT + 1;
    // 存活时间
    private static final long KEEP_ALIVE = 5L;
    // Http 代理
    private static final int PROXY_TYPE_HTTP = 1;
    // Socks5 代理
    private static final int PROXY_TYPE_SOCKS5 = 2;
    // 切换 https 为 http 的配置
    private static String mRollbackHttps2Http = "";
    // 是否切换 https 为 http
    private static boolean mNeedRollbackHttps2Http = false;

    // 创建线程池对象
    private static final Executor mThreadPoolExecutor = new ThreadPoolExecutor(
        CORE_POOL_SIZE, MAX_POOL_SIZE, KEEP_ALIVE, TimeUnit.SECONDS, new LinkedBlockingQueue<Runnable>());

    private static native void nativeProgressCallback(int action, int currentSize, int totalSize, long userData);

    private static native void nativeStatisticsCallback(boolean isDnsSuccess, int dnsCostTime, boolean isConnectSuccess,
        int connectCostTime, int requestResponseCostTime, String remoteIp, int remotePort, int httpStatus,
        long userData);

    private static native void nativeResponseCallback(
        int code, String[] headerKeys, String[] headerValues, byte[] response, long userData);

    private static native boolean nativeSetHostname(Socket sslSocket, String host);

    public interface HttpRequestListener {
        void onProgress(final int action, final int currentSize, final int totalSize);

        void onStatistics(final boolean isDnsSuccess, final int dnsCostTime, final boolean isConnectSuccess,
            final int connectCostTime, final int requestResponseCostTime, final String remoteIp, final int remotePort,
            final int httpStatus);

        void onCompleted(final int code, final Map<String, String> headers, final byte[] response);
    }

    // 代理的密码验证类
    static class BasicAuthenticator extends Authenticator {
        private String userName = "";
        private String password = "";

        public BasicAuthenticator(String userName, String password) {
            this.userName = userName;
            this.password = password;
        }

        @Override
        protected PasswordAuthentication getPasswordAuthentication() {
            return new PasswordAuthentication(userName, password.toCharArray());
        }
    }

    private static void httpRequest(final String method, final String url, final boolean useOriginIp,
        final Map<String, String> headers, final byte[] content, final String uploadFile, final String downloadFile,
        final int proxyType, final String proxyHost, final int proxyPort, final String proxyUserName,
        final String proxyPassword, final int connectTimeout, final int recvTimeout, final String rollbackHttps2Http,
        final boolean isStatisticsEnabled, final HttpRequestListener listener) {
        // 创建一个新的请求任务
        Runnable requestRunnable = new Runnable() {
            @Override
            public void run() {
                HttpURLConnection conn = null;
                InputStream responseInputStream = null;
                int code = HttpURLConnection.HTTP_OK;
                Map<String, String> rspHeaders = null;
                byte[] response = null;
                String requestUrl = url;

                boolean isDnsSuccess = false;
                int dnsCostTime = 0;
                boolean isConnectSuccess = false;
                int connectCostTime = 0;
                int requestResponseCostTime = 0;
                String remoteIp = "";
                int remotePort = 0;
                int httpStatus = 0;

                if (url.startsWith("https")) {
                    if (!mRollbackHttps2Http.equals(rollbackHttps2Http)) {
                        mRollbackHttps2Http = rollbackHttps2Http;
                        mNeedRollbackHttps2Http = needRollbackHttps2Http(rollbackHttps2Http);
                    }

                    if (mNeedRollbackHttps2Http) {
                        requestUrl = url.replaceFirst("https", "http");
                    }
                }

                try {
                    URL url = null;
                    if (!isStatisticsEnabled) {
                        url = new URL(requestUrl);
                    } else {
                        URL urlAddress = new URL(requestUrl);

                        final long dns_begin_time = SystemClock.uptimeMillis();
                        InetAddress ipAddress = InetAddress.getByName(urlAddress.getHost());
                        isDnsSuccess = true;
                        dnsCostTime = (int) (SystemClock.uptimeMillis() - dns_begin_time);
                        remoteIp = ipAddress.getHostAddress();
                        remotePort = urlAddress.getPort();
                        if (-1 == remotePort) {
                            remotePort = urlAddress.getDefaultPort();
                        }

                        // 此处已完成 DNS 解析，创建 IP 形式的 url，注意要配置 IP 直连的参数，即调用 setSSLSocketFactory
                        url = new URL(urlAddress.getProtocol(), ipAddress.getHostAddress(), urlAddress.getPort(),
                            urlAddress.getFile());
                    }

                    if (PROXY_TYPE_HTTP == proxyType && !proxyHost.isEmpty() && proxyPort != 0) {
                        Proxy proxy = new Proxy(Proxy.Type.HTTP, new InetSocketAddress(proxyHost, proxyPort));
                        conn = (HttpURLConnection) (url.openConnection(proxy));
                        if (!proxyUserName.isEmpty() && !proxyPassword.isEmpty()) {
                            Authenticator.setDefault(new BasicAuthenticator(proxyUserName, proxyPassword));
                        }
                    } else if (PROXY_TYPE_SOCKS5 == proxyType && !proxyHost.isEmpty() && proxyPort != 0) {
                        Proxy proxy = new Proxy(Proxy.Type.SOCKS, new InetSocketAddress(proxyHost, proxyPort));
                        conn = (HttpURLConnection) (url.openConnection(proxy));
                        if (!proxyUserName.isEmpty() && !proxyPassword.isEmpty()) {
                            Authenticator.setDefault(new BasicAuthenticator(proxyUserName, proxyPassword));
                        }
                    } else {
                        conn = (HttpURLConnection) (url.openConnection());
                    }

                    conn.setRequestMethod(method);
                    conn.setConnectTimeout(connectTimeout);
                    conn.setReadTimeout(recvTimeout);
                    conn.setUseCaches(false);
                    conn.setDoInput(true);

                    if (headers != null) {
                        for (Map.Entry<String, String> entry : headers.entrySet()) {
                            conn.setRequestProperty(entry.getKey(), entry.getValue());
                        }
                    }

                    if ((isStatisticsEnabled || useOriginIp) && conn instanceof HttpsURLConnection) {
                        final HttpsURLConnection httpsConn = (HttpsURLConnection) conn;

                        httpsConn.setSSLSocketFactory(new SSLSocketFactory() {
                            @Override
                            public Socket createSocket(Socket s, String host, int port, boolean autoClose)
                                throws IOException {
                                String realHost = httpsConn.getRequestProperty("Host");
                                SSLSocket sslSocket =
                                    (SSLSocket) HttpsURLConnection.getDefaultSSLSocketFactory().createSocket(
                                        s, realHost, port, autoClose);
                                sslSocket.setEnabledProtocols(sslSocket.getSupportedProtocols());

                                if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N) {
                                    SSLParameters sslParam = sslSocket.getSSLParameters();
                                    sslParam.setServerNames(Arrays.asList(new SNIHostName(realHost)));

                                    sslSocket.setSSLParameters(sslParam);
                                } else if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN_MR1) {
                                    if (nativeSetHostname(sslSocket, realHost) == false) {
                                        IMLog.e(TAG, "setHostname failed");
                                    }
                                } else {
                                    try {
                                        java.lang.reflect.Method setHostnameMethod =
                                            sslSocket.getClass().getMethod("setHostname", String.class);
                                        setHostnameMethod.invoke(sslSocket, realHost);
                                    } catch (Exception e) {
                                        IMLog.e(TAG, "SNI not useable" + e.getLocalizedMessage());
                                    }
                                }

                                return sslSocket;
                            }

                            @Override
                            public Socket createSocket(String host, int port) throws IOException, UnknownHostException {
                                return HttpsURLConnection.getDefaultSSLSocketFactory().createSocket(host, port);
                            }

                            @Override
                            public Socket createSocket(String host, int port, InetAddress localHost, int localPort)
                                throws IOException, UnknownHostException {
                                return HttpsURLConnection.getDefaultSSLSocketFactory().createSocket(
                                    host, port, localHost, localPort);
                            }

                            @Override
                            public Socket createSocket(InetAddress host, int port) throws IOException {
                                return HttpsURLConnection.getDefaultSSLSocketFactory().createSocket(host, port);
                            }

                            @Override
                            public Socket createSocket(InetAddress address, int port, InetAddress localAddress,
                                int localPort) throws IOException {
                                return HttpsURLConnection.getDefaultSSLSocketFactory().createSocket(
                                    address, port, localAddress, localPort);
                            }

                            @Override
                            public String[] getDefaultCipherSuites() {
                                return HttpsURLConnection.getDefaultSSLSocketFactory().getDefaultCipherSuites();
                            }

                            @Override
                            public String[] getSupportedCipherSuites() {
                                return HttpsURLConnection.getDefaultSSLSocketFactory().getSupportedCipherSuites();
                            }
                        });

                        httpsConn.setHostnameVerifier(new HostnameVerifier() {
                            @Override
                            public boolean verify(String hostname, SSLSession session) {
                                String host = httpsConn.getRequestProperty("Host");
                                return HttpsURLConnection.getDefaultHostnameVerifier().verify(host, session);
                            }
                        });
                    }

                    if (isStatisticsEnabled) {
                        // 为统计连接耗时，此处显示触发 connect 建立连接，注意其他连接后才能调用的接口，会隐式触发 connect，
                        // 例如 getOutputStream/getInputStream/getResponseCode/getHeaderFields/getContentLength
                        final long connect_begin_time = SystemClock.uptimeMillis();
                        conn.connect();
                        connectCostTime = (int) (SystemClock.uptimeMillis() - connect_begin_time);
                        isConnectSuccess = true;
                    }

                    // 写入请求数据
                    final long request_response_begin_time = SystemClock.uptimeMillis();
                    boolean hasContent = content != null && content.length > 0;
                    boolean hasUploadFile = uploadFile != null && uploadFile.length() != 0;
                    boolean hasOutput = hasContent || hasUploadFile;
                    if (!method.equalsIgnoreCase("GET") && hasOutput) {
                        conn.setDoOutput(true);
                        InputStream inputStream = null;
                        if (hasUploadFile) {
                            inputStream = new FileInputStream(uploadFile);
                        } else {
                            inputStream = new ByteArrayInputStream(content);
                        }
                        if (inputStream != null) {
                            int totalSize = inputStream.available(); // TODO alderzhang：此处大于2G的文件会出现问题
                            conn.setFixedLengthStreamingMode(totalSize);
                            int bytesTransferred = 0;
                            byte[] buf = new byte[4096];
                            while (true) {
                                int len = inputStream.read(buf);
                                if (len < 0) {
                                    break;
                                }
                                bytesTransferred += len;
                                conn.getOutputStream().write(buf, 0, len);
                                if (listener != null) {
                                    listener.onProgress(HTTP_ACTION_REQUEST, bytesTransferred, totalSize);
                                }
                            }
                            inputStream.close();
                        }
                    }

                    code = conn.getResponseCode();
                    httpStatus = code;

                    int headerCount = conn.getHeaderFields().size();
                    if (headerCount > 0) {
                        rspHeaders = new HashMap<String, String>();
                        for (int i = 0; i < headerCount; ++i) {
                            rspHeaders.put(conn.getHeaderFieldKey(i), conn.getHeaderField(i));
                        }
                    }

                    int totalSize = conn.getContentLength();
                    if (totalSize == -1) {
                        totalSize = 0;
                    }

                    // 读取返回数据
                    boolean hasDownloadFile = downloadFile != null && downloadFile.length() != 0;
                    if (code == HttpURLConnection.HTTP_OK) {
                        responseInputStream = new BufferedInputStream(conn.getInputStream());
                        OutputStream outputStream = null;
                        if (hasDownloadFile) {
                            outputStream = new FileOutputStream(downloadFile);
                        } else {
                            outputStream = new ByteArrayOutputStream();
                        }
                        if (outputStream != null) {
                            int bytesReceived = 0;
                            byte[] buf = new byte[4096];
                            while (true) {
                                int len = responseInputStream.read(buf);
                                if (len < 0) {
                                    break;
                                }
                                bytesReceived += len;
                                outputStream.write(buf, 0, len);
                                if (listener != null) {
                                    listener.onProgress(HTTP_ACTION_RESPONSE, bytesReceived, totalSize);
                                }
                            }
                            if (hasDownloadFile) {
                                response = new byte[] {};
                            } else {
                                response = ((ByteArrayOutputStream) outputStream).toByteArray();
                            }
                            outputStream.close();
                        }
                    }

                    requestResponseCostTime = (int) (SystemClock.uptimeMillis() - request_response_begin_time);
                } catch (UnknownHostException e) {
                    IMLog.e(TAG, "http request failed." + e.getLocalizedMessage());
                    code = HttpURLConnection.HTTP_NOT_FOUND;
                } catch (IOException e) {
                    IMLog.e(TAG, "http request failed." + e.getLocalizedMessage());

                    String exceptionMessage = e.getMessage();
                    if (url.startsWith("http") && exceptionMessage != null
                        && exceptionMessage.toLowerCase().contains("cleartext http traffic")) {
                        code = BaseConstants.ERR_HTTP_NO_CLEARTEXT_TRAFFIC_PERMISSION;
                    } else {
                        code = BaseConstants.ERR_HTTP_REQ_FAILED;
                    }
                } catch (Exception e) {
                    IMLog.e(TAG, "http request failed." + e.getLocalizedMessage());
                    code = BaseConstants.ERR_HTTP_REQ_FAILED;
                    response = Log.getStackTraceString(e).getBytes();
                } finally {
                    // 清理资源
                    if (responseInputStream != null) {
                        try {
                            responseInputStream.close();
                        } catch (IOException e) {
                            e.printStackTrace();
                        }
                    }

                    if (conn != null) {
                        conn.disconnect();
                    }

                    if (listener != null) {
                        if (isStatisticsEnabled) {
                            listener.onStatistics(isDnsSuccess, dnsCostTime, isConnectSuccess, connectCostTime,
                                requestResponseCostTime, remoteIp, remotePort, httpStatus);
                        }

                        listener.onCompleted(code, rspHeaders, response);
                    }
                }
            }
        };
        mThreadPoolExecutor.execute(requestRunnable);
    }

    private static void httpRequest(final String method, final String url, final boolean useOriginIp,
        final String[] headerKeys, final String[] headerValues, final byte[] content, final String uploadFile,
        final String downloadFile, final int proxyType, final String proxyHost, final int proxyPort,
        final String proxyUserName, final String proxyPassword, final int connectTimeout, final int recvTimeout,
        final String rollbackHttps2Http, final boolean isStatisticsEnabled, final long userData) {
        Map<String, String> headers = null;
        if (headerKeys != null && headerValues != null && headerKeys.length == headerValues.length) {
            headers = new HashMap<String, String>();
            for (int i = 0; i < headerKeys.length; ++i) {
                headers.put(headerKeys[i], headerValues[i]);
            }
        }
        httpRequest(method, url, useOriginIp, headers, content, uploadFile, downloadFile, proxyType, proxyHost,
            proxyPort, proxyUserName, proxyPassword, connectTimeout, recvTimeout, rollbackHttps2Http,
            isStatisticsEnabled, new HttpRequestListener() {
                @Override
                public void onProgress(int action, int currentSize, int totalSize) {
                    if (userData != 0) {
                        nativeProgressCallback(action, currentSize, totalSize, userData);
                    }
                }

                @Override
                public void onStatistics(boolean isDnsSuccess, int dnsCostTime, boolean isConnectSuccess,
                    int connectCostTime, int requestResponseCostTime, String remoteIp, int remotePort, int httpStatus) {
                    if (userData != 0) {
                        nativeStatisticsCallback(isDnsSuccess, dnsCostTime, isConnectSuccess, connectCostTime,
                            requestResponseCostTime, remoteIp, remotePort, httpStatus, userData);
                    }
                }

                @Override
                public void onCompleted(int code, Map<String, String> headers, byte[] response) {
                    if (userData != 0) {
                        String[] rspHeaderKeys = null;
                        String[] rspHeaderValues = null;
                        if (headers != null) {
                            rspHeaderKeys = new String[headers.size()];
                            rspHeaderValues = new String[headers.size()];
                            int i = 0;
                            for (Map.Entry<String, String> entry : headers.entrySet()) {
                                rspHeaderKeys[i] = entry.getKey();
                                rspHeaderValues[i] = entry.getValue();
                                i++;
                            }
                        }
                        nativeResponseCallback(code, rspHeaderKeys, rspHeaderValues, response, userData);
                    }
                }
            });
    }

    private static boolean needRollbackHttps2Http(String rollbackHttps2Http) {
        boolean result = false;

        if (TextUtils.isEmpty(rollbackHttps2Http)) {
            return result;
        }

        try {
            JSONArray rollbackHttps2HttpJsonArray = new JSONArray(rollbackHttps2Http);
            // 厂商名均未小写，与云控保持一致
            String brand = "";
            if (SystemUtil.isBrandOppo()) {
                brand = "oppo";
            } else if (SystemUtil.isBrandVivo()) {
                brand = "vivo";
            } else if (SystemUtil.isBrandHuawei()) {
                brand = "huawei";
            } else if (SystemUtil.isBrandXiaoMi()) {
                brand = "xiaomi";
            } else if (SystemUtil.isBrandMeizu()) {
                brand = "meizu";
            }

            for (int i = 0; i < rollbackHttps2HttpJsonArray.length(); i++) {
                JSONObject brandObject = rollbackHttps2HttpJsonArray.getJSONObject(i);
                String brandConfig = brandObject.getString("brand");
                int belowVersion = brandObject.getInt("below_version");
                if (brand.equals(brandConfig)) {
                    int androidVersion = SystemUtil.getSDKVersion();
                    if (androidVersion <= belowVersion) {
                        result = true;
                    }

                    break;
                }
            }
        } catch (JSONException e) {
            e.printStackTrace();
        }

        return result;
    }
}
