package com.tencent.imsdk.group;

import android.text.TextUtils;
import com.tencent.imsdk.BaseConstants;
import com.tencent.imsdk.common.IMCallback;
import com.tencent.imsdk.common.IMContext;
import com.tencent.imsdk.community.TopicInfo;
import com.tencent.imsdk.manager.BaseManager;
import java.io.UnsupportedEncodingException;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

public class GroupManager {
    private GroupListener mGroupInternalListener;
    private GroupListener mGroupListener;

    private static class GroupManagerHolder {
        private static final GroupManager groupManager = new GroupManager();
    }

    public static GroupManager getInstance() {
        return GroupManager.GroupManagerHolder.groupManager;
    }

    public void init() {
        initGroupListener();
    }

    private void initGroupListener() {
        // 统一转到 主线程 抛出去
        if (mGroupInternalListener == null) {
            mGroupInternalListener = new GroupListener() {
                @Override
                public void onMemberEnter(final String groupID, final List<GroupMemberInfo> memberList) {
                    IMContext.getInstance().runOnMainThread(new Runnable() {
                        @Override
                        public void run() {
                            if (mGroupListener != null) {
                                mGroupListener.onMemberEnter(groupID, memberList);
                            }
                        }
                    });
                }

                @Override
                public void onMemberLeave(final String groupID, final GroupMemberInfo member) {
                    IMContext.getInstance().runOnMainThread(new Runnable() {
                        @Override
                        public void run() {
                            if (mGroupListener != null) {
                                mGroupListener.onMemberLeave(groupID, member);
                            }
                        }
                    });
                }

                @Override
                public void onMemberInvited(
                    final String groupID, final GroupMemberInfo opUser, final List<GroupMemberInfo> memberList) {
                    IMContext.getInstance().runOnMainThread(new Runnable() {
                        @Override
                        public void run() {
                            if (mGroupListener != null) {
                                mGroupListener.onMemberInvited(groupID, opUser, memberList);
                            }
                        }
                    });
                }

                @Override
                public void onMemberKicked(final String groupID, final GroupMemberInfo opUser,
                    final List<GroupMemberInfo> memberList, String reason) {
                    IMContext.getInstance().runOnMainThread(new Runnable() {
                        @Override
                        public void run() {
                            if (mGroupListener != null) {
                                mGroupListener.onMemberKicked(groupID, opUser, memberList, reason);
                            }
                        }
                    });
                }

                @Override
                public void onMemberInfoChanged(
                    final String groupID, final List<GroupMemberInfoChangeItem> v2TIMGroupMemberChangeInfoList) {
                    IMContext.getInstance().runOnMainThread(new Runnable() {
                        @Override
                        public void run() {
                            if (mGroupListener != null) {
                                mGroupListener.onMemberInfoChanged(groupID, v2TIMGroupMemberChangeInfoList);
                            }
                        }
                    });
                }

                @Override
                public void onMemberMarkChanged(final String groupID, final List<String> memberIDList,
                    final int markType, final boolean enableMark) {
                    IMContext.getInstance().runOnMainThread(new Runnable() {
                        @Override
                        public void run() {
                            if (mGroupListener != null) {
                                mGroupListener.onMemberMarkChanged(groupID, memberIDList, markType, enableMark);
                            }
                        }
                    });
                }

                @Override
                public void onGroupCreated(final String groupID) {
                    IMContext.getInstance().runOnMainThread(new Runnable() {
                        @Override
                        public void run() {
                            if (mGroupListener != null) {
                                mGroupListener.onGroupCreated(groupID);
                            }
                        }
                    });
                }

                @Override
                public void onGroupDismissed(final String groupID, final GroupMemberInfo opUser) {
                    IMContext.getInstance().runOnMainThread(new Runnable() {
                        @Override
                        public void run() {
                            if (mGroupListener != null) {
                                mGroupListener.onGroupDismissed(groupID, opUser);
                            }
                        }
                    });
                }

                @Override
                public void onGroupRecycled(final String groupID, final GroupMemberInfo opUser) {
                    IMContext.getInstance().runOnMainThread(new Runnable() {
                        @Override
                        public void run() {
                            if (mGroupListener != null) {
                                mGroupListener.onGroupRecycled(groupID, opUser);
                            }
                        }
                    });
                }

                @Override
                public void onGroupInfoChanged(final String groupID, final List<GroupInfoChangeItem> changeInfos) {
                    IMContext.getInstance().runOnMainThread(new Runnable() {
                        @Override
                        public void run() {
                            if (mGroupListener != null) {
                                mGroupListener.onGroupInfoChanged(groupID, changeInfos);
                            }
                            GroupInfoChangeItem shutUpAllChangeInfo = null;
                            for (GroupInfoChangeItem info : changeInfos) {
                                if (info.getGroupInfoChangeType()
                                    == GroupInfoChangeItem.GROUP_INFO_CHANGE_TYPE_SHUT_UP_ALL) {
                                    shutUpAllChangeInfo = info;
                                    break;
                                }
                            }
                            if (shutUpAllChangeInfo != null && mGroupListener != null) {
                                mGroupListener.onAllGroupMembersMuted(
                                    groupID, shutUpAllChangeInfo.getBoolValueChanged());
                            }
                        }
                    });
                }

                @Override
                public void onReceiveJoinApplication(
                    final String groupID, final GroupMemberInfo member, final String opReason) {
                    IMContext.getInstance().runOnMainThread(new Runnable() {
                        @Override
                        public void run() {
                            if (mGroupListener != null) {
                                mGroupListener.onReceiveJoinApplication(groupID, member, opReason);
                            }
                        }
                    });
                }

                @Override
                public void onApplicationProcessed(final String groupID, final GroupMemberInfo opUser,
                    final boolean isAgreeJoin, final String opReason) {
                    IMContext.getInstance().runOnMainThread(new Runnable() {
                        @Override
                        public void run() {
                            if (mGroupListener != null) {
                                mGroupListener.onApplicationProcessed(groupID, opUser, isAgreeJoin, opReason);
                            }
                        }
                    });
                }

                @Override
                public void onReceiveInviteApplication(final String groupID, final int applicationType,
                    final GroupMemberInfo inviter, final List<GroupMemberInfo> inviteeList, final String opReason) {
                    IMContext.getInstance().runOnMainThread(new Runnable() {
                        @Override
                        public void run() {
                            if (mGroupListener != null) {
                                mGroupListener.onReceiveInviteApplication(
                                    groupID, applicationType, inviter, inviteeList, opReason);
                            }
                        }
                    });
                }

                @Override
                public void onGrantAdministrator(
                    final String groupID, final GroupMemberInfo opUser, final List<GroupMemberInfo> memberList) {
                    IMContext.getInstance().runOnMainThread(new Runnable() {
                        @Override
                        public void run() {
                            if (mGroupListener != null) {
                                mGroupListener.onGrantAdministrator(groupID, opUser, memberList);
                            }
                        }
                    });
                }

                @Override
                public void onRevokeAdministrator(
                    final String groupID, final GroupMemberInfo opUser, final List<GroupMemberInfo> memberList) {
                    IMContext.getInstance().runOnMainThread(new Runnable() {
                        @Override
                        public void run() {
                            if (mGroupListener != null) {
                                mGroupListener.onRevokeAdministrator(groupID, opUser, memberList);
                            }
                        }
                    });
                }

                @Override
                public void onQuitFromGroup(final String groupID) {
                    IMContext.getInstance().runOnMainThread(new Runnable() {
                        @Override
                        public void run() {
                            if (mGroupListener != null) {
                                mGroupListener.onQuitFromGroup(groupID);
                            }
                        }
                    });
                }

                @Override
                public void onReceiveRESTCustomData(final String groupID, final byte[] customData) {
                    IMContext.getInstance().runOnMainThread(new Runnable() {
                        @Override
                        public void run() {
                            if (mGroupListener != null) {
                                mGroupListener.onReceiveRESTCustomData(groupID, customData);
                            }
                        }
                    });
                }

                @Override
                public void onGroupAttributeChanged(final String groupID, final Map<String, String> groupAttributeMap) {
                    IMContext.getInstance().runOnMainThread(new Runnable() {
                        @Override
                        public void run() {
                            if (mGroupListener != null) {
                                mGroupListener.onGroupAttributeChanged(groupID, groupAttributeMap);
                            }
                        }
                    });
                }

                @Override
                public void onGroupCounterChanged(final String groupID, final Map<String, Long> counters) {
                    IMContext.getInstance().runOnMainThread(new Runnable() {
                        @Override
                        public void run() {
                            if (mGroupListener != null) {
                                mGroupListener.onGroupCounterChanged(groupID, counters);
                            }
                        }
                    });
                }

                @Override
                public void onGroupCounterDeleted(final String groupID, final List<String> keys) {
                    IMContext.getInstance().runOnMainThread(new Runnable() {
                        @Override
                        public void run() {
                            if (mGroupListener != null) {
                                mGroupListener.onGroupCounterDeleted(groupID, keys);
                            }
                        }
                    });
                }
            };
        }
        nativeSetGroupListener(mGroupInternalListener);
    }

    public void setGroupListener(GroupListener listener) {
        mGroupListener = listener;
    }

    public void notifyTopicCreated(String groupID, String topicID) {
        if (mGroupListener != null) {
            mGroupListener.onTopicCreated(groupID, topicID);
        }
    }

    public void notifyTopicDeleted(String groupID, List<String> topicIDList) {
        if (mGroupListener != null) {
            mGroupListener.onTopicDeleted(groupID, topicIDList);
        }
    }

    public void notifyTopicInfoChanged(String groupID, TopicInfo topicInfo) {
        if (mGroupListener != null) {
            mGroupListener.onTopicInfoChanged(groupID, topicInfo);
        }
    }

    public void notifyReceiveRESTCustomData(String topicID, byte[] customData) {
        if (mGroupListener != null) {
            mGroupListener.onReceiveRESTCustomData(topicID, customData);
        }
    }

    public void createGroup(String groupType, String groupID, String groupName, final IMCallback<String> callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }

        GroupInfo groupInfo = new GroupInfo();
        groupInfo.setGroupID(groupID);
        groupInfo.setGroupType(groupType);
        groupInfo.setGroupName(groupName);

        nativeCreateGroup(groupInfo, null, callback);
    }

    public void createGroup(GroupInfo info, List<GroupMemberInfo> memberList, IMCallback<String> callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }

        nativeCreateGroup(info, memberList, callback);
    }

    public void joinGroup(String groupID, String message, final IMCallback callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }

        byte[] messageBytes;
        if (!TextUtils.isEmpty(message)) {
            try {
                messageBytes = message.getBytes("UTF-8");
            } catch (UnsupportedEncodingException e) {
                e.printStackTrace();
                messageBytes = new byte[0];
            }
        } else {
            messageBytes = new byte[0];
        }

        nativeJoinGroup(groupID, messageBytes, callback);
    }

    public void quitGroup(String groupID, final IMCallback callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }

        nativeQuitGroup(groupID, callback);
    }

    public void dismissGroup(String groupID, final IMCallback callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }

        nativeDismissGroup(groupID, callback);
    }

    public void getJoinedGroupList(IMCallback<List<GroupInfo>> callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }

        nativeGetJoinedGroupList(callback);
    }

    public void getJoinedCommunityList(IMCallback<List<GroupInfo>> callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }

        nativeGetJoinedCommunityList(callback);
    }

    public void getGroupsInfo(List<String> groupIDList, IMCallback<List<GroupInfoGetResult>> callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }

        nativeGetGroupsInfo(groupIDList, callback);
    }

    public void searchLocalGroups(GroupSearchParam param, IMCallback<List<GroupInfo>> callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }

        nativeSearchLocalGroups(param, callback);
    }

    public void searchCloudGroups(GroupSearchParam param, IMCallback<GroupSearchResult> callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }

        nativeSearchCloudGroups(param, callback);
    }

    public void setGroupInfo(GroupInfoModifyParam param, IMCallback callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }

        nativeSetGroupInfo(param, callback);
    }

    public void setGroupReceiveMessageOpt(String groupID, int opt, IMCallback callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }

        nativeSetGroupMessageReceiveOption(groupID, opt, callback);
    }

    public void initGroupAttributes(String groupID, HashMap<String, String> attributes, IMCallback callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }

        nativeInitGroupAttributes(groupID, attributes, callback);
    }

    public void setGroupAttributes(String groupID, HashMap<String, String> attributes, IMCallback callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }

        nativeUpdateGroupAttributes(groupID, attributes, callback);
    }

    public void deleteGroupAttributes(String groupID, List<String> keys, IMCallback callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }

        nativeDeleteGroupAttributes(groupID, keys, callback);
    }

    public void getGroupAttributes(String groupID, List<String> keys, IMCallback<Map<String, String>> callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }

        nativeGetGroupAttributes(groupID, keys, callback);
    }

    public void getGroupOnlineMemberCount(String groupID, IMCallback<Integer> callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }

        nativeGetGroupOnlineMemberCount(groupID, callback);
    }

    public void setGroupCounters(
        String groupID, HashMap<String, Long> counters, IMCallback<Map<String, Long>> callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }

        nativeSetGroupCounters(groupID, counters, callback);
    }

    public void getGroupCounters(String groupID, List<String> keys, IMCallback<Map<String, Long>> callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }

        nativeGetGroupCounters(groupID, keys, callback);
    }

    public void increaseGroupCounter(String groupID, String key, long value, IMCallback<Map<String, Long>> callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }

        nativeIncreaseGroupCounter(groupID, key, value, callback);
    }

    public void decreaseGroupCounter(String groupID, String key, long value, IMCallback<Map<String, Long>> callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }

        nativeDecreaseGroupCounter(groupID, key, value, callback);
    }

    public void getGroupMemberList(
        String groupID, int filter, long nextSeq, IMCallback<GroupMemberInfoResult> callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }

        nativeGetGroupMemberList(groupID, filter, nextSeq, callback);
    }

    public void getGroupMembersInfo(
        String groupID, List<String> memberList, IMCallback<List<GroupMemberInfo>> callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }

        nativeGetGroupMembersInfo(groupID, memberList, callback);
    }

    public void setGroupMemberInfo(GroupMemberInfoModifyParam param, IMCallback callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }

        nativeSetGroupMemberInfo(param, callback);
    }

    public void searchLocalGroupMembersInfo(
        GroupMemberSearchParam groupMemberSearchParam, IMCallback<HashMap<String, List<GroupMemberInfo>>> callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }

        nativeSearchLocalGroupMembers(groupMemberSearchParam, callback);
    }

    public void searchCloudGroupMembers(
        GroupMemberSearchParam groupMemberSearchParam, IMCallback<GroupMemberSearchResult> callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }

        nativeSearchCloudGroupMembers(groupMemberSearchParam, callback);
    }

    public void muteGroupMember(String groupID, String userID, int seconds, IMCallback callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }

        nativeMuteGroupMember(groupID, userID, seconds, callback);
    }

    public void muteAllGroupMembers(String groupID, boolean isMute, IMCallback callback) {
        GroupInfo groupInfo = new GroupInfo();
        groupInfo.setGroupID(groupID);
        groupInfo.setAllShutUp(isMute);

        GroupInfoModifyParam param = new GroupInfoModifyParam();
        param.setGroupInfo(groupInfo);
        param.setModifyFlag(GroupInfo.GROUP_INFO_MODIFY_FLAG_SHUTUP_ALL);

        setGroupInfo(param, callback);
    }

    public void inviteUserToGroup(
        String groupID, List<String> userList, IMCallback<List<GroupMemberOperationResult>> callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }

        nativeInviteGroupMembers(groupID, userList, "", callback);
    }

    public void kickGroupMember(
        String groupID, List<String> memberList, String reason, IMCallback<List<GroupMemberOperationResult>> callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }

        nativeDeleteGroupMembers(groupID, memberList, reason, callback);
    }

    public void banGroupMember(String groupID, List<String> memberList, String reason, int duration,
        IMCallback<List<GroupMemberOperationResult>> callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }

        nativeBanGroupMembers(groupID, memberList, reason, duration, callback);
    }

    public void setGroupMemberRole(String groupID, String userID, int role, IMCallback callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }

        nativeSetGroupMemberRole(groupID, userID, role, callback);
    }

    public void markGroupMemberList(
        String groupID, List<String> memberIDList, int markType, boolean enableMark, IMCallback callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }

        nativeMarkGroupMemberList(groupID, memberIDList, markType, enableMark, callback);
    }

    public void transferGroupOwner(String groupID, String userID, IMCallback callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }

        nativeSetGroupOwner(groupID, userID, callback);
    }

    public void getGroupApplicationList(IMCallback<GroupApplicationResult> callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }

        nativeGetGroupApplicationList(callback);
    }

    public void acceptGroupApplication(GroupApplication application, String reason, IMCallback callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }

        application.setResponseType(GroupApplication.RESPONSE_TYPE_AGREE);
        application.setResponseMessage(reason);
        nativeResponseGroupApplication(application, callback);
    }

    public void refuseGroupApplication(GroupApplication application, String reason, IMCallback callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }

        application.setResponseType(GroupApplication.RESPONSE_TYPE_REFUSE);
        application.setResponseMessage(reason);
        nativeResponseGroupApplication(application, callback);
    }

    public void setGroupApplicationRead(IMCallback callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }

        nativeSetGroupApplicationListRead(callback);
    }

    ///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

    protected native void nativeSetGroupListener(GroupListener listener);

    protected native void nativeCreateGroup(
        GroupInfo groupInfo, List<GroupMemberInfo> memberInfoList, IMCallback<String> callback);

    protected native void nativeJoinGroup(String groupID, byte[] applyMessage, final IMCallback callback);

    protected native void nativeQuitGroup(String groupID, IMCallback callback);

    protected native void nativeDismissGroup(String groupID, final IMCallback callback);

    protected native void nativeInviteGroupMembers(
        String groupID, List<String> memberList, String inviteMessage, IMCallback callback);

    protected native void nativeDeleteGroupMembers(
        String groupID, List<String> memberList, String deleteMessage, IMCallback callback);

    protected native void nativeBanGroupMembers(
        String groupID, List<String> memberList, String deleteMessage, int duration, IMCallback callback);

    protected native void nativeGetJoinedGroupList(IMCallback callback);

    protected native void nativeGetJoinedCommunityList(IMCallback callback);

    protected native void nativeGetGroupsInfo(List<String> groupIDList, IMCallback callback);

    protected native void nativeSearchLocalGroups(GroupSearchParam param, IMCallback callback);

    protected native void nativeSearchCloudGroups(GroupSearchParam param, IMCallback callback);

    protected native void nativeSetGroupInfo(GroupInfoModifyParam param, IMCallback callback);

    protected native void nativeSetGroupOwner(String groupID, String ownerUserID, IMCallback callback);

    protected native void nativeGetGroupMemberList(
        String groupID, int filterFlag, long lastSequence, IMCallback callback);

    protected native void nativeGetGroupMembersInfo(String groupID, List<String> memberList, IMCallback callback);

    protected native void nativeSetGroupMemberInfo(GroupMemberInfoModifyParam param, IMCallback callback);

    protected native void nativeSearchLocalGroupMembers(GroupMemberSearchParam param, IMCallback callback);

    protected native void nativeSearchCloudGroupMembers(GroupMemberSearchParam param, IMCallback callback);

    protected native void nativeSetGroupMemberRole(String groupID, String userID, int role, IMCallback callback);

    protected native void nativeMarkGroupMemberList(
        String groupID, List<String> memberIDList, int markType, boolean enableMark, IMCallback callback);

    protected native void nativeMuteGroupMember(String groupID, String userID, int seconds, IMCallback callback);

    protected native void nativeSetGroupMessageReceiveOption(String groupID, int opt, IMCallback callback);

    protected native void nativeGetGroupApplicationList(IMCallback callback);

    protected native void nativeSetGroupApplicationListRead(IMCallback callback);

    protected native void nativeResponseGroupApplication(GroupApplication groupApplication, IMCallback callback);

    protected native void nativeInitGroupAttributes(
        String groupID, Map<String, String> groupAttributes, IMCallback callback);

    protected native void nativeUpdateGroupAttributes(
        String groupID, Map<String, String> groupAttributes, IMCallback callback);

    protected native void nativeDeleteGroupAttributes(
        String groupID, List<String> groupAttributeKeys, IMCallback callback);

    protected native void nativeGetGroupAttributes(
        String groupID, List<String> groupAttributeKeys, IMCallback callback);

    protected native void nativeGetGroupOnlineMemberCount(String groupID, IMCallback callback);

    protected native void nativeSetGroupCounters(String groupID, Map<String, Long> counters, IMCallback callback);

    protected native void nativeGetGroupCounters(String groupID, List<String> keys, IMCallback callback);

    protected native void nativeIncreaseGroupCounter(String groupID, String key, long value, IMCallback callback);

    protected native void nativeDecreaseGroupCounter(String groupID, String key, long value, IMCallback callback);
}
