package com.tencent.imsdk.v2;

import android.content.Context;
import android.text.TextUtils;
import com.tencent.imsdk.BaseConstants;
import com.tencent.imsdk.common.IMCallback;
import com.tencent.imsdk.common.IMContext;
import com.tencent.imsdk.common.IMLog;
import com.tencent.imsdk.common.SystemUtil;
import com.tencent.imsdk.conversation.ConversationKey;
import com.tencent.imsdk.conversation.ConversationManager;
import com.tencent.imsdk.group.GroupMemberInfo;
import com.tencent.imsdk.manager.BaseManager;
import com.tencent.imsdk.manager.CustomServerInfo;
import com.tencent.imsdk.manager.SDKConfig;
import com.tencent.imsdk.message.FileElement;
import com.tencent.imsdk.message.ImageElement;
import com.tencent.imsdk.message.Message;
import com.tencent.imsdk.message.MessageCenter;
import com.tencent.imsdk.message.MessageExtension;
import com.tencent.imsdk.message.MessageExtensionResult;
import com.tencent.imsdk.message.SoundElement;
import com.tencent.imsdk.message.TextElement;
import com.tencent.imsdk.message.VideoElement;
import com.tencent.imsdk.officialaccount.OfficialAccountInfo;
import com.tencent.imsdk.offlinepush.OfflinePushConfig;
import com.tencent.imsdk.offlinepush.OfflinePushEventItem;
import com.tencent.imsdk.offlinepush.OfflinePushManager;
import com.tencent.imsdk.relationship.RelationshipManager;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

public class V2TIMManagerExperimentImpl {
    private static final String TAG = V2TIMManagerExperimentImpl.class.getSimpleName();

    private static class V2TIMManagerExperimentImplHolder {
        private static final V2TIMManagerExperimentImpl v2TIMManagerExperiment = new V2TIMManagerExperimentImpl();
    }

    public static V2TIMManagerExperimentImpl getInstance() {
        return V2TIMManagerExperimentImpl.V2TIMManagerExperimentImplHolder.v2TIMManagerExperiment;
    }

    protected V2TIMManagerExperimentImpl() {}

    public void callExperimentalAPI(String api, Object param, V2TIMValueCallback<Object> callback) {
        if (TextUtils.isEmpty(api)) {
            callbackOnError(callback, BaseConstants.ERR_INVALID_PARAMETERS, "invalid api");
        }

        if (api.equals("setCustomServerInfo")) {
            setCustomServerInfo(param, callback);
        } else if (api.equals("enableQuicChannel")) {
            setQuicChannelEnabled(param, callback);
        } else if (api.equals("setProxyInfo")) {
            setProxyInfo(param, callback);
        } else if (api.equals("setLibraryPath")) {
            setLibraryPath(param, callback);
        } else if (api.equals("downloadLibrary")) {
            downloadLibrary(param, callback);
        } else if (api.equals("initLocalStorage")) {
            initLocalStorage(param, callback);
        } else if (api.equals("setTestEnvironment")) {
            setTestEnvironment(param, callback);
        } else if (api.equals("setIPv6Prior")) {
            setIPv6Prior(param, callback);
        } else if (api.equals("setCosSaveRegionForConversation")) {
            setCosSaveRegion(param, callback);
        } else if (api.equals("setUIPlatform")) {
            setUIPlatform(param, callback);
        } else if (api.equals("setBuildInfo")) {
            setBuildInfo(param, callback);
        } else if (api.equals("setDatabaseEncryptInfo")) {
            setDatabaseEncryptInfo(param, callback);
        } else if (api.equals("isCommercialAbilityEnabled")) {
            isCommercialAbilityEnabled(param, callback);
        } else if (api.equals("setPacketRetryInfo")) {
            setPacketRetryInfo(param, callback);
        } else if (api.equals("setOfflinePushState")) {
            setOfflinePushState(param, callback);
        } else if (api.equals("getOfflinePushState")) {
            getOfflinePushState(callback);
        } else if (api.equals("getMessageRevoker")) {
            getMessageRevoker(param, callback);
        } else if (api.equals("writeLog")) {
            writeLog(param, callback);
        } else if (api.equals("getAIDenoiseSignature")) {
            getAIDenoiseSignature(callback);
        } else if (api.equals("getVideoEditSignature")) {
            getVideoEditSignature(callback);
        } else if (api.equals("sendTRTCCustomData")) {
            sendTRTCCustomData(param, callback);
        } else if (api.equals("setMessageExtensionsBySequence")) {
            setMessageExtensionsBySequence(param, callback);
        } else if (api.equals("getMessageExtensionsBySequence")) {
            getMessageExtensionsBySequence(param, callback);
        } else if (api.equals("clearLocalHistoryMessage")) {
            clearLocalHistoryMessage(param, callback);
        } else if (api.equals("reportTUIComponentUsage")) {
            reportTUIComponentUsage(param, callback);
        } else if (api.equals("reportTUIFeatureUsage")) {
            reportTUIFeatureUsage(param, callback);
        } else if (api.equals("setConversationGroupOrderData")) {
            setConversationGroupOrderData(param, callback);
        } else if (api.equals("getConversationGroupOrderData")) {
            getConversationGroupOrderData(param, callback);
        } else if (api.equals("reportOfflinePushEvent")) {
            reportOfflinePushEvent(param, callback);
        } else if (api.equals("setOfflinePushInfo")) {
            setOfflinePushInfo(param, callback);
        } else if (api.equals("setApplicationID")) {
            setApplicationID(param, callback);
        } else if (api.equals("findMergerMessages")) {
            findMergerMessages(param, callback);
        } else if (api.equals("getBriefGroupMemberList")) {
            getBriefGroupMemberList(param, callback);
        } else if (api.equals("pushLogin")) {
            pushLogin(param, callback);
        } else if (api.equals("reportPushSDKEvent")) {
            reportPushSDKEvent(param, callback);
        } else if (api.equals("setCustomLoginInfo")) {
            setCustomLoginInfo(param, callback);
        } else if (api.equals("disableHttpRequest")) {
            disableHttpRequest(param, callback);
        } else if (api.equals("getLoginAccountType")) {
            getLoginAccountType(param, callback);
        } else if (api.equals("cancelSendingMessage")) {
            cancelSendingMessage(param, callback);
        } else if (api.equals("getMessageSource")) {
            getMessageSource(param, callback);
        } else if (api.equals("getGroupMessageReceipts")) {
            getGroupMessageReceipts(param, callback);
        } else if (api.equals("getPushMessageID")) {
            getPushMessageID(param, callback);
        } else if (api.equals("getOfficialAccountList")) {
            getOfficialAccountList(param, callback);
        } else if (api.equals("setForceRequestCloudMessage")) {
            setForceRequestCloudMessage(param, callback);
        } else if (api.equals("uploadFile")) {
            uploadFile(param, callback);
        } else if (api.equals("convertVoiceToText")) {
            convertVoiceToText(param, callback);
        } else if (api.equals("reportRoomEngineEvent")) {
            reportRoomEngineEvent(param, callback);
        } else {
            callbackOnError(callback, BaseConstants.ERR_INVALID_PARAMETERS, "unsupported api");
        }
    }

    public Object callExperimentalAPI(String api, Object param) {
        if (TextUtils.isEmpty(api)) {
            IMLog.e(TAG, "empty api");
            return null;
        }

        if (api.equals("createImageMessage")) {
            return createImageMessage(param);
        } else if (api.equals("createSoundMessage")) {
            return createSoundMessage(param);
        } else if (api.equals("createVideoMessage")) {
            return createVideoMessage(param);
        } else if (api.equals("createFileMessage")) {
            return createFileMessage(param);
        } else {
            IMLog.e(TAG, "invalid api: " + api);
        }

        return null;
    }

    private void setConversationGroupOrderData(Object param, final V2TIMValueCallback<Object> callback) {
        if (null == param || !(param instanceof String)) {
            callbackOnError(callback, BaseConstants.ERR_INVALID_PARAMETERS, "param is not string");
            return;
        }
        String conversationGroupOrderData = (String) param;
        ConversationManager.getInstance().setConversationGroupOrderData(
            conversationGroupOrderData, new IMCallback(callback) {
                @Override
                public void success(Object data) {
                    super.success(data);
                }

                @Override
                public void fail(int code, String errorMessage) {
                    super.fail(code, errorMessage);
                }
            });
    }

    private void getConversationGroupOrderData(Object param, final V2TIMValueCallback<Object> callback) {
        V2TIMValueCallback<String> v2TIMValueCallback = new V2TIMValueCallback<String>() {
            @Override
            public void onSuccess(String orderData) {
                callback.onSuccess(orderData);
            }

            @Override
            public void onError(int code, String desc) {
                if (callback != null) {
                    callback.onError(code, desc);
                }
            }
        };

        ConversationManager.getInstance().getConversationGroupOrderData(new IMCallback<String>(v2TIMValueCallback) {
            @Override
            public void success(String orderData) {
                super.success(orderData);
            }

            @Override
            public void fail(int code, String errorMessage) {
                super.fail(code, errorMessage);
            }
        });
    }

    private void reportTUIComponentUsage(Object param, final V2TIMValueCallback<Object> callback) {
        if (null == param || !(param instanceof String)) {
            callbackOnError(callback, BaseConstants.ERR_INVALID_PARAMETERS, "param is not string");
            return;
        }

        String json = (String) param;
        if (TextUtils.isEmpty(json)) {
            callbackOnError(callback, BaseConstants.ERR_INVALID_PARAMETERS, "param is empty");
            return;
        }

        try {
            JSONObject jsonObject = new JSONObject(json);
            long uiComponentType = jsonObject.optLong("UIComponentType");
            long uiStyleType = jsonObject.optLong("UIStyleType");

            BaseManager.getInstance().reportTUIComponentUsage(uiComponentType, uiStyleType);

            callbackOnSuccess(callback, null);
        } catch (JSONException e) {
            callbackOnError(callback, BaseConstants.ERR_INVALID_PARAMETERS, "convert param to json failed");
            e.printStackTrace();
        }
    }

    private void reportTUIFeatureUsage(Object param, final V2TIMValueCallback<Object> callback) {
        if (null == param || !(param instanceof String)) {
            callbackOnError(callback, BaseConstants.ERR_INVALID_PARAMETERS, "param is not string");
            return;
        }

        String json = (String) param;
        if (TextUtils.isEmpty(json)) {
            callbackOnError(callback, BaseConstants.ERR_INVALID_PARAMETERS, "param is empty");
            return;
        }

        try {
            JSONObject jsonObject = new JSONObject(json);
            if (!jsonObject.has("UIComponentType") && !jsonObject.has("UIFeatureType")) {
                callbackOnError(callback, BaseConstants.ERR_INVALID_PARAMETERS, "invalid param");
                return;
            }

            long componentType = 0;
            long featureType = 0;
            if (jsonObject.has("UIComponentType")) {
                componentType = jsonObject.optLong("UIComponentType");
            }
            if (jsonObject.has("UIFeatureType")) {
                featureType = jsonObject.optLong("UIFeatureType");
            }

            BaseManager.getInstance().reportTUIFeatureUsage(componentType, featureType);

            callbackOnSuccess(callback, null);
        } catch (JSONException e) {
            callbackOnError(callback, BaseConstants.ERR_INVALID_PARAMETERS, "convert param to json failed");
            e.printStackTrace();
        }
    }

    private void setMessageExtensionsBySequence(Object param, final V2TIMValueCallback<Object> callback) {
        if (null == param || !(param instanceof String)) {
            callbackOnError(callback, BaseConstants.ERR_INVALID_PARAMETERS, "param is not string");
            return;
        }

        String json = (String) param;
        if (TextUtils.isEmpty(json)) {
            callbackOnError(callback, BaseConstants.ERR_INVALID_PARAMETERS, "param is empty");
            return;
        }

        String groupID = "";
        long messageSequence = 0;
        List<MessageExtension> extensionList = new ArrayList<>();
        try {
            JSONObject jsonObject = new JSONObject(json);
            groupID = jsonObject.optString("groupID");
            messageSequence = jsonObject.optLong("messageSequence");

            JSONArray extensionJsonArray = jsonObject.optJSONArray("messageExtensionList");
            for (int i = 0; i < extensionJsonArray.length(); i++) {
                JSONObject extensionJson = extensionJsonArray.getJSONObject(i);
                MessageExtension extension = new MessageExtension();
                extension.setExtensionKey(extensionJson.optString("extensionKey"));
                extension.setExtensionValue(extensionJson.optString("extensionValue"));
                extensionList.add(extension);
            }
        } catch (JSONException e) {
            e.printStackTrace();
        }

        final V2TIMValueCallback<List<MessageExtensionResult>> v2TIMValueCallback =
            new V2TIMValueCallback<List<MessageExtensionResult>>() {
                @Override
                public void onSuccess(List<MessageExtensionResult> messageExtensionResults) {
                    if (callback != null) {
                        List<V2TIMMessageExtensionResult> v2TIMMessageExtensionResults = new ArrayList<>();
                        for (MessageExtensionResult result : messageExtensionResults) {
                            V2TIMMessageExtensionResult v2TIMMessageExtensionResult = new V2TIMMessageExtensionResult();
                            v2TIMMessageExtensionResult.setMessageExtensionResult(result);
                            v2TIMMessageExtensionResults.add(v2TIMMessageExtensionResult);
                        }
                        callback.onSuccess(v2TIMMessageExtensionResults);
                    }
                }

                @Override
                public void onError(int code, String desc) {
                    if (callback != null) {
                        callback.onError(code, desc);
                    }
                }
            };

        Message message = new Message();
        message.setMessageStatus(Message.V2TIM_MSG_STATUS_SUCCESS);
        message.setSupportMessageExtension(true);
        message.addElement(new TextElement());
        message.setMessageType(V2TIMConversation.V2TIM_GROUP);
        message.setGroupID(groupID);
        message.setSeq(messageSequence);

        MessageCenter.getInstance().setMessageExtensions(
            message, extensionList, new IMCallback<List<MessageExtensionResult>>(v2TIMValueCallback) {
                @Override
                public void success(List<MessageExtensionResult> data) {
                    super.success(data);
                }

                @Override
                public void fail(int code, String errorMessage) {
                    super.fail(code, errorMessage);
                }
            });
    }

    private void getMessageExtensionsBySequence(Object param, final V2TIMValueCallback<Object> callback) {
        if (null == param || !(param instanceof String)) {
            callbackOnError(callback, BaseConstants.ERR_INVALID_PARAMETERS, "param is not string");
            return;
        }

        String json = (String) param;
        if (TextUtils.isEmpty(json)) {
            callbackOnError(callback, BaseConstants.ERR_INVALID_PARAMETERS, "param is empty");
            return;
        }

        String groupID = "";
        long messageSequence = 0;
        List<MessageExtension> extensionList = new ArrayList<>();
        try {
            JSONObject jsonObject = new JSONObject(json);
            groupID = jsonObject.optString("groupID");
            messageSequence = jsonObject.optLong("messageSequence");
        } catch (JSONException e) {
            e.printStackTrace();
        }

        final V2TIMValueCallback<List<MessageExtension>> v2TIMValueCallback =
            new V2TIMValueCallback<List<MessageExtension>>() {
                @Override
                public void onSuccess(List<MessageExtension> extensions) {
                    if (callback != null) {
                        List<V2TIMMessageExtension> v2TIMMessageExtensions = new ArrayList<>();
                        for (MessageExtension extension : extensions) {
                            V2TIMMessageExtension v2TIMMessageExtension = new V2TIMMessageExtension();
                            v2TIMMessageExtension.setMessageExtension(extension);
                            v2TIMMessageExtensions.add(v2TIMMessageExtension);
                        }
                        callback.onSuccess(v2TIMMessageExtensions);
                    }
                }

                @Override
                public void onError(int code, String desc) {
                    if (callback != null) {
                        callback.onError(code, desc);
                    }
                }
            };

        Message message = new Message();
        message.setMessageStatus(Message.V2TIM_MSG_STATUS_SUCCESS);
        message.setSupportMessageExtension(true);
        message.addElement(new TextElement());
        message.setMessageType(V2TIMConversation.V2TIM_GROUP);
        message.setGroupID(groupID);
        message.setSeq(messageSequence);

        MessageCenter.getInstance().getMessageExtensions(
            message, new IMCallback<List<MessageExtension>>(v2TIMValueCallback) {
                @Override
                public void success(List<MessageExtension> data) {
                    super.success(data);
                }

                @Override
                public void fail(int code, String errorMessage) {
                    super.fail(code, errorMessage);
                }
            });
    }

    private void clearLocalHistoryMessage(Object param, final V2TIMValueCallback<Object> callback) {
        if (null == param || !(param instanceof String)) {
            callbackOnError(callback, BaseConstants.ERR_INVALID_PARAMETERS, "param is not string");
            return;
        }

        String json = (String) param;
        if (TextUtils.isEmpty(json)) {
            callbackOnError(callback, BaseConstants.ERR_INVALID_PARAMETERS, "param is empty");
            return;
        }

        try {
            JSONObject jsonObject = new JSONObject(json);
            String conversationID = jsonObject.optString("conversationID");
            ConversationKey conversationKey =
                V2TIMConversationManagerImpl.getInstance().getConversationKey(conversationID);
            long beginTimestamp = jsonObject.optLong("beginTimestamp");
            long endTimestamp = jsonObject.optLong("endTimestamp");

            MessageCenter.getInstance().clearLocalHistoryMessage(
                conversationKey, beginTimestamp, endTimestamp, new IMCallback(callback) {
                    @Override
                    public void success(Object data) {
                        super.success(data);
                    }

                    @Override
                    public void fail(int code, String errorMessage) {
                        super.fail(code, errorMessage);
                    }
                });
        } catch (JSONException e) {
            callbackOnError(callback, BaseConstants.ERR_INVALID_PARAMETERS, "convert param to json failed");
            e.printStackTrace();
        }
    }

    private void sendTRTCCustomData(Object param, V2TIMValueCallback<Object> callback) {
        if (null == param || !(param instanceof byte[])) {
            callbackOnError(callback, BaseConstants.ERR_INVALID_PARAMETERS, "param is not byte array");
            return;
        }

        byte[] data = (byte[]) param;

        BaseManager.getInstance().sendTRTCCustomData(data, new IMCallback(callback) {
            @Override
            public void success(Object data) {
                super.success(data);
            }

            @Override
            public void fail(int code, String errorMessage) {
                super.fail(code, errorMessage);
            }
        });
    }

    private void getAIDenoiseSignature(final V2TIMValueCallback<Object> callback) {
        BaseManager.getInstance().getAIDenoiseSignature(new IMCallback(callback) {
            @Override
            public void success(Object data) {
                super.success(data);
            }

            @Override
            public void fail(int code, String errorMessage) {
                super.fail(code, errorMessage);
            }
        });
    }

    private void getVideoEditSignature(final V2TIMValueCallback<Object> callback) {
        BaseManager.getInstance().getVideoEditSignature(new IMCallback(callback) {
            @Override
            public void success(Object data) {
                super.success(data);
            }

            @Override
            public void fail(int code, String errorMessage) {
                super.fail(code, errorMessage);
            }
        });
    }

    private void writeLog(Object param, final V2TIMValueCallback<Object> callback) {
        if (null == param || !(param instanceof String)) {
            callbackOnError(callback, BaseConstants.ERR_INVALID_PARAMETERS, "param is not string");
            return;
        }

        String json = (String) param;
        if (TextUtils.isEmpty(json)) {
            callbackOnError(callback, BaseConstants.ERR_INVALID_PARAMETERS, "param is empty");
            return;
        }

        try {
            JSONObject jsonObject = new JSONObject(json);
            int logLevel = jsonObject.optInt("logLevel");
            String logContent = jsonObject.optString("logContent");
            String fileName = jsonObject.optString("fileName");

            switch (logLevel) {
                case IMLog.LOG_LEVEL_VERBOSE:
                    IMLog.v(fileName, logContent);
                    break;
                case IMLog.LOG_LEVEL_DEBUG:
                    IMLog.d(fileName, logContent);
                    break;
                case IMLog.LOG_LEVEL_INFO:
                    IMLog.i(fileName, logContent);
                    break;
                case IMLog.LOG_LEVEL_WARN:
                    IMLog.w(fileName, logContent);
                    break;
                case IMLog.LOG_LEVEL_ERROR:
                    IMLog.e(fileName, logContent);
                    break;
                default:
                    callbackOnError(callback, BaseConstants.ERR_INVALID_PARAMETERS, "invalid logLevel");
                    return;
            }

            callbackOnSuccess(callback, null);
        } catch (JSONException e) {
            callbackOnError(callback, BaseConstants.ERR_INVALID_PARAMETERS, "convert param to json failed");
            e.printStackTrace();
        }
    }

    private void getMessageRevoker(Object param, final V2TIMValueCallback<Object> callback) {
        if (!(param instanceof List)) {
            callbackOnError(callback, BaseConstants.ERR_INVALID_PARAMETERS, "param is not list");
            return;
        }
        List messageIDList = (List) param;
        for (Object messageID : messageIDList) {
            if (!(messageID instanceof String)) {
                callbackOnError(callback, BaseConstants.ERR_INVALID_PARAMETERS, "item is no String type");
                return;
            }
            break;
        }

        V2TIMValueCallback<List<Message>> v2TIMValueCallback = new V2TIMValueCallback<List<Message>>() {
            @Override
            public void onSuccess(List<Message> messageList) {
                if (messageList.size() == 0) {
                    if (callback != null) {
                        callback.onError(BaseConstants.ERR_INVALID_PARAMETERS, "local messages do not exist");
                    }
                } else {
                    if (callback != null) {
                        HashMap<String, String> revokerMap = new HashMap<>();
                        for (Message message : messageList) {
                            revokerMap.put(message.getMsgID(), message.getRevokerInfo().getUserID());
                        }
                        callback.onSuccess(revokerMap);
                    }
                }
            }

            @Override
            public void onError(int code, String desc) {
                if (callback != null) {
                    callback.onError(code, desc);
                }
            }
        };

        MessageCenter.getInstance().findMessageByMessageId(
            messageIDList, new IMCallback<List<Message>>(v2TIMValueCallback) {
                @Override
                public void success(List<Message> data) {
                    super.success(data);
                }

                @Override
                public void fail(int code, String errorMessage) {
                    super.fail(code, errorMessage);
                }
            });
    }

    private void setOfflinePushState(Object param, V2TIMValueCallback<Object> callback) {
        if (null == param || !(param instanceof Integer)) {
            callbackOnError(callback, BaseConstants.ERR_INVALID_PARAMETERS, "param is not int");
            return;
        }

        OfflinePushConfig pushConfig = new OfflinePushConfig();
        pushConfig.setOpenOfflinePush((int) param);
        OfflinePushManager.getInstance().setOfflinePushConfig(pushConfig, new IMCallback(callback) {
            @Override
            public void success(Object data) {
                super.success(data);
            }

            @Override
            public void fail(int code, String errorMessage) {
                super.fail(code, errorMessage);
            }
        });
    }

    private void getOfflinePushState(V2TIMValueCallback<Object> valueCallback) {
        OfflinePushManager.getInstance().getOfflinePushConfig(new IMCallback(valueCallback) {
            @Override
            public void success(Object data) {
                OfflinePushConfig pushConfig = (OfflinePushConfig) data;
                final int openPush = pushConfig.getOpenOfflinePush();
                IMContext.getInstance().runOnMainThread(new Runnable() {
                    @Override
                    public void run() {
                        if (valueCallback != null) {
                            valueCallback.onSuccess(openPush);
                        }
                    }
                });
            }

            @Override
            public void fail(int code, String errorMessage) {
                super.fail(code, errorMessage);
            }
        });
    }

    private void setBuildInfo(Object param, V2TIMValueCallback<Object> callback) {
        if (null == param || !(param instanceof String)) {
            callbackOnError(callback, BaseConstants.ERR_INVALID_PARAMETERS, "param is not string");
            return;
        }

        String json = (String) param;
        if (TextUtils.isEmpty(json)) {
            callbackOnError(callback, BaseConstants.ERR_INVALID_PARAMETERS, "param is empty");
            return;
        }

        try {
            JSONObject jsonObject = new JSONObject(json);
            String buildBrand = jsonObject.optString("buildBrand");
            String buildManufacturer = jsonObject.optString("buildManufacturer");
            String buildModel = jsonObject.optString("buildModel");
            String buildVersionRelease = jsonObject.optString("buildVersionRelease");
            int buildVersionSDKInt = jsonObject.optInt("buildVersionSDKInt");

            SystemUtil.setBuildBrand(buildBrand);
            SystemUtil.setBuildManufacturer(buildManufacturer);
            SystemUtil.setBuildModel(buildModel);
            SystemUtil.setBuildVersionRelease(buildVersionRelease);
            SystemUtil.setBuildVersionSDKInt(buildVersionSDKInt);

            callbackOnSuccess(callback, null);
        } catch (JSONException e) {
            callbackOnError(callback, BaseConstants.ERR_INVALID_PARAMETERS, "convert param to json failed");
            e.printStackTrace();
        }
    }

    private void setDatabaseEncryptInfo(Object param, V2TIMValueCallback<Object> callback) {
        if (null == param || !(param instanceof String)) {
            callbackOnError(callback, BaseConstants.ERR_INVALID_PARAMETERS, "param is not string");
            return;
        }

        String json = (String) param;
        if (TextUtils.isEmpty(json)) {
            callbackOnError(callback, BaseConstants.ERR_INVALID_PARAMETERS, "param is empty");
            return;
        }

        try {
            JSONObject jsonObject = new JSONObject(json);
            if (!jsonObject.has("encryptType")) {
                callbackOnError(callback, BaseConstants.ERR_INVALID_PARAMETERS, "encryptType must be set");
                return;
            }

            SDKConfig.DatabaseEncryptInfo databaseEncryptInfo = new SDKConfig.DatabaseEncryptInfo();
            databaseEncryptInfo.encryptType = jsonObject.optInt("encryptType");
            if (jsonObject.has("encryptKey")) {
                databaseEncryptInfo.encryptKey = jsonObject.optString("encryptKey");
            }

            BaseManager.getInstance().setDatabaseEncryptInfo(databaseEncryptInfo);

            callbackOnSuccess(callback, null);
        } catch (JSONException e) {
            callbackOnError(callback, BaseConstants.ERR_INVALID_PARAMETERS, "convert param to json failed");
            e.printStackTrace();
        }
    }

    private void isCommercialAbilityEnabled(Object param, V2TIMValueCallback<Object> callback) {
        if (null == param || !(param instanceof Long)) {
            callbackOnError(callback, BaseConstants.ERR_INVALID_PARAMETERS, "param is not int");
            return;
        }

        Long longParam = (Long) param;
        BaseManager.getInstance().isCommercialAbilityEnabled(longParam.longValue(), new IMCallback<Object>(callback) {
            @Override
            public void success(Object data) {
                super.success(data);
            }

            @Override
            public void fail(int code, String errorMessage) {
                super.fail(code, errorMessage);
            }
        });
    }

    private void setPacketRetryInfo(Object param, V2TIMValueCallback<Object> callback) {
        if (null == param || !(param instanceof String)) {
            callbackOnError(callback, BaseConstants.ERR_INVALID_PARAMETERS, "param is not string");
            return;
        }

        String json = (String) param;
        if (TextUtils.isEmpty(json)) {
            callbackOnError(callback, BaseConstants.ERR_INVALID_PARAMETERS, "param is empty");
            return;
        }

        try {
            JSONObject jsonObject = new JSONObject(json);
            SDKConfig.PacketRetryInfo info = new SDKConfig.PacketRetryInfo();

            if (jsonObject.has("maxRetryCount")) {
                info.maxRetryCount = jsonObject.optInt("maxRetryCount");
            }

            if (jsonObject.has("packetRequestTimeout")) {
                info.packetRequestTimeout = jsonObject.optInt("packetRequestTimeout");
            }

            BaseManager.getInstance().setPacketRetryInfo(info);

            callbackOnSuccess(callback, null);
        } catch (JSONException e) {
            callbackOnError(callback, BaseConstants.ERR_INVALID_PARAMETERS, "convert param to json failed");
            e.printStackTrace();
        }
    }

    private void setCosSaveRegion(Object param, V2TIMValueCallback<Object> callback) {
        if (null == param || !(param instanceof String)) {
            callbackOnError(callback, BaseConstants.ERR_INVALID_PARAMETERS, "param is not string");
            return;
        }

        String json = (String) param;
        if (TextUtils.isEmpty(json)) {
            callbackOnError(callback, BaseConstants.ERR_INVALID_PARAMETERS, "param is empty");
            return;
        }

        try {
            JSONObject jsonObject = new JSONObject(json);
            String conversationID = jsonObject.optString("conversationID");
            String cosSaveRegion = jsonObject.optString("cosSaveRegion");
            if (TextUtils.isEmpty(conversationID) || TextUtils.isEmpty(cosSaveRegion)) {
                callbackOnError(callback, BaseConstants.ERR_INVALID_PARAMETERS, "invalid param");
                return;
            }

            ConversationKey conversationKey =
                V2TIMConversationManagerImpl.getInstance().getConversationKey(conversationID);
            ConversationManager.getInstance().setCosSaveRegionForConversation(
                conversationKey, cosSaveRegion, new IMCallback(callback) {
                    @Override
                    public void success(Object data) {
                        super.success(data);
                    }

                    @Override
                    public void fail(int code, String errorMessage) {
                        super.fail(code, errorMessage);
                    }
                });

        } catch (JSONException e) {
            e.printStackTrace();
        }
    }

    private void callbackOnError(V2TIMValueCallback<Object> callback, int code, String desc) {
        if (callback != null) {
            callback.onError(code, desc);
        }
    }

    private void callbackOnSuccess(V2TIMValueCallback<Object> callback, Object result) {
        if (callback != null) {
            callback.onSuccess(result);
        }
    }

    private void setLibraryPath(Object param, V2TIMValueCallback<Object> callback) {
        if (!(param instanceof String)) {
            callbackOnError(callback, BaseConstants.ERR_INVALID_PARAMETERS, "param is not string");
            return;
        }
        String path = (String) param;
        boolean isSuccess = BaseManager.getInstance().setLibraryPath(path);
        if (isSuccess) {
            callbackOnSuccess(callback, null);
        } else {
            callbackOnError(callback, BaseConstants.ERR_INVALID_PARAMETERS, "system load so library failed");
        }
    }

    private void downloadLibrary(Object param, V2TIMValueCallback<Object> callback) {
        if (param == null || !(param instanceof Context)) {
            callbackOnError(callback, BaseConstants.ERR_INVALID_PARAMETERS, "param is not context");
        }

        BaseManager.getInstance().downloadLibrary((Context) param, new IMCallback(callback) {
            @Override
            public void success(Object data) {
                super.success(data);
            }

            @Override
            public void fail(int code, String errorMessage) {
                super.fail(code, errorMessage);
            }
        });
    }

    private void initLocalStorage(Object param, V2TIMValueCallback<Object> callback) {
        if (!(param instanceof String)) {
            callbackOnError(callback, BaseConstants.ERR_INVALID_PARAMETERS, "param is not string");
            return;
        }

        String userID = (String) param;
        if (TextUtils.isEmpty(userID)) {
            callbackOnError(callback, BaseConstants.ERR_INVALID_PARAMETERS, "invalid userID");
            return;
        }

        BaseManager.getInstance().initLocalStorage(userID, new IMCallback(callback) {
            @Override
            public void success(Object data) {
                super.success(data);
            }

            @Override
            public void fail(int code, String errorMessage) {
                super.fail(code, errorMessage);
            }
        });
    }

    private void setCustomServerInfo(Object param, V2TIMValueCallback<Object> callback) {
        if (null == param || !(param instanceof String)) {
            callbackOnError(callback, BaseConstants.ERR_INVALID_PARAMETERS, "param is not string");
            return;
        }

        String json = (String) param;
        if (TextUtils.isEmpty(json)) {
            callbackOnError(callback, BaseConstants.ERR_INVALID_PARAMETERS, "param is empty");
            return;
        }

        CustomServerInfo customServerInfo = new CustomServerInfo();
        try {
            // 仅长连接支持 Quic 通道
            JSONObject jsonObject = new JSONObject(json);
            JSONArray longconnectionAddressList = jsonObject.optJSONArray("longconnectionAddressList");
            if (longconnectionAddressList != null && longconnectionAddressList.length() > 0) {
                ArrayList list = new ArrayList();
                for (int i = 0; i < longconnectionAddressList.length(); i++) {
                    JSONObject address = longconnectionAddressList.getJSONObject(i);
                    CustomServerInfo.ServerAddress serverAddress = new CustomServerInfo.ServerAddress();
                    serverAddress.ip = address.optString("ip");
                    serverAddress.port = address.optInt("port");
                    serverAddress.isIPv6 = (address.has("isIPv6") ? address.optBoolean("isIPv6") : false);
                    serverAddress.isQuic = (address.has("isQuic") ? address.optBoolean("isQuic") : false);
                    serverAddress.isTLS = (address.has("isTLS") ? address.optBoolean("isTLS") : false);

                    list.add(serverAddress);
                }
                customServerInfo.longconnectionAddressList = list;
            }

            JSONArray shortconnectionAddressList = jsonObject.optJSONArray("shortconnectionAddressList");
            if (shortconnectionAddressList != null && shortconnectionAddressList.length() > 0) {
                ArrayList list = new ArrayList();
                for (int i = 0; i < shortconnectionAddressList.length(); i++) {
                    JSONObject address = shortconnectionAddressList.getJSONObject(i);
                    CustomServerInfo.ServerAddress serverAddress = new CustomServerInfo.ServerAddress();
                    serverAddress.ip = address.optString("ip");
                    serverAddress.port = address.optInt("port");
                    serverAddress.isIPv6 = (address.has("isIPv6") ? address.optBoolean("isIPv6") : false);
                    serverAddress.isTLS = (address.has("isTLS") ? address.optBoolean("isTLS") : false);

                    list.add(serverAddress);
                }
                customServerInfo.shortconnectionAddressList = list;
            }

            customServerInfo.serverPublicKey = jsonObject.optString("serverPublicKey");

            BaseManager.getInstance().setCustomServerInfo(customServerInfo);

            callbackOnSuccess(callback, null);
        } catch (JSONException e) {
            callbackOnError(callback, BaseConstants.ERR_INVALID_PARAMETERS, "convert param to json failed");
            e.printStackTrace();
        }
    }

    private void setQuicChannelEnabled(Object param, V2TIMValueCallback<Object> callback) {
        if (null == param || !(param instanceof String)) {
            callbackOnError(callback, BaseConstants.ERR_INVALID_PARAMETERS, "param is not string");
            return;
        }

        String json = (String) param;
        if (TextUtils.isEmpty(json)) {
            callbackOnError(callback, BaseConstants.ERR_INVALID_PARAMETERS, "param is empty");
            return;
        }

        try {
            JSONObject jsonObject = new JSONObject(json);
            boolean enableQuic = jsonObject.optBoolean("enableQuic");

            BaseManager.getInstance().setQuicEnabled(enableQuic);

            callbackOnSuccess(callback, null);
        } catch (JSONException e) {
            callbackOnError(callback, BaseConstants.ERR_INVALID_PARAMETERS, "convert param to json failed");
            e.printStackTrace();
        }
    }

    private void setProxyInfo(Object param, V2TIMValueCallback<Object> callback) {
        if (null == param || !(param instanceof String)) {
            callbackOnError(callback, BaseConstants.ERR_INVALID_PARAMETERS, "param is not string");
            return;
        }

        String json = (String) param;
        if (TextUtils.isEmpty(json)) {
            callbackOnError(callback, BaseConstants.ERR_INVALID_PARAMETERS, "param is empty");
            return;
        }

        try {
            JSONObject jsonObject = new JSONObject(json);
            if (!jsonObject.has("proxyType")) {
                callbackOnError(callback, BaseConstants.ERR_INVALID_PARAMETERS, "proxyType must be set");
                return;
            }

            // http 和 socks5 代理，需要设置 host 和 port，user_name 和 password 可选
            // pac 代理，不需要设置 host 和 port，user_name 和 password 可选
            // 因此，host 和 port，要么都设置，要么都不设置
            if ((jsonObject.has("proxyHost") && !jsonObject.has("proxyPort"))
                || (!jsonObject.has("proxyHost") && jsonObject.has("proxyPort"))) {
                callbackOnError(callback, BaseConstants.ERR_INVALID_PARAMETERS,
                    "proxyHost and proxyPort must be set together if need");
                return;
            }

            SDKConfig.ProxyInfo proxyInfo = new SDKConfig.ProxyInfo();
            proxyInfo.proxyType = jsonObject.optInt("proxyType");

            if (jsonObject.has("proxyHost") && jsonObject.has("proxyPort")) {
                proxyInfo.proxyHost = jsonObject.optString("proxyHost");
                proxyInfo.proxyPort = jsonObject.optInt("proxyPort");
            }

            if (jsonObject.has("proxyUsername")) {
                proxyInfo.proxyUsername = jsonObject.optString("proxyUsername");
            }

            if (jsonObject.has("proxyPassword")) {
                proxyInfo.proxyPassword = jsonObject.optString("proxyPassword");
            }

            BaseManager.getInstance().setProxyInfo(proxyInfo);

            callbackOnSuccess(callback, null);
        } catch (JSONException e) {
            callbackOnError(callback, BaseConstants.ERR_INVALID_PARAMETERS, "convert param to json failed");
            e.printStackTrace();
        }
    }

    private void setUIPlatform(Object param, V2TIMValueCallback<Object> callback) {
        if (null == param) {
            callbackOnError(callback, BaseConstants.ERR_INVALID_PARAMETERS, "param is null");
            return;
        }

        String stringUIPlatform = "";
        int numberUIPlatform = 0;
        if (param instanceof String) {
            stringUIPlatform = (String) param;
        } else if (param instanceof Integer) {
            numberUIPlatform = (Integer) param;
        } else {
            callbackOnError(callback, BaseConstants.ERR_INVALID_PARAMETERS, "param is invalid");
        }

        BaseManager.getInstance().setCustomUIPlatform(stringUIPlatform, numberUIPlatform);

        callbackOnSuccess(callback, null);
    }

    private void setTestEnvironment(Object param, V2TIMValueCallback<Object> callback) {
        if (param != null && param instanceof Boolean) {
            BaseManager.getInstance().setTestEnvironment(((Boolean) param).booleanValue());
        }

        callbackOnSuccess(callback, null);
    }

    private void setIPv6Prior(Object param, V2TIMValueCallback<Object> callback) {
        if (param != null && param instanceof Boolean) {
            BaseManager.getInstance().setIPv6Prior(((Boolean) param).booleanValue());
        }

        callbackOnSuccess(callback, null);
    }

    private void setApplicationID(Object param, V2TIMValueCallback<Object> callback) {
        if (param != null && param instanceof Integer) {
            BaseManager.getInstance().setApplicationID(((Integer) param).intValue());
        }

        callbackOnSuccess(callback, null);
    }

    private void disableHttpRequest(Object param, V2TIMValueCallback<Object> callback) {
        if (param != null && param instanceof Boolean) {
            BaseManager.getInstance().disableHttpRequest(((Boolean) param).booleanValue());
        }

        callbackOnSuccess(callback, null);
    }

    private void getLoginAccountType(Object param, V2TIMValueCallback<Object> callback) {
        BaseManager.getInstance().getLoginAccountType(new IMCallback(callback) {
            @Override
            public void success(Object data) {
                super.success(data);
            }

            @Override
            public void fail(int code, String errorMessage) {
                super.fail(code, errorMessage);
            }
        });
    }

    private void findMergerMessages(Object param, V2TIMValueCallback<Object> callback) {
        if (null == param || !(param instanceof String)) {
            callbackOnError(callback, BaseConstants.ERR_INVALID_PARAMETERS, "param is not string");
            return;
        }

        String json = (String) param;
        if (TextUtils.isEmpty(json)) {
            callbackOnError(callback, BaseConstants.ERR_INVALID_PARAMETERS, "param is empty");
            return;
        }

        try {
            JSONObject jsonObject = new JSONObject(json);
            String messageID = jsonObject.optString("messageID");
            if (TextUtils.isEmpty(messageID)) {
                callbackOnError(callback, BaseConstants.ERR_INVALID_PARAMETERS, "messageID is empty");
                return;
            }

            JSONArray mergerMessageIDList = jsonObject.optJSONArray("mergerMessageIDList");
            List<String> mergerMessageIDItemList = new ArrayList<String>();
            if (mergerMessageIDList != null && mergerMessageIDList.length() > 0) {
                for (int i = 0; i < mergerMessageIDList.length(); i++) {
                    String mergerMessageID = mergerMessageIDList.getString(i);
                    if (null == mergerMessageID || !(mergerMessageID instanceof String)) {
                        callbackOnError(callback, BaseConstants.ERR_INVALID_PARAMETERS, "mergerMessageID is empty");
                        return;
                    }
                    mergerMessageIDItemList.add(mergerMessageID);
                }
            }
            if (0 == mergerMessageIDItemList.size()) {
                callbackOnError(callback, BaseConstants.ERR_INVALID_PARAMETERS, "mergerMessageIDList is empty");
                return;
            }

            V2TIMValueCallback<List<Message>> v2TIMValueCallback = new V2TIMValueCallback<List<Message>>() {
                @Override
                public void onSuccess(List<Message> messageList) {
                    List<V2TIMMessage> v2TIMMessageList = new ArrayList<>();

                    for (Message message : messageList) {
                        V2TIMMessage v2TIMMessage = new V2TIMMessage();
                        v2TIMMessage.setMessage(message);

                        v2TIMMessageList.add(v2TIMMessage);
                    }

                    if (callback != null) {
                        callback.onSuccess(v2TIMMessageList);
                    }
                }

                @Override
                public void onError(int code, String desc) {
                    if (callback != null) {
                        callback.onError(code, desc);
                    }
                }
            };
            MessageCenter.getInstance().findRelayMessageList(
                messageID, mergerMessageIDItemList, new IMCallback<List<Message>>(v2TIMValueCallback) {
                    @Override
                    public void success(List<Message> data) {
                        super.success(data);
                    }

                    @Override
                    public void fail(int code, String errorMessage) {
                        super.fail(code, errorMessage);
                    }
                });
        } catch (JSONException e) {
            callbackOnError(callback, BaseConstants.ERR_INVALID_PARAMETERS, "convert param to json failed");
            e.printStackTrace();
        }
    }

    private void getBriefGroupMemberList(Object param, V2TIMValueCallback<Object> callback) {
        if (null == param || !(param instanceof String)) {
            callbackOnError(callback, BaseConstants.ERR_INVALID_PARAMETERS, "param is not string");
            return;
        }

        String groupID = (String) param;

        V2TIMValueCallback<List<GroupMemberInfo>> v2Callback = new V2TIMValueCallback<List<GroupMemberInfo>>() {
            @Override
            public void onSuccess(List<GroupMemberInfo> memberInfoList) {
                if (callback != null) {
                    List<V2TIMGroupMemberFullInfo> resultList = new ArrayList<>();
                    for (GroupMemberInfo item : memberInfoList) {
                        V2TIMGroupMemberFullInfo member = new V2TIMGroupMemberFullInfo();
                        member.setGroupMemberInfo(item);
                        resultList.add(member);
                    }
                    callback.onSuccess(resultList);
                }
            }

            @Override
            public void onError(int code, String desc) {
                if (callback != null) {
                    callback.onError(code, desc);
                }
            }
        };

        BaseManager.getInstance().getBriefGroupMemberList(groupID, new IMCallback<List<GroupMemberInfo>>(v2Callback) {
            @Override
            public void success(List<GroupMemberInfo> data) {
                super.success(data);
            }

            @Override
            public void fail(int code, String errorMessage) {
                super.fail(code, errorMessage);
            }
        });
    }

    private void reportOfflinePushEvent(Object param, final V2TIMValueCallback<Object> callback) {
        if (null == param || !(param instanceof String)) {
            callbackOnError(callback, BaseConstants.ERR_INVALID_PARAMETERS, "param is not string");
            return;
        }

        String json = (String) param;
        if (TextUtils.isEmpty(json)) {
            callbackOnError(callback, BaseConstants.ERR_INVALID_PARAMETERS, "param is empty");
            return;
        }

        List<OfflinePushEventItem> reportEventItemList = new ArrayList<>();
        try {
            JSONObject jsonObject = new JSONObject(json);
            JSONArray reportEventList = jsonObject.optJSONArray("eventList");
            if (reportEventList != null && reportEventList.length() > 0) {
                for (int i = 0; i < reportEventList.length(); i++) {
                    JSONObject eventItemJson = reportEventList.getJSONObject(i);
                    OfflinePushEventItem eventItem = new OfflinePushEventItem();
                    eventItem.setEventType(eventItemJson.optInt("type"));
                    eventItem.setEventTime(eventItemJson.getLong("time"));
                    eventItem.setPushId(eventItemJson.optString("pushId"));
                    reportEventItemList.add(eventItem);
                }
            }

            OfflinePushManager.getInstance().reportOfflinePushEvent(reportEventItemList, new IMCallback(callback) {
                @Override
                public void success(Object data) {
                    super.success(data);
                }

                @Override
                public void fail(int code, String errorMessage) {
                    super.fail(code, errorMessage);
                }
            });
        } catch (JSONException e) {
            callbackOnError(callback, BaseConstants.ERR_INVALID_PARAMETERS, "convert param to json failed");
            e.printStackTrace();
        }
    }

    private void setOfflinePushInfo(Object param, final V2TIMValueCallback<Object> callback) {
        if (null == param || !(param instanceof String)) {
            callbackOnError(callback, BaseConstants.ERR_INVALID_PARAMETERS, "param is not string");
            return;
        }

        String json = (String) param;
        if (TextUtils.isEmpty(json)) {
            callbackOnError(callback, BaseConstants.ERR_INVALID_PARAMETERS, "param is empty");
            return;
        }

        try {
            JSONObject jsonObject = new JSONObject(json);
            int status = jsonObject.optInt("notification_bar_state");
            String version = jsonObject.optString("tim_push_plugin_version");
            String systemLanguage = jsonObject.optString("system_language");

            // 开关状态：1，开启； 2，关闭； 0，未采集
            if (status < 0 || status > 2) {
                status = 0;
            }

            OfflinePushManager.getInstance().setOfflinePushInfo(version, status, systemLanguage);
            callbackOnSuccess(callback, null);

        } catch (JSONException e) {
            callbackOnError(callback, BaseConstants.ERR_INVALID_PARAMETERS, "convert param to json failed");
            e.printStackTrace();
        }
    }

    private void pushLogin(Object param, final V2TIMValueCallback<Object> callback) {
        if (null == param || !(param instanceof String)) {
            callbackOnError(callback, BaseConstants.ERR_INVALID_PARAMETERS, "param is not string");
            return;
        }

        String json = (String) param;
        if (TextUtils.isEmpty(json)) {
            callbackOnError(callback, BaseConstants.ERR_INVALID_PARAMETERS, "param is empty");
            return;
        }

        try {
            JSONObject jsonObject = new JSONObject(json);
            String userID = jsonObject.optString("userID");
            String appKey = jsonObject.optString("appKey");

            BaseManager.getInstance().login(userID, appKey, new IMCallback(callback) {
                @Override
                public void success(Object data) {
                    super.success(data);
                }

                @Override
                public void fail(int code, String errorMessage) {
                    super.fail(code, errorMessage);
                }
            }, BaseManager.ACCOUNT_TYPE_PUSH);
        } catch (JSONException e) {
            callbackOnError(callback, BaseConstants.ERR_INVALID_PARAMETERS, "convert param to json failed");
            e.printStackTrace();
        }
    }

    private void reportPushSDKEvent(Object param, final V2TIMValueCallback<Object> callback) {
        if (null == param || !(param instanceof String)) {
            callbackOnError(callback, BaseConstants.ERR_INVALID_PARAMETERS, "param is not string");
            return;
        }

        String json = (String) param;
        if (TextUtils.isEmpty(json)) {
            callbackOnError(callback, BaseConstants.ERR_INVALID_PARAMETERS, "param is empty");
            return;
        }

        try {
            JSONObject jsonObject = new JSONObject(json);
            long eventID = jsonObject.optLong("event_id");
            long eventCode = jsonObject.optLong("event_code");
            long eventResult = jsonObject.optLong("event_result");
            String eventMessage = jsonObject.optString("event_message");
            String moreMessage = jsonObject.optString("more_message");
            String extensionMessage = jsonObject.optString("extension_message");

            BaseManager.getInstance().reportPushSDKEvent(
                eventID, eventCode, eventResult, eventMessage, moreMessage, extensionMessage, new IMCallback(callback) {
                    @Override
                    public void success(Object data) {
                        super.success(data);
                    }

                    @Override
                    public void fail(int code, String errorMessage) {
                        super.fail(code, errorMessage);
                    }
                });
        } catch (JSONException e) {
            callbackOnError(callback, BaseConstants.ERR_INVALID_PARAMETERS, "convert param to json failed");
            e.printStackTrace();
        }
    }

    private void setCustomLoginInfo(Object param, V2TIMValueCallback<Object> callback) {
        if (null == param || !(param instanceof String)) {
            callbackOnError(callback, BaseConstants.ERR_INVALID_PARAMETERS, "param is not string");
            return;
        }

        String json = (String) param;
        if (TextUtils.isEmpty(json)) {
            callbackOnError(callback, BaseConstants.ERR_INVALID_PARAMETERS, "param is empty");
            return;
        }

        try {
            JSONObject jsonObject = new JSONObject(json);
            if (!jsonObject.has("userID") || !jsonObject.has("customLoginInfo")) {
                callbackOnError(callback, BaseConstants.ERR_INVALID_PARAMETERS, "invalid param");
                return;
            }

            String userID = jsonObject.optString("userID");
            String customLoginInfo = jsonObject.optString("customLoginInfo");
            BaseManager.getInstance().setCustomLoginInfo(userID, customLoginInfo, new IMCallback(callback) {
                @Override
                public void success(Object data) {
                    super.success(data);
                }

                @Override
                public void fail(int code, String errorMessage) {
                    super.fail(code, errorMessage);
                }
            });
        } catch (JSONException e) {
            callbackOnError(callback, BaseConstants.ERR_INVALID_PARAMETERS, "convert param to json failed");
            e.printStackTrace();
        }
    }

    private void cancelSendingMessage(Object param, V2TIMValueCallback<Object> callback) {
        if (null == param || !(param instanceof String)) {
            callbackOnError(callback, BaseConstants.ERR_INVALID_PARAMETERS, "param is not string");
            return;
        }

        String messageID = (String) param;
        MessageCenter.getInstance().cancelSendingMessage(messageID, new IMCallback(callback) {
            @Override
            public void success(Object data) {
                super.success(data);
            }

            @Override
            public void fail(int code, String errorMessage) {
                super.fail(code, errorMessage);
            }
        });
    }

    private void getMessageSource(Object param, V2TIMValueCallback<Object> callback) {
        if (null == param || !(param instanceof V2TIMMessage)) {
            callbackOnError(callback, BaseConstants.ERR_INVALID_PARAMETERS, "param invalid");
            return;
        }

        V2TIMMessage v2TIMMessage = (V2TIMMessage) param;
        Message message = v2TIMMessage.getMessage();
        int messageSource = 0;
        if (message != null) {
            messageSource = message.getMessageSource();
        } else {
            messageSource = 0;
        }

        callbackOnSuccess(callback, new Integer(messageSource));
    }

    private void getGroupMessageReceipts(Object param, V2TIMValueCallback<Object> callback) {
        if (null == param || !(param instanceof String)) {
            callbackOnError(callback, BaseConstants.ERR_INVALID_PARAMETERS, "param is not string");
            return;
        }

        String json = (String) param;
        if (TextUtils.isEmpty(json)) {
            callbackOnError(callback, BaseConstants.ERR_INVALID_PARAMETERS, "param is empty");
            return;
        }

        Map<Integer, List<String>> requestMessageReadInfo = new HashMap<>();
        String groupID = "";
        try {
            JSONObject jsonObject = new JSONObject(json);
            groupID = jsonObject.optString("groupID");
            if (TextUtils.isEmpty(groupID)) {
                callbackOnError(callback, BaseConstants.ERR_INVALID_PARAMETERS, "groupID is empty");
                return;
            }

            JSONArray jsonArray = jsonObject.optJSONArray("requestList");
            for (int i = 0; i < jsonArray.length(); i++) {
                JSONObject item = jsonArray.getJSONObject(i);

                if (!item.has("messageSequence") || !item.has("memberIDList")) {
                    callbackOnError(callback, BaseConstants.ERR_INVALID_PARAMETERS, "invalid param");
                    return;
                }

                int messageSequence = item.getInt("messageSequence");
                JSONArray memberIdArray = item.getJSONArray("memberIDList");
                List<String> memberIdList = new ArrayList<>();

                for (int j = 0; j < memberIdArray.length(); j++) {
                    memberIdList.add(memberIdArray.getString(j));
                }

                requestMessageReadInfo.put(messageSequence, memberIdList);
            }
        } catch (JSONException e) {
            callbackOnError(callback, BaseConstants.ERR_INVALID_PARAMETERS, "convert param to json failed");
            e.printStackTrace();
        }

        V2TIMValueCallback<Map<Integer, Map<Integer, List<String>>>> v2Callback =
            new V2TIMValueCallback<Map<Integer, Map<Integer, List<String>>>>() {
                @Override
                public void onSuccess(Map<Integer, Map<Integer, List<String>>> readInfo) {
                    if (callback != null) {
                        try {
                            JSONArray jsonArray = new JSONArray();

                            for (Map.Entry<Integer, Map<Integer, List<String>>> entry : readInfo.entrySet()) {
                                Map<Integer, List<String>> innerMap = entry.getValue();
                                if (innerMap == null || innerMap.isEmpty()) {
                                    continue;
                                }

                                Iterator<Map.Entry<Integer, List<String>>> iterator = innerMap.entrySet().iterator();
                                if (!iterator.hasNext()) {
                                    continue;
                                }

                                Map.Entry<Integer, List<String>> innerEntry = iterator.next();

                                List<String> memberIds = innerEntry.getValue();
                                JSONArray readInfoArray = new JSONArray();
                                for (String userId : memberIds) {
                                    JSONObject readInfoObj = new JSONObject();
                                    readInfoObj.put("memberID", userId);
                                    readInfoArray.put(readInfoObj);
                                }

                                int messageSequence = entry.getKey();
                                int memberErrorCode = innerEntry.getKey();
                                JSONObject jsonObject = new JSONObject();
                                jsonObject.put("messageSequence", messageSequence);
                                jsonObject.put("errorCode", memberErrorCode);
                                jsonObject.put("readInfoList", readInfoArray);

                                jsonArray.put(jsonObject);
                            }
                            callback.onSuccess(jsonArray);
                        } catch (JSONException e) {
                            callbackOnError(
                                callback, BaseConstants.ERR_INVALID_PARAMETERS, "convert param to json failed");
                            e.printStackTrace();
                        }
                    }
                }

                @Override
                public void onError(int code, String desc) {
                    if (callback != null) {
                        callback.onError(code, desc);
                    }
                }
            };

        BaseManager.getInstance().getGroupMessageReceipt(
            groupID, requestMessageReadInfo, new IMCallback<Map<Integer, Map<Integer, List<String>>>>(v2Callback) {
                @Override
                public void success(Map<Integer, Map<Integer, List<String>>> data) {
                    super.success(data);
                }

                @Override
                public void fail(int code, String errorMessage) {
                    super.fail(code, errorMessage);
                }
            });
    }

    private void getPushMessageID(Object param, V2TIMValueCallback<Object> callback) {
        if (null == param || !(param instanceof V2TIMMessage)) {
            callbackOnError(callback, BaseConstants.ERR_INVALID_PARAMETERS, "param invalid");
            return;
        }

        V2TIMMessage v2TIMMessage = (V2TIMMessage) param;
        Message message = v2TIMMessage.getMessage();
        String pushMessageID = "";
        if (message != null) {
            pushMessageID = message.getPushMessageID();
        }

        callbackOnSuccess(callback, pushMessageID);
    }

    private Object createImageMessage(Object param) {
        if (null == param || !(param instanceof String)) {
            IMLog.e(TAG, "param is not string");
            return null;
        }

        String json = (String) param;
        if (TextUtils.isEmpty(json)) {
            IMLog.e(TAG, "param is empty");
            return null;
        }

        try {
            JSONObject jsonObject = new JSONObject(json);

            if (!jsonObject.has("originImageUrl") || !jsonObject.has("originImageUuid")) {
                IMLog.e(TAG, "originImageUrl/originImageUuid is empty");
                return null;
            }
            String originImageUrl = jsonObject.optString("originImageUrl", "");
            String originImageUuid = jsonObject.optString("originImageUuid", "");

            String filePath = jsonObject.optString("filePath", "");
            int originFileSize = jsonObject.optInt("originFileSize", 0);
            int originImageWidth = jsonObject.optInt("originImageWidth", 0);
            int originImageHeight = jsonObject.optInt("originImageHeight", 0);
            ImageElement imageElement = new ImageElement();
            imageElement.setOriginImageFilePath(filePath);
            imageElement.setOriginImageUrl(originImageUrl);
            imageElement.setOriginImageUUID(originImageUuid);
            imageElement.setOriginImageFileSize(originFileSize);
            imageElement.setOriginImageWidth(originImageWidth);
            imageElement.setOriginImageHeight(originImageHeight);

            String thumbImageUrl = jsonObject.optString("thumbImageUrl", "");
            String thumbImageUuid = jsonObject.optString("thumbImageUuid", "");
            int thumbFileSize = jsonObject.optInt("thumbFileSize", 0);
            int thumbImageWidth = jsonObject.optInt("thumbImageWidth", 0);
            int thumbImageHeight = jsonObject.optInt("thumbImageHeight", 0);

            imageElement.setThumbImageUrl(thumbImageUrl);
            imageElement.setThumbImageUUID(thumbImageUuid);
            imageElement.setThumbImageFileSize(thumbFileSize);
            imageElement.setThumbImageWidth(thumbImageWidth);
            imageElement.setThumbImageHeight(thumbImageHeight);

            String largeImageUrl = jsonObject.optString("largeImageUrl", "");
            String largeImageUuid = jsonObject.optString("largeImageUuid", "");
            int largeFileSize = jsonObject.optInt("largeFileSize", 0);
            int largeImageWidth = jsonObject.optInt("largeImageWidth", 0);
            int largeImageHeight = jsonObject.optInt("largeImageHeight", 0);

            imageElement.setLargeImageUrl(largeImageUrl);
            imageElement.setLargeImageUUID(largeImageUuid);
            imageElement.setLargeImageFileSize(largeFileSize);
            imageElement.setLargeImageWidth(largeImageWidth);
            imageElement.setLargeImageHeight(largeImageHeight);

            Message msg = new Message();
            msg.addElement(imageElement);

            V2TIMMessage v2TIMMessage = new V2TIMMessage();
            v2TIMMessage.setMessage(msg);
            return v2TIMMessage;
        } catch (JSONException e) {
            IMLog.e(TAG, "convert param to json failed");
        }

        return null;
    }

    private Object createSoundMessage(Object param) {
        if (null == param || !(param instanceof String)) {
            IMLog.e(TAG, "param is not string");
            return null;
        }

        String json = (String) param;
        if (TextUtils.isEmpty(json)) {
            IMLog.e(TAG, "param is empty");
            return null;
        }

        try {
            JSONObject jsonObject = new JSONObject(json);

            if (!jsonObject.has("url") || !jsonObject.has("uuid") || !jsonObject.has("duration")) {
                IMLog.e(TAG, "url/uuid/duration is empty");
                return null;
            }

            String url = jsonObject.optString("url", "");
            String uuid = jsonObject.optString("uuid", "");
            int duration = jsonObject.optInt("duration", 0);

            String filePath = jsonObject.optString("filePath", "");
            int fileSize = jsonObject.optInt("fileSize", 0);
            SoundElement soundElement = new SoundElement();
            soundElement.setSoundFilePath(filePath);
            soundElement.setSoundDownloadUrl(url);
            soundElement.setSoundUUID(uuid);
            soundElement.setSoundDuration(duration);
            soundElement.setSoundFileSize(fileSize);

            Message msg = new Message();
            msg.addElement(soundElement);

            V2TIMMessage v2TIMMessage = new V2TIMMessage();
            v2TIMMessage.setMessage(msg);
            return v2TIMMessage;
        } catch (JSONException e) {
            IMLog.e(TAG, "convert param to json failed");
        }

        return null;
    }

    private Object createVideoMessage(Object param) {
        if (null == param || !(param instanceof String)) {
            IMLog.e(TAG, "param is not string");
            return null;
        }

        String json = (String) param;
        if (TextUtils.isEmpty(json)) {
            IMLog.e(TAG, "param is empty");
            return null;
        }

        try {
            JSONObject jsonObject = new JSONObject(json);

            if (!jsonObject.has("videoUrl") || !jsonObject.has("videoUuid") || !jsonObject.has("duration")
                || !jsonObject.has("snapshotUrl") || !jsonObject.has("snapshotUuid")) {
                IMLog.e(TAG, "videoUrl/videoUuid/duration/snapshotUrl/snapshotUuid is empty");
                return null;
            }

            String videoUrl = jsonObject.optString("videoUrl", "");
            String videoUuid = jsonObject.optString("videoUuid", "");
            int duration = jsonObject.optInt("duration", 0);
            String snapshotUrl = jsonObject.optString("snapshotUrl", "");
            String snapshotUuid = jsonObject.optString("snapshotUuid", "");

            String videoFilePath = jsonObject.optString("videoFilePath", "");
            String videoFileType = jsonObject.optString("videoType", "");
            int videoFileSize = jsonObject.optInt("videoFileSize", 0);
            int snapshotWidth = jsonObject.optInt("snapshotWidth", 0);
            int snapshotHeight = jsonObject.optInt("snapshotHeight", 0);
            String snapshotFilePath = jsonObject.optString("snapshotFilePath", "");
            int snapshotFileSize = jsonObject.optInt("snapshotFileSize", 0);
            VideoElement videoElement = new VideoElement();
            videoElement.setVideoFilePath(videoFilePath);
            videoElement.setVideoType(videoFileType);
            videoElement.setVideoDownloadUrl(videoUrl);
            videoElement.setVideoUUID(videoUuid);
            videoElement.setVideoDuration(duration);
            videoElement.setSnapshotFilePath(snapshotFilePath);
            videoElement.setSnapshotDownloadUrl(snapshotUrl);
            videoElement.setSnapshotUUID(snapshotUuid);
            videoElement.setVideoFileSize(videoFileSize);
            videoElement.setSnapshotWidth(snapshotWidth);
            videoElement.setSnapshotHeight(snapshotHeight);
            videoElement.setSnapshotFileSize(snapshotFileSize);

            Message msg = new Message();
            msg.addElement(videoElement);

            V2TIMMessage v2TIMMessage = new V2TIMMessage();
            v2TIMMessage.setMessage(msg);
            return v2TIMMessage;
        } catch (JSONException e) {
            IMLog.e(TAG, "convert param to json failed");
        }

        return null;
    }

    private Object createFileMessage(Object param) {
        if (null == param || !(param instanceof String)) {
            IMLog.e(TAG, "param is not string");
            return null;
        }

        String json = (String) param;
        if (TextUtils.isEmpty(json)) {
            IMLog.e(TAG, "param is empty");
            return null;
        }

        try {
            JSONObject jsonObject = new JSONObject(json);

            if (!jsonObject.has("url") || !jsonObject.has("uuid")) {
                IMLog.e(TAG, "url/uuid is empty");
                return null;
            }
            String url = jsonObject.optString("url", "");
            String uuid = jsonObject.optString("uuid", "");

            String filePath = jsonObject.optString("filePath", "");
            int fileSize = jsonObject.optInt("fileSize", 0);
            String fileName = jsonObject.optString("fileName", "");

            FileElement fileElement = new FileElement();
            fileElement.setFilePath(filePath);
            fileElement.setFileDownloadUrl(url);
            fileElement.setFileUUID(uuid);
            fileElement.setFileSize(fileSize);
            fileElement.setFileName(fileName);

            Message msg = new Message();
            msg.addElement(fileElement);

            V2TIMMessage v2TIMMessage = new V2TIMMessage();
            v2TIMMessage.setMessage(msg);
            return v2TIMMessage;
        } catch (JSONException e) {
            IMLog.e(TAG, "convert param to json failed");
        }

        return null;
    }

    private void getOfficialAccountList(Object param, V2TIMValueCallback<Object> callback) {
        if (null == param || !(param instanceof String)) {
            callbackOnError(callback, BaseConstants.ERR_INVALID_PARAMETERS, "param is not string");
            return;
        }

        String json = (String) param;
        if (TextUtils.isEmpty(json)) {
            callbackOnError(callback, BaseConstants.ERR_INVALID_PARAMETERS, "param is empty");
            return;
        }

        int count = 0;
        long offset = 0;
        try {
            JSONObject jsonObject = new JSONObject(json);
            count = jsonObject.optInt("count");
            offset = jsonObject.optLong("offset");
        } catch (JSONException e) {
            e.printStackTrace();
        }

        V2TIMValueCallback<HashMap<String, Object>> v2TIMCallback = new V2TIMValueCallback<HashMap<String, Object>>() {
            @Override
            public void onSuccess(HashMap<String, Object> officialAccountGetResult) {
                if (callback != null) {
                    if (officialAccountGetResult.get("official_account_list") != null) {
                        List<OfficialAccountInfo> infoList =
                            (List<OfficialAccountInfo>) officialAccountGetResult.get("official_account_list");
                        List<V2TIMOfficialAccountInfo> v2TIMOfficialAccountInfoList = new ArrayList<>();
                        for (OfficialAccountInfo info : infoList) {
                            V2TIMOfficialAccountInfo v2TIMOfficialAccountInfo = new V2TIMOfficialAccountInfo();
                            v2TIMOfficialAccountInfo.setOfficialAccountInfo(info);
                            v2TIMOfficialAccountInfoList.add(v2TIMOfficialAccountInfo);
                        }
                        officialAccountGetResult.put("official_account_list", v2TIMOfficialAccountInfoList);
                    } else {
                        officialAccountGetResult.put("official_account_list", new ArrayList<>());
                    }

                    callback.onSuccess(officialAccountGetResult);
                }
            }

            @Override
            public void onError(int code, String desc) {
                if (callback != null) {
                    callback.onError(code, desc);
                }
            }
        };

        RelationshipManager.getInstance().getOfficialAccountList(
            count, offset, new IMCallback<HashMap<String, Object>>(v2TIMCallback) {
                @Override
                public void success(HashMap<String, Object> data) {
                    super.success(data);
                }

                @Override
                public void fail(int code, String errorMessage) {
                    super.fail(code, errorMessage);
                }
            });
    }

    private void setForceRequestCloudMessage(Object param, V2TIMValueCallback<Object> callback) {
        if (param != null && param instanceof Boolean) {
            BaseManager.getInstance().setForceRequestCloudMessage(((Boolean) param).booleanValue());
        }

        callbackOnSuccess(callback, null);
    }

    private void uploadFile(Object param, final V2TIMValueCallback<Object> callback) {
        if (null == param || !(param instanceof String)) {
            callbackOnError(callback, BaseConstants.ERR_INVALID_PARAMETERS, "param is not string");
            return;
        }

        String json = (String) param;
        if (TextUtils.isEmpty(json)) {
            callbackOnError(callback, BaseConstants.ERR_INVALID_PARAMETERS, "param is empty");
            return;
        }

        try {
            JSONObject jsonObject = new JSONObject(json);
            if (!jsonObject.has("filePath")) {
                callbackOnError(callback, BaseConstants.ERR_INVALID_PARAMETERS, "filePath must be set");
                return;
            }

            String filePath = jsonObject.optString("filePath");
            int fileType = jsonObject.optInt("fileType", 0);

            if (TextUtils.isEmpty(filePath)) {
                callbackOnError(callback, BaseConstants.ERR_INVALID_PARAMETERS, "filePath is invalid");
                return;
            }

            BaseManager.getInstance().uploadFile(filePath, fileType, new IMCallback(callback) {
                @Override
                public void success(Object data) {
                    super.success(data);
                }

                @Override
                public void fail(int code, String errorMessage) {
                    super.fail(code, errorMessage);
                }
            });
        } catch (JSONException e) {
            callbackOnError(callback, BaseConstants.ERR_INVALID_PARAMETERS, "convert param to json failed");
            e.printStackTrace();
        }
    }

    private void convertVoiceToText(Object param, final V2TIMValueCallback<Object> callback) {
        if (null == param || !(param instanceof String)) {
            callbackOnError(callback, BaseConstants.ERR_INVALID_PARAMETERS, "param is not string");
            return;
        }

        String json = (String) param;
        if (TextUtils.isEmpty(json)) {
            callbackOnError(callback, BaseConstants.ERR_INVALID_PARAMETERS, "param is empty");
            return;
        }

        try {
            JSONObject jsonObject = new JSONObject(json);
            if (!jsonObject.has("url")) {
                callbackOnError(callback, BaseConstants.ERR_INVALID_PARAMETERS, "url must be set");
                return;
            }

            String url = jsonObject.optString("url");
            String language = jsonObject.optString("language", "");

            if (TextUtils.isEmpty(url)) {
                callbackOnError(callback, BaseConstants.ERR_INVALID_PARAMETERS, "url is invalid");
                return;
            }

            MessageCenter.getInstance().convertVoiceToText(url, language, new IMCallback(callback) {
                @Override
                public void success(Object data) {
                    super.success(data);
                }

                @Override
                public void fail(int code, String errorMessage) {
                    super.fail(code, errorMessage);
                }
            });
        } catch (JSONException e) {
            callbackOnError(callback, BaseConstants.ERR_INVALID_PARAMETERS, "convert param to json failed");
            e.printStackTrace();
        }
    }

    private void reportRoomEngineEvent(Object param, final V2TIMValueCallback<Object> callback) {
        if (null == param || !(param instanceof String)) {
            callbackOnError(callback, BaseConstants.ERR_INVALID_PARAMETERS, "param is not string");
            return;
        }

        String json = (String) param;
        if (TextUtils.isEmpty(json)) {
            callbackOnError(callback, BaseConstants.ERR_INVALID_PARAMETERS, "param is empty");
            return;
        }

        try {
            JSONObject jsonObject = new JSONObject(json);
            long eventID = jsonObject.optLong("event_id");
            long eventCode = jsonObject.optLong("event_code");
            long eventResult = jsonObject.optLong("event_result");
            String eventMessage = jsonObject.optString("event_message");
            String moreMessage = jsonObject.optString("more_message");
            String extensionMessage = jsonObject.optString("extension_message");

            BaseManager.getInstance().reportRoomEngineEvent(
                eventID, eventCode, eventResult, eventMessage, moreMessage, extensionMessage, new IMCallback(callback) {
                    @Override
                    public void success(Object data) {
                        super.success(data);
                    }

                    @Override
                    public void fail(int code, String errorMessage) {
                        super.fail(code, errorMessage);
                    }
                });
        } catch (JSONException e) {
            callbackOnError(callback, BaseConstants.ERR_INVALID_PARAMETERS, "convert param to json failed");
            e.printStackTrace();
        }
    }
}
