package com.tencent.imsdk.manager;

import android.content.Context;
import android.os.Handler;
import android.os.HandlerThread;
import android.util.Log;
import com.tencent.imsdk.common.IMCallback;
import com.tencent.imsdk.common.SystemUtil;
import java.io.File;
import java.io.FileOutputStream;
import java.io.InputStream;
import java.net.HttpURLConnection;
import java.net.URL;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CopyOnWriteArrayList;

public class LibraryDownloader {
    private static final String TAG = "LibraryDownloader";

    private static final int ERROR_SUCCESS = 0;
    private static final int ERROR_DOWNLOAD_LIBRARY_FAILED = 101;
    private static final int ERROR_UNCOMPRESS_LIBRARY_FAILED = 102;
    private static final int ERROR_CHECK_LIBRARY_MD5_FAILED = 103;
    private static final int ERROR_LOAD_LIBRARY_FAILED = 104;

    private static final String LIBRARY_NAME = "libImSDK.so";
    private static final String LIBRARY_DIR_NAME = "imsdk-plus-";

    // 下面三个常量的定义，在蓝盾编译构建的时候会自动完成填充替换
    private static final String LIBRARY_VERSION = "SDK_VERSION";
    private static final String LIBRARY_MD5_ARM64 = "ARM64_SO_MD5";
    private static final String LIBRARY_MD5_ARM32 = "ARM32_SO_MD5";

    private boolean mIsLoading = false;
    private CopyOnWriteArrayList<IMCallback> mCallbackList = new CopyOnWriteArrayList();

    private String mLibraryPath = "";

    private HandlerThread mWorkerThread = null;

    private Handler mMainHandler = null;
    private Handler mWorkerHandler = null;

    private static class LibraryLoaderHolder {
        private static final LibraryDownloader IM_LIBRARY_LOADER = new LibraryDownloader();
    }

    public static LibraryDownloader getInstance() {
        return LibraryDownloader.LibraryLoaderHolder.IM_LIBRARY_LOADER;
    }

    public void downloadLibrary(Context context, IMCallback callback) {
        mLibraryPath =
            context.getFilesDir().toString() + File.separator + LIBRARY_DIR_NAME + LIBRARY_VERSION + File.separator;

        if (mWorkerThread == null) {
            mWorkerThread = new HandlerThread("libraryLoader");
            mWorkerThread.start();
        }

        if (mMainHandler == null) {
            mMainHandler = new Handler(context.getMainLooper());
        }

        if (mWorkerHandler == null) {
            mWorkerHandler = new Handler(mWorkerThread.getLooper());
        }

        mWorkerHandler.post(new Runnable() {
            @Override
            public void run() {
                mCallbackList.add(callback);
                if (mIsLoading) {
                    return;
                }

                mIsLoading = true;

                // 1. SO 已存在 && MD5 检验成功 && SO 加载成功
                if (ERROR_SUCCESS == loadLibrary()) {
                    Log.i(TAG, "check library md5 success and load library success");
                    notifyResult(ERROR_SUCCESS, "SUCCESS");
                    return;
                }

                // 删除旧 SO && 创建新目录
                SystemUtil.deleteDirectory(mLibraryPath);
                SystemUtil.createDirectory(mLibraryPath);

                // 2. SO 不存在
                Log.i(TAG, "start download library");
                boolean result = false;

                // 2.1 从后台服务器下载 SO
                String libraryName = "imsdk-plus-" + LIBRARY_VERSION + ".aar";
                result = fetchLibrary(libraryName);
                if (!result) {
                    Log.e(TAG, "download library failed");
                    notifyResult(ERROR_DOWNLOAD_LIBRARY_FAILED, "download library failed");
                    return;
                }
                Log.i(TAG, "download library success");

                // 2.2 解压 SO
                result = uncompressLibrary(libraryName);
                if (!result) {
                    Log.e(TAG, "uncompress library failed");
                    notifyResult(ERROR_UNCOMPRESS_LIBRARY_FAILED, "uncompress library failed");
                    return;
                }
                Log.i(TAG, "uncompress library success");

                // 2.3 加载 SO
                int error = loadLibrary();
                if (ERROR_SUCCESS != error) {
                    String message =
                        (error == ERROR_CHECK_LIBRARY_MD5_FAILED ? "check library md5 failed" : "load library failed");
                    Log.e(TAG, message);
                    notifyResult(error, message);
                    return;
                }

                Log.i(TAG, "load library success");
                notifyResult(ERROR_SUCCESS, "SUCCESS");
            }
        });
    }

    private int loadLibrary() {
        int error = ERROR_SUCCESS;

        // 优先尝试 arm64：校验 MD5 && 加载 SO
        String arm64Path = mLibraryPath + "jni" + File.separator + "arm64-v8a";
        String arm64MD5 = SystemUtil.getFileMD5(arm64Path + File.separator + LIBRARY_NAME);
        if (arm64MD5.equalsIgnoreCase(LIBRARY_MD5_ARM64)) {
            if (BaseManager.getInstance().setLibraryPath(arm64Path)) {
                error = ERROR_SUCCESS;
            } else {
                error = ERROR_CHECK_LIBRARY_MD5_FAILED;
            }
        } else {
            error = ERROR_CHECK_LIBRARY_MD5_FAILED;
        }

        if (error != ERROR_SUCCESS) {
            // 再次尝试 arm32：校验 MD5 && 加载 SO
            String arm32Path = mLibraryPath + "jni" + File.separator + "armeabi-v7a";
            String arm32MD5 = SystemUtil.getFileMD5(arm32Path + File.separator + LIBRARY_NAME);
            if (arm32MD5.equalsIgnoreCase(LIBRARY_MD5_ARM32)) {
                if (BaseManager.getInstance().setLibraryPath(arm32Path)) {
                    error = ERROR_SUCCESS;
                } else {
                    error = ERROR_CHECK_LIBRARY_MD5_FAILED;
                }
            } else {
                error = ERROR_CHECK_LIBRARY_MD5_FAILED;
            }
        }

        return error;
    }

    private boolean fetchLibrary(String libraryName) {
        List<String> urlList = new ArrayList<>();
        urlList.add("https://im.sdk.qcloud.com/download/plus/" + LIBRARY_VERSION + "/" + libraryName);
        urlList.add("https://sdk-im-1252463788.cos.accelerate.myqcloud.com/download/plus/" + LIBRARY_VERSION + "/"
            + libraryName);
        urlList.add("https://sdk-im-1252463788.file.myqcloud.com/download/plus/" + LIBRARY_VERSION + "/" + libraryName);
        urlList.add("https://sdk-im-1252463788.cos.ap-hongkong.myqcloud.com/download/plus/" + LIBRARY_VERSION + "/"
            + libraryName);

        for (int i = 0; i < urlList.size(); ++i) {
            try {
                URL url = new URL(urlList.get(i));
                HttpURLConnection connection = (HttpURLConnection) url.openConnection();
                connection.setRequestMethod("GET");
                connection.connect();
                if (connection.getResponseCode() != HttpURLConnection.HTTP_OK) {
                    // 当前 URL 连接失败，尝试下一个 URL
                    Log.w(TAG, "fetch library failed|url:" + urlList.get(i));
                    continue;
                }

                // 读取数据并写入文件
                File outputFile = new File(mLibraryPath, libraryName);
                InputStream inputStream = connection.getInputStream();
                FileOutputStream outputStream = new FileOutputStream(outputFile);
                byte[] buffer = new byte[4096];
                int bytesRead;
                while ((bytesRead = inputStream.read(buffer)) != -1) {
                    outputStream.write(buffer, 0, bytesRead);
                }
                inputStream.close();
                outputStream.close();

                // 下载成功
                return true;
            } catch (Exception e) {
                e.printStackTrace();
                return false;
            }
        }

        return false;
    }

    private boolean uncompressLibrary(String libraryName) {
        String fullPath = mLibraryPath + libraryName;
        File file = new File(fullPath);
        if (!file.exists()) {
            return false;
        }

        return SystemUtil.uncompressFile(fullPath, mLibraryPath);
    }

    private void notifyResult(int errorCode, String errorMessage) {
        if (errorCode != ERROR_SUCCESS) {
            SystemUtil.deleteDirectory(mLibraryPath);
        }

        CopyOnWriteArrayList<IMCallback> callbackList = new CopyOnWriteArrayList<>();
        for (IMCallback callback : mCallbackList) {
            callbackList.add(callback);
        }

        mIsLoading = false;
        mCallbackList.clear();

        mMainHandler.post(new Runnable() {
            @Override
            public void run() {
                for (IMCallback callback : callbackList) {
                    if (errorCode == 0) {
                        callback.success("OK");
                    } else {
                        callback.fail(errorCode, errorMessage);
                    }
                }
            }
        });
    }
}
