package com.tencent.imsdk.common;

import android.annotation.SuppressLint;
import android.app.Application;
import android.content.Context;
import android.content.SharedPreferences;
import android.content.pm.ApplicationInfo;
import android.content.pm.PackageManager;
import android.os.AsyncTask;
import android.os.Build;
import android.os.Debug;
import android.text.TextUtils;
import android.util.Log;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.lang.reflect.Constructor;
import java.lang.reflect.Method;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.Enumeration;
import java.util.UUID;
import java.util.regex.Pattern;
import java.util.zip.ZipEntry;
import java.util.zip.ZipException;
import java.util.zip.ZipFile;

public class SystemUtil {
    private static final String TAG = SystemUtil.class.getSimpleName();

    private static final int TIME_INTERVAL = 15000;
    private static boolean mFirstTimeRun = true;
    private static boolean sRunningMemCheck = false;
    private static int lastMemUsage = 0;
    private static float lastAppCpu = 0;
    private static float lastSysCpu = 0;
    private static long lastMemCheckTimeStamps = 0;
    private static long lastAppCpuTime = 0;
    private static long lastSysCpuTime = 0;
    private static CpuUsageMeasurer sCpuUsageMeasurer;

    private static String buildBrand = "";
    private static String buildManufacturer = "";
    private static String buildModel = "";
    private static String buildVersionRelease = "";
    private static int buildVersionSDKInt = 0;
    private static String currentProcessName = "";

    private static final String DEVICE_INFO = "DeviceInfo";
    private static final String DEVICE_ID = "DeviceID";

    private static boolean sIsLoadLibrarySuccess = false;

    private static boolean sIsLoadQuicLibrarySuccess = false;

    private static boolean sIsLoadAdvancedEncryptionLibrarySuccess = false;

    public static boolean isQuicPluginLibraryLoaded() {
        return sIsLoadQuicLibrarySuccess;
    }

    /**
     * 从默认路径加载 ImSDK 和 TIM 插件的 so 文件
     * @return 是否加载成功
     */
    public static boolean loadIMLibrary() {
        return loadIMLibrary(null);
    }

    /**
     * 从指定文件夹路径加载 ImSDK 和 TIM 插件的 so 文件
     * @param libraryDirPath so 所在的文件夹路径
     * @return 是否加载成功
     */
    @SuppressLint("UnsafeDynamicallyLoadedCode")
    public static boolean loadIMLibrary(String libraryDirPath) {
        if (false == sIsLoadLibrarySuccess) {
            sIsLoadLibrarySuccess = loadIMSDKLibrary(libraryDirPath);
        }

        if (false == sIsLoadQuicLibrarySuccess) {
            sIsLoadQuicLibrarySuccess =
                loadPluginLibrary(libraryDirPath, "com.tencent.imsdk.timquicplugin.QuicPluginLoader");
        }

        if (false == sIsLoadAdvancedEncryptionLibrarySuccess) {
            sIsLoadAdvancedEncryptionLibrarySuccess = loadPluginLibrary(
                libraryDirPath, "com.tencent.imsdk.timadvancedencryptionplugin.AdvancedEncryptionPluginLoader");
        }

        return sIsLoadLibrarySuccess;
    }

    private static boolean loadIMSDKLibrary(String libraryDirPath) {
        try {
            if (!TextUtils.isEmpty(libraryDirPath)) {
                String libraryName = "libImSDK.so";
                String libraryFullPath = libraryDirPath + File.separator + libraryName;
                System.load(libraryFullPath);
                Log.i(TAG, "system load so library success: " + libraryFullPath);
            } else {
                System.loadLibrary("ImSDK");
                Log.i(TAG, "system load so library success, libImSDK.so");
            }
        } catch (UnsatisfiedLinkError e) {
            Log.e(TAG, "system load so library failed, " + e.getMessage());
            return false;
        } catch (Exception e) {
            Log.e(TAG, "system load so library failed, " + e.getMessage());
            return false;
        }

        return true;
    }

    private static boolean loadPluginLibrary(String libraryDirPath, String pluginClassName) {
        boolean isLoadLibrarySuccess = false;

        try {
            Class<?> pluginClass = Class.forName(pluginClassName);
            Constructor<?> constructor = pluginClass.getConstructor();
            Object pluginInstance = constructor.newInstance();
            Method loadMethod = pluginClass.getMethod("load", String.class);
            isLoadLibrarySuccess = (boolean) loadMethod.invoke(pluginInstance, libraryDirPath);
        } catch (Exception e) {
            isLoadLibrarySuccess = false;
        }

        return isLoadLibrarySuccess;
    }

    public static void setBuildModel(String model) {
        buildModel = model;
    }

    public static void setBuildBrand(String brand) {
        buildBrand = brand;
    }

    public static void setBuildManufacturer(String manufacturer) {
        buildManufacturer = manufacturer;
    }

    public static void setBuildVersionRelease(String versionRelease) {
        buildVersionRelease = versionRelease;
    }

    public static void setBuildVersionSDKInt(int versionSDKInt) {
        buildVersionSDKInt = versionSDKInt;
    }

    public static String getDeviceType() {
        if (TextUtils.isEmpty(buildModel)) {
            buildModel = Build.MODEL;
        }

        return buildModel;
    }

    public static String getSystemVersion() {
        if (TextUtils.isEmpty(buildVersionRelease)) {
            buildVersionRelease = Build.VERSION.RELEASE;
        }

        return buildVersionRelease;
    }

    public static int getSDKVersion() {
        if (buildVersionSDKInt == 0) {
            buildVersionSDKInt = Build.VERSION.SDK_INT;
        }

        return buildVersionSDKInt;
    }

    public static String getDeviceID() {
        Context context = IMContext.getInstance().getApplicationContext();
        if (null == context) {
            return "";
        }

        String deviceId = "";
        SharedPreferences preferences = context.getSharedPreferences(DEVICE_INFO, Context.MODE_PRIVATE);
        if (!preferences.contains(DEVICE_ID)) {
            deviceId = UUID.randomUUID().toString();
        } else {
            // 从 8.1 版本开始 deviceID 是底层来向上层获取并保存到 config 文件，不再保存到 SharedPreferences 中，减少 sp 的使用，避免 ANR。
            // 获取时检查 sp 是否有保存过 deviceID 的原因是：使用老版本的客户，底层 config 文件没有保存 deviceID，
            // 升级后要从上层获取在平台层持久化保存的 deviceID，保证给后台的一致。
            deviceId = preferences.getString(DEVICE_ID, "");
            // 检测是否符合UUID的形式，不符合则重新生成
            String pattern = "\\w{8}(-\\w{4}){3}-\\w{12}";
            boolean isMatch = Pattern.matches(pattern, deviceId);
            if (!isMatch || TextUtils.isEmpty(deviceId)) {
                deviceId = UUID.randomUUID().toString();
            }
        }

        return deviceId;
    }

    public static int getInstanceType() {
        int vendorId = 2002;

        if (isBrandXiaoMi()) {
            vendorId = 2000;
        } else if (isBrandNewHonor()) {
            vendorId = 2006;
        } else if (isBrandHuawei()) {
            vendorId = 2001;
        } else if (isBrandMeizu()) {
            vendorId = 2003;
        } else if (isBrandOppo()) {
            vendorId = 2004;
        } else if (isBrandVivo()) {
            vendorId = 2005;
        } else if (isBrandNIO()) {
            return 2007;
        }

        return vendorId;
    }

    static boolean isBrandXiaoMi() {
        String brandXiaoMi = "xiaomi";
        String subBrandXiaoMi = "Redmi";
        return brandXiaoMi.equalsIgnoreCase(getBuildBrand()) || brandXiaoMi.equalsIgnoreCase(getBuildManufacturer())
            || subBrandXiaoMi.equalsIgnoreCase(getBuildBrand())
            || subBrandXiaoMi.equalsIgnoreCase(getBuildManufacturer());
    }

    static boolean isBrandHuawei() {
        String brandHuaWei = "huawei";
        String brandHonor = "honor";
        return brandHuaWei.equalsIgnoreCase(getBuildBrand()) || brandHuaWei.equalsIgnoreCase(getBuildManufacturer())
            || brandHonor.equalsIgnoreCase(getBuildBrand());
    }

    static boolean isBrandMeizu() {
        String brandMeiZu = "meizu";
        return brandMeiZu.equalsIgnoreCase(getBuildBrand()) || brandMeiZu.equalsIgnoreCase(getBuildManufacturer());
    }

    static boolean isBrandOppo() {
        return "oppo".equalsIgnoreCase(getBuildBrand()) || "oppo".equalsIgnoreCase(getBuildManufacturer())
            || "realme".equalsIgnoreCase(getBuildBrand()) || "realme".equalsIgnoreCase(getBuildManufacturer())
            || "oneplus".equalsIgnoreCase(getBuildBrand()) || "oneplus".equalsIgnoreCase(getBuildManufacturer());
    }

    static boolean isBrandVivo() {
        String brandVivo = "vivo";
        String subBrandVivo = "iQOO";
        return brandVivo.equalsIgnoreCase(getBuildBrand()) || brandVivo.equalsIgnoreCase(getBuildManufacturer())
            || subBrandVivo.equalsIgnoreCase(getBuildBrand()) || subBrandVivo.equalsIgnoreCase(getBuildManufacturer());
    }

    static boolean isBrandNewHonor() {
        return "honor".equalsIgnoreCase(getBuildBrand()) && "honor".equalsIgnoreCase(getBuildManufacturer());
    }

    static boolean isBrandNIO() {
        return "nio".equalsIgnoreCase(getBuildBrand()) && "nio".equalsIgnoreCase(getBuildManufacturer());
    }

    private static String getBuildBrand() {
        if (TextUtils.isEmpty(buildBrand)) {
            buildBrand = Build.BRAND;
        }

        return buildBrand;
    }

    private static String getBuildManufacturer() {
        if (TextUtils.isEmpty(buildManufacturer)) {
            buildManufacturer = Build.MANUFACTURER;
        }

        return buildManufacturer;
    }

    public static String getSDKInitPath() {
        Context context = IMContext.getInstance().getApplicationContext();
        if (null == context) {
            return "";
        }

        String initPath = context.getFilesDir().toString();
        String mainProcessName = context.getPackageName();
        String currentProcessName = getCurrentProcessName();
        if (!TextUtils.isEmpty(currentProcessName) && !mainProcessName.equals(currentProcessName)) {
            try {
                String pathSuffix = currentProcessName;
                int index = currentProcessName.lastIndexOf(":");
                if (index < 0) {
                    index = currentProcessName.lastIndexOf(".");
                }

                if (index >= 0) {
                    pathSuffix = currentProcessName.substring(index + 1);
                }

                initPath += File.separator + pathSuffix + File.separator;
            } catch (Exception e) {
                Log.e(TAG, "getSDKInitPath exception = " + e);
            }
        }

        Log.d(TAG, "SDK Init Path: " + initPath);

        return initPath;
    }

    public static String getSDKLogPath() {
        Context context = IMContext.getInstance().getApplicationContext();
        if (null == context) {
            return "";
        }

        // Android 10 系统开始使用分区存储，因此使用应用专有目录。
        // 1、优先自己组装日志目录，防止 ANR
        String logPath = "/sdcard/Android/data/" + context.getPackageName() + "/files/log/tencent/imsdk";
        File fileDir = new File(logPath);
        boolean createFilesDirSuccess = false;
        if (!fileDir.exists()) {
            createFilesDirSuccess = fileDir.mkdirs();
        } else {
            createFilesDirSuccess = true;
        }

        // 2、创建失败，再用系统方式获取
        if (!createFilesDirSuccess) {
            fileDir = context.getExternalFilesDir(null);
            if (fileDir != null) {
                logPath = fileDir.getAbsolutePath() + File.separator + "log" + File.separator + "tencent"
                    + File.separator + "imsdk";
                fileDir = new File(logPath);
                if (!fileDir.exists()) {
                    createFilesDirSuccess = fileDir.mkdirs();
                } else {
                    createFilesDirSuccess = true;
                }
            }
        }

        // 3、上面路径都失败，最后再用 data/data/包名/files 目录
        if (!createFilesDirSuccess) {
            fileDir = context.getFilesDir();
            Log.w(TAG, "create log folder failed");
        }

        logPath = fileDir.getAbsolutePath() + File.separator;
        String mainProcessName = context.getPackageName();
        String currentProcessName = getCurrentProcessName();
        if (!TextUtils.isEmpty(currentProcessName) && !mainProcessName.equals(currentProcessName)) {
            try {
                String pathSuffix = currentProcessName;
                int index = currentProcessName.lastIndexOf(":");
                if (index < 0) {
                    index = currentProcessName.lastIndexOf(".");
                }

                if (index >= 0) {
                    pathSuffix = currentProcessName.substring(index + 1);
                }

                logPath += pathSuffix + File.separator;
            } catch (Exception e) {
                Log.e(TAG, "getSDKLogPath exception = " + e);
            }
        }

        Log.d(TAG, "SDK LOG Path: " + logPath);

        return logPath;
    }

    /**
     * 计算字符串 MD5
     *
     * @param string
     * @return
     */
    public static String getStringMD5(String string) {
        if (TextUtils.isEmpty(string)) {
            return "";
        }
        MessageDigest md5 = null;
        try {
            md5 = MessageDigest.getInstance("MD5");
            byte[] bytes = md5.digest(string.getBytes());
            String result = "";
            for (byte b : bytes) {
                String temp = Integer.toHexString(b & 0xff);
                if (temp.length() == 1) {
                    temp = "0" + temp;
                }
                result += temp;
            }
            return result;
        } catch (NoSuchAlgorithmException e) {
            e.printStackTrace();
        }
        return "";
    }

    /**
     * 计算文件 MD5
     *
     * @param filePath
     * @return
     */
    public static String getFileMD5(String filePath) {
        File file = new File(filePath);
        if (file == null || !file.exists() || !file.isFile()) {
            return "";
        }

        MessageDigest digest = null;
        FileInputStream fileInputStream = null;
        byte[] buffer = new byte[1024];
        int length = 0;
        try {
            // 创建 MessageDigest 实例，指定算法为 MD5
            digest = MessageDigest.getInstance("MD5");

            // 读取文件并更新摘要
            fileInputStream = new FileInputStream(file);
            while ((length = fileInputStream.read(buffer)) != -1) {
                digest.update(buffer, 0, length);
            }

            // 计算最终的 MD5 值
            byte[] md5Bytes = digest.digest();

            // 将字节数组转换为十六进制字符串
            StringBuilder hexString = new StringBuilder();
            for (byte b : md5Bytes) {
                String hex = Integer.toHexString(0xff & b);
                if (hex.length() == 1) {
                    hexString.append('0');
                }
                hexString.append(hex);
            }

            return hexString.toString();
        } catch (NoSuchAlgorithmException | IOException e) {
            e.printStackTrace();
            return "";
        } finally {
            try {
                if (fileInputStream != null) {
                    fileInputStream.close();
                }
            } catch (IOException e) {
                e.printStackTrace();
            }
        }
    }

    public static float getAppCpuUsage() {
        long currentTimeMillis = System.currentTimeMillis();
        if (currentTimeMillis - lastAppCpuTime < TIME_INTERVAL) {
            return lastAppCpu;
        }

        int[] cpuUsage = getProcessCPURate();
        int appCpuUsage = cpuUsage[0] / 10;
        float appCpuUsagePercent = (float) appCpuUsage / 100;
        lastAppCpu = appCpuUsagePercent;
        lastAppCpuTime = currentTimeMillis;
        return appCpuUsagePercent;
    }

    public static float getSysCpuUsage() {
        long currentTimeMillis = System.currentTimeMillis();
        if (currentTimeMillis - lastSysCpuTime < TIME_INTERVAL) {
            return lastSysCpu;
        }

        int[] cpuUsage = getProcessCPURate();
        int systemCpuUsage = cpuUsage[1] / 10;
        float systemCpuUsagePercent = (float) systemCpuUsage / 100;
        lastSysCpu = systemCpuUsagePercent;
        lastSysCpuTime = currentTimeMillis;
        return systemCpuUsagePercent;
    }

    static int[] getProcessCPURate() {
        if (sCpuUsageMeasurer == null) {
            sCpuUsageMeasurer = new CpuUsageMeasurer();
        }
        if (mFirstTimeRun) {
            mFirstTimeRun = false;
            sCpuUsageMeasurer.getCpuUsage();
            return new int[] {0, 0};
        }

        return sCpuUsageMeasurer.getCpuUsage();
    }

    public static float getAppMemory() {
        final long currentTimeMillis = System.currentTimeMillis();
        if (sRunningMemCheck
            || (lastMemCheckTimeStamps != 0 && currentTimeMillis - lastMemCheckTimeStamps < TIME_INTERVAL)) {
            ///< 请求间隔小于15秒的，直接返回上一次的计算结果。
            return (float) lastMemUsage;
        }
        sRunningMemCheck = true;
        AsyncTask.execute(new Runnable() {
            @Override
            public void run() {
                int totalPss = 0;
                long ts = System.currentTimeMillis();
                sRunningMemCheck = false;
                try {
                    Debug.MemoryInfo memoryInfo = new Debug.MemoryInfo();
                    Debug.getMemoryInfo(memoryInfo);
                    totalPss = memoryInfo.getTotalPss();
                    lastMemCheckTimeStamps = currentTimeMillis;
                    lastMemUsage = totalPss / 1024;
                } catch (Exception e) {
                    totalPss = 0;
                }
            }
        });
        return (float) lastMemUsage;
    }

    public static String getCurrentProcessName() {
        if (!TextUtils.isEmpty(currentProcessName)) {
            return currentProcessName;
        }

        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.P) {
            currentProcessName = Application.getProcessName();
            return currentProcessName;
        }

        try {
            final Method declaredMethod =
                Class.forName("android.app.ActivityThread", false, Application.class.getClassLoader())
                    .getDeclaredMethod("currentProcessName", (Class<?>[]) new Class[0]);
            declaredMethod.setAccessible(true);
            final Object invoke = declaredMethod.invoke(null, new Object[0]);
            if (invoke instanceof String) {
                currentProcessName = (String) invoke;
            }
        } catch (Throwable e) {
            e.printStackTrace();
        }

        return currentProcessName;
    }

    public static String getAppName() {
        String appName = "";
        Context context = IMContext.getInstance().getApplicationContext();
        PackageManager packageManager = context.getPackageManager();
        try {
            ApplicationInfo applicationInfo =
                packageManager.getApplicationInfo(context.getPackageName(), PackageManager.GET_META_DATA);
            packageManager.getApplicationLabel(applicationInfo);
            CharSequence labelCharSequence = applicationInfo.loadLabel(packageManager);
            if (labelCharSequence != null && labelCharSequence.length() > 0) {
                appName = labelCharSequence.toString();
            }
        } catch (PackageManager.NameNotFoundException e) {
            Log.e(TAG, "getAppName exception:" + e.getMessage());
        }

        return appName;
    }

    /**
     * 创建目录
     *
     * @param dirPath 目录路径
     * @return 是否创建成功
     */
    public static boolean createDirectory(String dirPath) {
        File dir = new File(dirPath);

        // 如果目录已经存在，直接返回 true
        if (dir.exists()) {
            return dir.isDirectory();
        }

        // 创建目录（包括所有不存在的父目录）
        return dir.mkdirs();
    }

    /**
     * 删除目录
     * @param dirPath
     */
    public static boolean deleteDirectory(String dirPath) {
        if (TextUtils.isEmpty(dirPath)) {
            return false;
        }

        File directory = new File(dirPath);
        if (directory == null || !directory.exists()) {
            return false;
        }

        // 如果是文件，直接删除
        if (directory.isFile()) {
            return directory.delete();
        }

        // 如果是目录，递归删除其所有子文件和子目录
        if (directory.isDirectory()) {
            File[] files = directory.listFiles();
            if (files != null) {
                for (File file : files) {
                    deleteDirectory(file.getAbsolutePath()); // 递归删除子文件或子目录
                }
            }
        }

        // 删除空目录
        return directory.delete();
    }

    /**
     * 解压 ZIP 文件到指定目录
     *
     * @param zipFilePath ZIP 文件路径
     * @param destDirPath 解压目标目录路径
     */
    public static boolean uncompressFile(String zipFilePath, String destDirPath) {
        byte[] buffer = new byte[1024];
        try {
            String canonicalDestinationDirPath = new File(destDirPath).getCanonicalPath();
            // 创建 Zip 文件对象
            ZipFile zipFile = new ZipFile(zipFilePath);
            Enumeration<?> entries = zipFile.entries();
            while (entries.hasMoreElements()) {
                ZipEntry entry = (ZipEntry) entries.nextElement();
                String filePath = destDirPath + File.separator + entry.getName();
                if (!entry.isDirectory()) {
                    // 避免 Zip Slip 漏洞
                    File destinationfile = new File(filePath);
                    String canonicalDestinationFile = destinationfile.getCanonicalPath();
                    if (!canonicalDestinationFile.startsWith(canonicalDestinationDirPath + File.separator)) {
                        throw new ZipException("Entry is outside of the target dir: " + entry.getName());
                    }
                    // 创建所有非目录的父目录
                    File parent = destinationfile.getParentFile();
                    if (!parent.isDirectory()) {
                        parent.mkdirs();
                    }
                    // 写入文件内容到磁盘
                    try {
                        InputStream is = zipFile.getInputStream(entry);
                        FileOutputStream fos = new FileOutputStream(filePath);
                        int len;
                        while ((len = is.read(buffer)) > 0) {
                            fos.write(buffer, 0, len);
                        }
                        is.close();
                        fos.close();
                    } catch (Exception e) {
                        e.printStackTrace();
                        return false;
                    }
                } else {
                    // 创建目录结构（如果需要）
                    // 注意：在某些情况下，你可能不需要这一行，因为 ZipFile 会自动创建目录结构。但保留它通常不会有问题。
                    File dir = new File(filePath);
                    dir.mkdirs();
                }
            }
            // 关闭 Zip 文件对象以释放资源
            zipFile.close();
            return true;
        } catch (IOException e) {
            e.printStackTrace();
            return false;
        }
    }
}
