package com.tencent.imsdk.conversation;

import com.tencent.imsdk.BaseConstants;
import com.tencent.imsdk.common.IMCallback;
import com.tencent.imsdk.common.IMContext;
import com.tencent.imsdk.manager.BaseManager;
import com.tencent.imsdk.message.DraftMessage;
import com.tencent.imsdk.v2.V2TIMCallback;
import com.tencent.imsdk.v2.V2TIMConversationListFilter;
import com.tencent.imsdk.v2.V2TIMValueCallback;
import java.util.List;

public class ConversationManager {
    private static final String TAG = "ConversationManager";

    private static class ConversationManagerHolder {
        private static final ConversationManager conversationManager = new ConversationManager();
    }

    public static ConversationManager getInstance() {
        return ConversationManager.ConversationManagerHolder.conversationManager;
    }

    private ConversationListener mInternalConversationListener;
    private ConversationListener mConversationListener;

    public void init() {
        initInternalConversationListener();
    }

    private void initInternalConversationListener() {
        if (mInternalConversationListener == null) {
            mInternalConversationListener = new ConversationListener() {
                @Override
                public void onSyncServerStart() {
                    IMContext.getInstance().runOnMainThread(new Runnable() {
                        @Override
                        public void run() {
                            if (mConversationListener != null) {
                                mConversationListener.onSyncServerStart();
                            }
                        }
                    });
                }

                @Override
                public void onSyncServerFinish() {
                    IMContext.getInstance().runOnMainThread(new Runnable() {
                        @Override
                        public void run() {
                            if (mConversationListener != null) {
                                mConversationListener.onSyncServerFinish();
                            }
                        }
                    });
                }

                @Override
                public void onSyncServerFailed() {
                    IMContext.getInstance().runOnMainThread(new Runnable() {
                        @Override
                        public void run() {
                            if (mConversationListener != null) {
                                mConversationListener.onSyncServerFailed();
                            }
                        }
                    });
                }

                @Override
                public void onNewConversation(final List<Conversation> conversationList) {
                    IMContext.getInstance().runOnMainThread(new Runnable() {
                        @Override
                        public void run() {
                            if (mConversationListener != null) {
                                mConversationListener.onNewConversation(conversationList);
                            }
                        }
                    });
                }

                @Override
                public void onConversationChanged(final List<Conversation> conversationList) {
                    IMContext.getInstance().runOnMainThread(new Runnable() {
                        @Override
                        public void run() {
                            if (mConversationListener != null) {
                                mConversationListener.onConversationChanged(conversationList);
                            }
                        }
                    });
                }

                @Override
                public void onConversationDeleted(final List<String> conversationIDList) {
                    IMContext.getInstance().runOnMainThread(new Runnable() {
                        @Override
                        public void run() {
                            if (mConversationListener != null) {
                                mConversationListener.onConversationDeleted(conversationIDList);
                            }
                        }
                    });
                }

                @Override
                public void onUnreadMessageCountChanged(final UnreadMessageCountResult result) {
                    IMContext.getInstance().runOnMainThread(new Runnable() {
                        @Override
                        public void run() {
                            if (mConversationListener != null) {
                                mConversationListener.onUnreadMessageCountChanged(result);
                            }
                        }
                    });
                }

                @Override
                public void onConversationGroupCreated(
                    final String groupName, final List<Conversation> conversationList) {
                    IMContext.getInstance().runOnMainThread(new Runnable() {
                        @Override
                        public void run() {
                            if (mConversationListener != null) {
                                mConversationListener.onConversationGroupCreated(groupName, conversationList);
                            }
                        }
                    });
                }

                @Override
                public void onConversationGroupDeleted(final String groupName) {
                    IMContext.getInstance().runOnMainThread(new Runnable() {
                        @Override
                        public void run() {
                            if (mConversationListener != null) {
                                mConversationListener.onConversationGroupDeleted(groupName);
                            }
                        }
                    });
                }

                @Override
                public void onConversationGroupNameChanged(final String oldName, final String newName) {
                    IMContext.getInstance().runOnMainThread(new Runnable() {
                        @Override
                        public void run() {
                            if (mConversationListener != null) {
                                mConversationListener.onConversationGroupNameChanged(oldName, newName);
                            }
                        }
                    });
                }

                @Override
                public void onConversationsAddedToGroup(
                    final String groupName, final List<Conversation> conversationList) {
                    IMContext.getInstance().runOnMainThread(new Runnable() {
                        @Override
                        public void run() {
                            if (mConversationListener != null) {
                                mConversationListener.onConversationsAddedToGroup(groupName, conversationList);
                            }
                        }
                    });
                }

                @Override
                public void onConversationsDeletedFromGroup(
                    final String groupName, final List<Conversation> conversationList, final int reason) {
                    IMContext.getInstance().runOnMainThread(new Runnable() {
                        @Override
                        public void run() {
                            if (mConversationListener != null) {
                                mConversationListener.onConversationsDeletedFromGroup(
                                    groupName, conversationList, reason);
                            }
                        }
                    });
                }

                @Override
                public void onConversationGroupOrderDataChanged(final String conversationGroupOrderData) {
                    IMContext.getInstance().runOnMainThread(new Runnable() {
                        @Override
                        public void run() {
                            BaseManager.getInstance().notifyExperimentalNotify(
                                "onConversationGroupOrderDataChanged", conversationGroupOrderData);
                        }
                    });
                }
            };
        }
        nativeSetConversationListener(mInternalConversationListener);
    }

    public void setConversationListener(ConversationListener conversationListener) {
        mConversationListener = conversationListener;
    }

    public void getConversationList(long nextSeq, int count, IMCallback<ConversationResult> callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }
        nativeGetConversationList(nextSeq, count, callback);
    }

    public void getConversationList(
        List<ConversationKey> conversationKeyList, IMCallback<List<Conversation>> callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }
        nativeGetConversations(conversationKeyList, callback);
    }

    public void getConversationListByFilter(
        ConversationListFilter filter, long nextSeq, int count, IMCallback<ConversationResult> callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }
        nativeGetConversationListByFilter(filter, nextSeq, count, callback);
    }

    public void getConversationInfo(ConversationKey conversationKey, IMCallback<Conversation> callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }
        nativeGetConversationInfo(conversationKey, callback);
    }

    public void deleteConversationList(
        List<ConversationKey> conversationKeyList, boolean clearMessage, IMCallback callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }
        nativeDeleteConversationList(conversationKeyList, clearMessage, callback);
    }

    public void setConversationDraft(ConversationKey conversationKey, DraftMessage draftMessage, IMCallback callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }
        nativeSetConversationDraft(conversationKey, draftMessage, callback);
    }

    public void pinConversation(ConversationKey conversationKey, boolean isPinned, IMCallback callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }
        nativePinConversation(conversationKey, isPinned, callback);
    }

    public void getTotalUnreadMessageCount(
        ConversationListFilter filter, IMCallback<UnreadMessageCountResult> callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }
        nativeGetTotalUnreadMessageCount(filter, callback);
    }

    public void subscribeUnreadMessageCountByFilter(ConversationListFilter filter) {
        if (!BaseManager.getInstance().isInited()) {
            return;
        }
        nativeSubscribeUnreadMessageCountByFilter(filter);
    }

    public void unsubscribeUnreadMessageCountByFilter(ConversationListFilter filter) {
        if (!BaseManager.getInstance().isInited()) {
            return;
        }
        nativeUnsubscribeUnreadMessageCountByFilter(filter);
    }

    public void setCosSaveRegionForConversation(
        ConversationKey conversationKey, String cosSaveRegion, IMCallback callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }
        nativeSetCosSaveRegionForConversation(conversationKey, cosSaveRegion, callback);
    }

    public void clearUnreadMessage(
        boolean clearC2CUnreadMessage, boolean clearGroupUnreadMessage, IMCallback callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }
        nativeClearUnreadMessage(clearC2CUnreadMessage, clearGroupUnreadMessage, callback);
    }

    public void markConversation(
        List<ConversationKey> conversationKeyList, long markType, boolean enableMark, IMCallback callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }
        nativeMarkConversation(conversationKeyList, markType, enableMark, callback);
    }

    public void setConversationCustomData(
        List<ConversationKey> conversationKeyList, String customData, IMCallback callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }
        nativeSetConversationCustomData(conversationKeyList, customData, callback);
    }

    public void createConversationGroup(
        String groupName, List<ConversationKey> conversationKeyList, IMCallback callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }
        nativeCreateConversationGroup(groupName, conversationKeyList, callback);
    }

    public void getConversationGroupList(IMCallback callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }
        nativeGetConversationGroupList(callback);
    }

    public void deleteConversationGroup(String groupName, IMCallback callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }
        nativeDeleteConversationGroup(groupName, callback);
    }

    public void renameConversationGroup(String oldName, String newName, IMCallback callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }
        nativeRenameConversationGroup(oldName, newName, callback);
    }

    public void addConversationsToGroup(
        String groupName, List<ConversationKey> conversationKeyList, IMCallback callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }
        nativeAddConversationsToGroup(groupName, conversationKeyList, callback);
    }

    public void deleteConversationsFromGroup(
        String groupName, List<ConversationKey> conversationKeyList, IMCallback callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }
        nativeDeleteConversationsFromGroup(groupName, conversationKeyList, callback);
    }

    public void setConversationGroupOrderData(String conversationGroupOrderData, IMCallback callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }
        nativeSetConversationGroupOrderData(conversationGroupOrderData, callback);
    }

    public void getConversationGroupOrderData(IMCallback<String> callback) {
        if (!BaseManager.getInstance().isInited()) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }
        nativeGetConversationGroupOrderData(callback);
    }

    protected native void nativeSetConversationListener(ConversationListener listener);

    protected native void nativeGetConversationList(long nextSeq, int count, IMCallback callback);

    protected native void nativeGetConversationListByFilter(
        ConversationListFilter filter, long nextSeq, int count, IMCallback callback);

    protected native void nativeGetConversations(List<ConversationKey> conversationKeyList, IMCallback callback);

    protected native void nativeGetConversationInfo(ConversationKey conversationKey, IMCallback callback);

    protected native void nativeDeleteConversationList(
        List<ConversationKey> conversationKeyList, boolean clearMessage, IMCallback callback);

    protected native void nativeSetConversationDraft(
        ConversationKey conversationKey, DraftMessage draftMessage, IMCallback callback);

    protected native void nativePinConversation(ConversationKey conversationKey, boolean isPinned, IMCallback callback);

    protected native void nativeGetTotalUnreadMessageCount(ConversationListFilter filter, IMCallback callback);

    protected native void nativeSubscribeUnreadMessageCountByFilter(ConversationListFilter filter);

    protected native void nativeUnsubscribeUnreadMessageCountByFilter(ConversationListFilter filter);

    protected native void nativeSetCosSaveRegionForConversation(
        ConversationKey conversationKey, String cosSaveRegion, IMCallback callback);

    protected native void nativeClearUnreadMessage(
        boolean clearC2CUnreadMessage, boolean clearGroupUnreadMessage, IMCallback callback);

    protected native void nativeMarkConversation(
        List<ConversationKey> conversationKeyList, long markType, boolean enableMark, IMCallback callback);

    protected native void nativeSetConversationCustomData(
        List<ConversationKey> conversationKeyList, String customData, IMCallback callback);

    protected native void nativeCreateConversationGroup(
        String groupName, List<ConversationKey> conversationKeyList, IMCallback callback);

    protected native void nativeGetConversationGroupList(IMCallback callback);

    protected native void nativeDeleteConversationGroup(String groupName, IMCallback callback);

    protected native void nativeRenameConversationGroup(String oldName, String newName, IMCallback callback);

    protected native void nativeAddConversationsToGroup(
        String groupName, List<ConversationKey> conversationKeyList, IMCallback callback);

    protected native void nativeDeleteConversationsFromGroup(
        String groupName, List<ConversationKey> conversationKeyList, IMCallback callback);

    protected native void nativeSetConversationGroupOrderData(String conversationGroupOrderData, IMCallback callback);

    protected native void nativeGetConversationGroupOrderData(IMCallback callback);
}
