package com.tencent.imsdk.manager;

import android.content.Context;
import android.os.Build;
import android.util.LayoutDirection;
import android.util.Log;
import com.tencent.imsdk.BaseConstants;
import com.tencent.imsdk.common.IMCallback;
import com.tencent.imsdk.common.IMContext;
import com.tencent.imsdk.common.IMLog;
import com.tencent.imsdk.common.NetworkInfoCenter;
import com.tencent.imsdk.common.SystemUtil;
import com.tencent.imsdk.community.CommunityManager;
import com.tencent.imsdk.conversation.ConversationManager;
import com.tencent.imsdk.group.GroupManager;
import com.tencent.imsdk.group.GroupMemberInfo;
import com.tencent.imsdk.message.MessageCenter;
import com.tencent.imsdk.relationship.ReceiveMessageOptInfo;
import com.tencent.imsdk.relationship.RelationshipManager;
import com.tencent.imsdk.relationship.UserInfo;
import com.tencent.imsdk.relationship.UserStatus;
import com.tencent.imsdk.signaling.SignalingManager;
import java.lang.ref.WeakReference;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

public class BaseManager implements NetworkInfoCenter.INetworkChangeListener {
    private static final String TAG = BaseManager.class.getSimpleName();

    // 账号类型
    public static final int ACCOUNT_TYPE_UNKOWN = 0;
    public static final int ACCOUNT_TYPE_IM = 1;
    public static final int ACCOUNT_TYPE_PUSH = 2;

    // TUI 组件类型
    public static final long TUI_COMPONENT_UNKNOWN = 0;
    public static final long TUI_COMPONENT_CORE = 1;
    public static final long TUI_COMPONENT_CONVERSATION = 2;
    public static final long TUI_COMPONENT_CHAT = 3;
    public static final long TUI_COMPONENT_CONTACT = 4;
    /**
     * @deprecated
     * TUIGroup 组件在 8.4 版本已删除
     */
    public static final long TUI_COMPONENT_GROUP = 5;
    public static final long TUI_COMPONENT_SEARCH = 6;
    public static final long TUI_COMPONENT_OFFLINEPUSH = 7;
    public static final long TUI_COMPONENT_COMMUNITY = 8;
    public static final long TUI_PLUGIN_GROUP_NOTE = 9;
    public static final long TUI_PLUGIN_POLL = 10;
    public static final long TUI_PLUGIN_CONVERSATION_MARK = 11;
    public static final long TUI_PLUGIN_CONVERSATION_GROUP = 12;
    public static final long TUI_TIMPUSH_SDK = 13;
    public static final long TUI_TPUSH_SDK = 19;
    public static final long TUI_PLUGIN_CUSTOMER_SERVICE = 14;
    public static final long TUI_PLUGIN_CHAT_BOT = 15;
    public static final int TUI_COMPONENT_MULTIMEDIA = 20;
    public static final int TUI_COMPONENT_MULTIMEDIA_ADVANCED = 21;
    // TUI 组件风格类型
    public static final long TUI_STYLE_TYPE_CLASSIC = 0;
    public static final long TUI_STYLE_TYPE_MINIMALIST = 1;

    // 检查 TUI 组件的最大次数限制
    private static final int TUI_COMPONENT_CHECK_COUNT_LIMIT = 5;
    // 检查 TUI 组件的堆栈遍历深度
    private static final int TUI_COMPONENT_STACK_LAYER_LIMIT = 10;

    // UIPlatform
    private static final int UI_PLATFORM_UNKNOWN = 0;
    private static final int UI_PLATFORM_FLUTTER = 1;
    private static final int UI_PLATFORM_FLUTTER_UIKIT = 2;
    private static final int UI_PLATFORM_UNITY = 5;
    private static final int UI_PLATFORM_UNITY_UIKIT = 6;
    private static final int UI_PLATFORM_TUIKIT = 15;
    private static final int UI_PLATFORM_TUIKIT_MINIMALIST = 26;
    private static final int UI_PLATFORM_TUIKIT_RTL = 33;
    private static final int UI_PLATFORM_TUIKIT_MINIMALIST_RTL = 34;
    private static final int UI_PLATFORM_ATOMICXCORE = 47;

    private boolean mInvokeFromTUIKit = false;
    private boolean mInvokeFromTUICore = false;

    private Map<Long, Long> mTUIComponentMap = new HashMap<>();
    private HashMap<Long, Integer> mTUIComponentCheckCountMap = new HashMap<>();

    private SDKConfig.NetworkInfo mLastNetworkInfo = new SDKConfig.NetworkInfo();

    private SDKConfig.ProxyInfo mProxyInfo = new SDKConfig.ProxyInfo();

    private SDKConfig.DatabaseEncryptInfo mDatabaseEncryptInfo = new SDKConfig.DatabaseEncryptInfo();

    private SDKConfig.PacketRetryInfo mPacketRetryInfo = new SDKConfig.PacketRetryInfo();

    private WeakReference<SDKListener> sdkListenerWeakReference;

    private static boolean mLoadLibrarySuccess = false;

    static {
        try {
            mLoadLibrarySuccess = SystemUtil.loadIMLibrary();
        } catch (Exception e) {
            Log.e(TAG, e.toString());
        }
    }

    private String mStringUIPlatform = "";
    private int mNumberUIPlatform = 0;
    private int mApplicationID = 0;

    private boolean isInit = false;
    private boolean isTestEnvironment = false;
    private boolean enableQuic = false;
    private boolean isIPv6Prior = false;
    private boolean disableHttpRequest = false;

    private UserPreference userPreference = new UserPreference();

    private static class BaseManagerHolder {
        private static final BaseManager baseManager = new BaseManager();
    }

    public static BaseManager getInstance() {
        return BaseManager.BaseManagerHolder.baseManager;
    }

    public boolean initSDK(Context context, SDKConfig sdkConfig, boolean needLogCallback, SDKListener listener) {
        if (false == mLoadLibrarySuccess) {
            IMLog.e(TAG, "libImSDK.so is not loaded");
            return false;
        }

        if (sdkConfig.sdkAppId <= 0) {
            Log.e(TAG, "invalid sdkAppID:" + sdkConfig.sdkAppId);
            return false;
        }

        if (null == context) {
            Log.e(TAG, "context is null");
            return false;
        }

        if (isInit) {
            IMLog.w(TAG, "Has initSDK");
            return true;
        }

        IMContext.getInstance().init(context.getApplicationContext());
        NetworkInfoCenter.getInstance().init(context.getApplicationContext(), this);

        sdkConfig.sdkInitPath = SystemUtil.getSDKInitPath();
        sdkConfig.sdkInstanceType = SystemUtil.getInstanceType();
        sdkConfig.isTestEnvironment = isTestEnvironment;
        sdkConfig.enableQuic = enableQuic;
        sdkConfig.isIPv6Prior = isIPv6Prior;
        sdkConfig.deviceInfo.deviceType = SystemUtil.getDeviceType();
        sdkConfig.deviceInfo.deviceBrand = SystemUtil.getInstanceType();
        sdkConfig.deviceInfo.systemVersion = SystemUtil.getSystemVersion();
        sdkConfig.networkInfo.networkType = NetworkInfoCenter.getInstance().getNetworkType();
        sdkConfig.networkInfo.ipType = NetworkInfoCenter.getInstance().getIPType();
        sdkConfig.networkInfo.networkId = NetworkInfoCenter.getInstance().getNetworkID();
        sdkConfig.networkInfo.wifiNetworkHandle = NetworkInfoCenter.getInstance().getWifiNetworkHandle();
        sdkConfig.networkInfo.xgNetworkHandle = NetworkInfoCenter.getInstance().getXgNetworkHandle();
        sdkConfig.networkInfo.initializeCostTime = NetworkInfoCenter.getInstance().getInitializeCostTime();
        sdkConfig.networkInfo.networkConnected = NetworkInfoCenter.getInstance().isNetworkConnected();
        sdkConfig.proxyInfo = mProxyInfo;
        sdkConfig.databaseEncryptInfo = mDatabaseEncryptInfo;
        sdkConfig.packetRetryInfo = mPacketRetryInfo;
        sdkConfig.logSetting.enableConsoleLog = true;
        sdkConfig.logSetting.logFilePath = SystemUtil.getSDKLogPath();
        sdkConfig.stringUIPlatform = mStringUIPlatform;
        sdkConfig.numberUIPlatform = getUIPlatform();
        sdkConfig.packageName = context.getPackageName();
        sdkConfig.appName = SystemUtil.getAppName();
        sdkConfig.applicationID = mApplicationID;
        sdkConfig.disableHttpRequest = disableHttpRequest;

        mLastNetworkInfo = sdkConfig.networkInfo;

        nativeInitSDK(sdkConfig, needLogCallback, listener);

        MessageCenter.getInstance().init();
        GroupManager.getInstance().init();
        ConversationManager.getInstance().init();
        RelationshipManager.getInstance().init();
        SignalingManager.getInstance().init();
        CommunityManager.getInstance().init();

        sdkListenerWeakReference = new WeakReference<>(listener);

        reportTUIComponentUsage();

        isInit = true;
        return true;
    }

    public void unInitSDK(boolean needReleaseIMCore) {
        if (needReleaseIMCore) {
            nativeUninitSDK();
        }
        mStringUIPlatform = "";
        mNumberUIPlatform = 0;
        isInit = false;
        isTestEnvironment = false;
        enableQuic = false;
        isIPv6Prior = false;
        mLastNetworkInfo.clean();
        mProxyInfo.clean();
        mDatabaseEncryptInfo.clean();
        mPacketRetryInfo.clean();
        mInvokeFromTUIKit = false;
        mInvokeFromTUICore = false;
        mTUIComponentMap.clear();
        mTUIComponentCheckCountMap.clear();
    }

    private int getUIPlatform() {
        if (mNumberUIPlatform != UI_PLATFORM_UNKNOWN) {
            return mNumberUIPlatform;
        }

        mInvokeFromTUIKit = isTUIKit();
        boolean hasFlutter = isFlutter();
        boolean hasUnity = isUnity();
        if (hasFlutter) {
            if (mInvokeFromTUIKit) {
                return UI_PLATFORM_FLUTTER_UIKIT;
            } else {
                return UI_PLATFORM_FLUTTER;
            }
        } else if (hasUnity) {
            if (mInvokeFromTUIKit) {
                return UI_PLATFORM_UNITY_UIKIT;
            } else {
                return UI_PLATFORM_UNITY;
            }
        } else if (hasAtomicXCore()) {
            return UI_PLATFORM_ATOMICXCORE;
        } else if (mInvokeFromTUIKit) {
            boolean isRTL = false;
            if (SystemUtil.getSDKVersion() >= Build.VERSION_CODES.KITKAT) {
                Context context = IMContext.getInstance().getApplicationContext();
                isRTL = context.getResources().getConfiguration().getLayoutDirection() == LayoutDirection.RTL;
            }
            if (hasTUIKitMinimalist()) {
                if (isRTL) {
                    return UI_PLATFORM_TUIKIT_MINIMALIST_RTL;
                } else {
                    return UI_PLATFORM_TUIKIT_MINIMALIST;
                }
            } else {
                if (isRTL) {
                    return UI_PLATFORM_TUIKIT_RTL;
                } else {
                    return UI_PLATFORM_TUIKIT;
                }
            }
        }

        return UI_PLATFORM_UNKNOWN;
    }

    private boolean isTUIKit() {
        try {
            Class classTUIKit = Class.forName("com.tencent.qcloud.tim.uikit.TUIKit");
            if (classTUIKit != null) {
                return true;
            }
        } catch (Exception e) {
            // 不含 TUIKit
        }

        try {
            Class classTUICore = Class.forName("com.tencent.qcloud.tuicore.TUICore");
            if (classTUICore != null) {
                return true;
            }
        } catch (ClassNotFoundException e) {
            // 不含 TUIKit
        }

        // 如果包名被修改，再判断下调用栈的是否包含类的关键字
        StackTraceElement[] stacks = new Throwable().getStackTrace();
        String callName = "";
        for (int i = 0; i < stacks.length; i++) {
            if (i > 15) {
                return false;
            }
            callName = stacks[i].getClassName();
            String lowerCaseCallName = callName.toLowerCase();
            if (lowerCaseCallName.contains("tuikitimpl") || lowerCaseCallName.contains("tuicore")) {
                return true;
            }
        }

        return false;
    }

    private boolean isTUICore() {
        try {
            Class classTUICore = Class.forName("com.tencent.qcloud.tuicore.TUICore");
            if (classTUICore != null) {
                return true;
            }
        } catch (ClassNotFoundException e) {
            // 不含 TUIKit
        }

        // 如果包名被修改，再判断下调用栈的是否包含类的关键字
        StackTraceElement[] stacks = new Throwable().getStackTrace();
        String callName = "";
        for (int i = 0; i < stacks.length; i++) {
            if (i > 15) {
                return false;
            }
            callName = stacks[i].getClassName();
            String lowerCaseCallName = callName.toLowerCase();
            if (lowerCaseCallName.contains("tuicore")) {
                return true;
            }
        }

        return false;
    }

    private boolean isFlutter() {
        try {
            Class c = Class.forName("com.qq.qcloud.tencent_im_sdk_plugin.tencent_im_sdk_plugin");
            if (c != null) {
                return true;
            }
        } catch (Exception e) {
            // 不含 Flutter
        }
        return false;
    }

    private boolean isUnity() {
        try {
            Class c = Class.forName("com.qcloud.tencentimsdk.TencentImSDKPluginUnity");
            if (c != null) {
                return true;
            }
        } catch (Exception e) {
            // 不含 Unity
        }
        return false;
    }

    private boolean hasAtomicXCore() {
        try {
            Class c = Class.forName("io.trtc.tuikit.atomicxcore.api.LoginStore");
            if (c != null) {
                return true;
            }
        } catch (Exception e) {
            // 不含 AtomicXCore
        }
        return false;
    }

    /**
     * SDK 启动时通过反射上报基础组件使用情况
     */
    private void reportTUIComponentUsage() {
        mInvokeFromTUICore = isTUICore();

        mTUIComponentCheckCountMap.put(TUI_COMPONENT_CONVERSATION, 0);
        mTUIComponentCheckCountMap.put(TUI_COMPONENT_CHAT, 0);
        mTUIComponentCheckCountMap.put(TUI_COMPONENT_CONTACT, 0);
        mTUIComponentCheckCountMap.put(TUI_COMPONENT_SEARCH, 0);
        mTUIComponentCheckCountMap.put(TUI_COMPONENT_COMMUNITY, 0);
        mTUIComponentCheckCountMap.put(TUI_PLUGIN_CUSTOMER_SERVICE, 0);
        mTUIComponentCheckCountMap.put(TUI_PLUGIN_CHAT_BOT, 0);

        if (mInvokeFromTUIKit) {
            if (mInvokeFromTUICore) {
                mTUIComponentMap.put(TUI_COMPONENT_CORE, TUI_STYLE_TYPE_CLASSIC);
                if (hasTUIConversation() && !mTUIComponentMap.containsKey(TUI_COMPONENT_CONVERSATION)) {
                    mTUIComponentMap.put(TUI_COMPONENT_CONVERSATION, getTUIConversationStyle());
                }

                if (hasTUIChat() && !mTUIComponentMap.containsKey(TUI_COMPONENT_CHAT)) {
                    mTUIComponentMap.put(TUI_COMPONENT_CHAT, getTUIChatStyle());
                }

                if (hasTUIContact() && !mTUIComponentMap.containsKey(TUI_COMPONENT_CONTACT)) {
                    mTUIComponentMap.put(TUI_COMPONENT_CONTACT, getTUIContactStyle());
                }

                if (hasTUISearch() && !mTUIComponentMap.containsKey(TUI_COMPONENT_SEARCH)) {
                    mTUIComponentMap.put(TUI_COMPONENT_SEARCH, getTUISearchStyle());
                }

                if (hasTUICommunity() && !mTUIComponentMap.containsKey(TUI_COMPONENT_COMMUNITY)) {
                    mTUIComponentMap.put(TUI_COMPONENT_COMMUNITY, TUI_STYLE_TYPE_CLASSIC);
                }
            } else {
                mTUIComponentMap.put(TUI_COMPONENT_UNKNOWN, TUI_STYLE_TYPE_CLASSIC);
            }

            nativeReportTUIComponentUsage(mTUIComponentMap);
        }
    }

    /**
     * 对【闭源】插件、flutter 等提供的组件上报接口
     */
    public void reportTUIComponentUsage(long componentType, long styleType) {
        Map<Long, Long> componentMap = new HashMap<>();
        componentMap.put(componentType, styleType);

        nativeReportTUIComponentUsage(componentMap);
    }

    /**
     * 通过接口调用堆栈检查是否集成了 TUI 【开源】组件并上报
     */
    public void reportTUIComponentUsage(long componentType) {
        if (!checkReportStatus(componentType)) {
            return;
        }

        String keyCallName = "";
        boolean needFindStyleType = true;
        if (componentType == TUI_COMPONENT_CONVERSATION) {
            keyCallName = "conversationprovider";
        } else if (componentType == TUI_COMPONENT_CHAT) {
            keyCallName = "chatprovider";
        } else if (componentType == TUI_COMPONENT_CONTACT) {
            keyCallName = "contactprovider";
        } else if (componentType == TUI_COMPONENT_SEARCH) {
            keyCallName = "searchdataprovider";
        } else if (componentType == TUI_COMPONENT_COMMUNITY) {
            keyCallName = "communityprovider";
            needFindStyleType = false;
        } else {
            IMLog.e(TAG, "unknown tui component type:" + componentType);
            return;
        }

        reportTUIComponentUsageFromStackTrace(keyCallName, componentType, needFindStyleType);
    }

    /**
     * 上报 TUI 组件功能点使用情况
     */
    public void reportTUIFeatureUsage(long componentType, long featureType) {
        nativeReportTUIFeatureUsage(componentType, featureType);
    }

    /**
     * 通过接口调用堆栈检查是否集成了 TUI 【开源】插件并上报
     */
    public void reportTUIPluginUsage(List<Long> pluginTypeList) {
        for (Long pluginType : pluginTypeList) {
            if (!checkReportStatus(pluginType)) {
                continue;
            }

            String keyCallName = "";
            boolean needFindStyleType = true;
            if (pluginType == TUI_PLUGIN_CUSTOMER_SERVICE) {
                keyCallName = "customerservicepresenter";
                needFindStyleType = false;
            } else if (pluginType == TUI_PLUGIN_CHAT_BOT) {
                keyCallName = "chatbotpresenter";
                needFindStyleType = false;
            } else {
                IMLog.e(TAG, "unknown tui plugin type:" + pluginType);
                return;
            }

            // 一次上报只会由一个插件触发，这里找到插件后就上报 return
            if (reportTUIComponentUsageFromStackTrace(keyCallName, pluginType, needFindStyleType)) {
                return;
            }
        }
    }

    private boolean checkReportStatus(long componentType) {
        if (!isInit) {
            return false;
        }

        // 没有使用 TUICore 不上报
        if (!mInvokeFromTUICore) {
            return false;
        }

        // 已经上报过，不再上报
        if (mTUIComponentMap.containsKey(componentType)) {
            return false;
        }

        // 没有初始化组件检测计数，不再上报
        if (!mTUIComponentCheckCountMap.containsKey(componentType)) {
            return false;
        }

        // 尝试上报次数超限，不再上报
        int checkCount = mTUIComponentCheckCountMap.get(componentType);
        if (checkCount >= TUI_COMPONENT_CHECK_COUNT_LIMIT) {
            return false;
        }

        checkCount++;
        mTUIComponentCheckCountMap.put(componentType, checkCount);

        return true;
    }

    private boolean reportTUIComponentUsageFromStackTrace(
        String keyCallName, long componentType, boolean needFindStyleType) {
        boolean reportSuccess = false;
        String minimalistUIName = "minimalist";
        String classicUIName = "classic";

        StackTraceElement[] stacks = new Throwable().getStackTrace();
        String callName = "";
        boolean hasFoundKeyCallName = false;
        boolean hasFoundStyleType = false;
        long styleType = TUI_STYLE_TYPE_CLASSIC;
        // 限制调用栈查找层级
        for (int i = 0; i < stacks.length; i++) {
            callName = stacks[i].getClassName();
            String lowerCaseCallName = callName.toLowerCase();
            if (!hasFoundKeyCallName) {
                if (lowerCaseCallName.contains(keyCallName)) {
                    hasFoundKeyCallName = true;
                }
            }

            if (needFindStyleType && !hasFoundStyleType) {
                if (lowerCaseCallName.contains(classicUIName)) {
                    hasFoundStyleType = true;
                } else if (lowerCaseCallName.contains(minimalistUIName)) {
                    hasFoundStyleType = true;
                    styleType = TUI_STYLE_TYPE_MINIMALIST;
                }
            }

            if (hasFoundKeyCallName && (!needFindStyleType || hasFoundStyleType)) {
                mTUIComponentMap.put(componentType, styleType);
                nativeReportTUIComponentUsage(mTUIComponentMap);
                reportSuccess = true;
                break;
            }

            if (i >= TUI_COMPONENT_STACK_LAYER_LIMIT) {
                if (hasFoundKeyCallName) {
                    mTUIComponentMap.put(componentType, styleType);
                    nativeReportTUIComponentUsage(mTUIComponentMap);
                    reportSuccess = true;
                }
                break;
            }
        }

        return reportSuccess;
    }

    private boolean hasTUIKitMinimalist() {
        return getTUIConversationStyle() == TUI_STYLE_TYPE_MINIMALIST || getTUIChatStyle() == TUI_STYLE_TYPE_MINIMALIST
            || getTUIContactStyle() == TUI_STYLE_TYPE_MINIMALIST || getTUISearchStyle() == TUI_STYLE_TYPE_MINIMALIST;
    }

    private boolean hasTUIConversation() {
        try {
            Class c = Class.forName("com.tencent.qcloud.tuikit.tuiconversation.model.ConversationProvider");
            if (c != null) {
                return true;
            }
        } catch (Exception e) {
            // 不含 conversation
        }
        return false;
    }

    private long getTUIConversationStyle() {
        try {
            Class c =
                Class.forName("com.tencent.qcloud.tuikit.tuiconversation.minimalistui.widget.ConversationListLayout");
            if (c != null) {
                return TUI_STYLE_TYPE_MINIMALIST;
            }
        } catch (Exception e) {
            // 不是 Minimalist UI
        }
        return TUI_STYLE_TYPE_CLASSIC;
    }

    private boolean hasTUIChat() {
        try {
            Class c = Class.forName("com.tencent.qcloud.tuikit.tuichat.model.ChatProvider");
            if (c != null) {
                return true;
            }
        } catch (Exception e) {
            // 不含 chat
        }
        return false;
    }

    private long getTUIChatStyle() {
        try {
            Class c = Class.forName("com.tencent.qcloud.tuikit.tuichat.minimalistui.MinimalistUIService");
            if (c != null) {
                return TUI_STYLE_TYPE_MINIMALIST;
            }
        } catch (Exception e) {
            // 不是 Minimalist UI
        }
        return TUI_STYLE_TYPE_CLASSIC;
    }

    private boolean hasTUIContact() {
        try {
            Class c = Class.forName("com.tencent.qcloud.tuikit.tuicontact.model.ContactProvider");
            if (c != null) {
                return true;
            }
        } catch (Exception e) {
            // 不含 contact
        }
        return false;
    }

    private long getTUIContactStyle() {
        try {
            Class c = Class.forName("com.tencent.qcloud.tuikit.tuicontact.minimalistui.widget.FriendProfileLayout");
            if (c != null) {
                return TUI_STYLE_TYPE_MINIMALIST;
            }
        } catch (Exception e) {
            // 不是 Minimalist UI
        }
        return TUI_STYLE_TYPE_CLASSIC;
    }

    private boolean hasTUISearch() {
        try {
            Class c = Class.forName("com.tencent.qcloud.tuikit.tuisearch.model.SearchDataProvider");
            if (c != null) {
                return true;
            }
        } catch (Exception e) {
            // 不含 search
        }
        return false;
    }

    private long getTUISearchStyle() {
        try {
            Class c =
                Class.forName("com.tencent.qcloud.tuikit.tuisearch.minimalistui.page.SearchMainMinimalistActivity");
            if (c != null) {
                return TUI_STYLE_TYPE_MINIMALIST;
            }
        } catch (Exception e) {
            // 不是 Minimalist UI
        }
        return TUI_STYLE_TYPE_CLASSIC;
    }

    private boolean hasTUICommunity() {
        try {
            Class c = Class.forName("com.tencent.qcloud.tuikit.tuicommunity.model.CommunityProvider");
            if (c != null) {
                return true;
            }
        } catch (Exception e) {
            // 不含 community
        }
        return false;
    }

    public void login(final String userID, final String userSig, final IMCallback callback, final int accountType) {
        if (!isInit) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }
        nativeLogin(userID, userSig, callback, accountType);
    }

    public void logout(final IMCallback callback) {
        if (!isInit) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }
        nativeLogout(callback);
    }

    public boolean setLibraryPath(String libraryPath) {
        mLoadLibrarySuccess = SystemUtil.loadIMLibrary(libraryPath);
        return mLoadLibrarySuccess;
    }

    public void downloadLibrary(Context context, IMCallback callback) {
        LibraryDownloader.getInstance().downloadLibrary(context, callback);
    }

    public void setCustomUIPlatform(String stringUIPlatform, int numberUIPlatform) {
        mStringUIPlatform = stringUIPlatform;
        mNumberUIPlatform = numberUIPlatform;
    }

    public void setTestEnvironment(boolean testEnvironment) {
        isTestEnvironment = testEnvironment;
    }

    public void setQuicEnabled(boolean enabled) {
        enableQuic = enabled;
    }

    public void setIPv6Prior(boolean prior) {
        isIPv6Prior = prior;
    }

    public void setCustomServerInfo(CustomServerInfo customServerInfo) {
        nativeSetCustomServerInfo(customServerInfo);
    }

    public void setProxyInfo(SDKConfig.ProxyInfo proxyInfo) {
        mProxyInfo = proxyInfo;
    }

    public void setDatabaseEncryptInfo(SDKConfig.DatabaseEncryptInfo databaseEncryptInfo) {
        mDatabaseEncryptInfo = databaseEncryptInfo;
    }

    public void isCommercialAbilityEnabled(long commercialAbilityKey, IMCallback<Object> callback) {
        nativeIsCommercialAbilityEnabled(commercialAbilityKey, callback);
    }

    public void setPacketRetryInfo(SDKConfig.PacketRetryInfo info) {
        mPacketRetryInfo = info;
    }

    public void setApplicationID(int applicationID) {
        mApplicationID = applicationID;
    }

    public void disableHttpRequest(boolean disable) {
        disableHttpRequest = disable;
    }

    public String getLoginUser() {
        if (!isInit) {
            Log.e(TAG, "sdk not init");
            return null;
        }
        return nativeGetLoginUser();
    }

    public int getLoginStatus() {
        if (!isInit) {
            Log.e(TAG, "sdk not init");
            return 3; // 无登录状态
        }
        return nativeGetLoginStatus();
    }

    public void getLoginAccountType(IMCallback callback) {
        if (!isInit) {
            if (callback != null) {
                callback.success(ACCOUNT_TYPE_UNKOWN);
            }
            return;
        }

        if (getLoginStatus() != 1) {
            if (callback != null) {
                callback.success(ACCOUNT_TYPE_UNKOWN);
            }
            return;
        }

        nativeGetLoginAccountType(callback);
    }

    public String getVersion() {
        if (!isInit) {
            Log.e(TAG, "sdk not init");
            return null;
        }
        return nativeGetSDKVersion();
    }

    public boolean isInited() {
        return isInit;
    }

    public long getClockTickInHz() {
        if (!isInit) {
            Log.e(TAG, "sdk not init");
            return 0;
        }
        return nativeGetClockTickInHz();
    }

    public long getTimeTick() {
        if (!isInit) {
            Log.e(TAG, "sdk not init");
            return 0;
        }
        return nativeGetTimeTick();
    }

    public long getServerTime() {
        if (!isInit) {
            Log.e(TAG, "sdk not init");
            return 0;
        }
        return nativeGetServerTime();
    }

    public void initLocalStorage(String userID, IMCallback callback) {
        if (!isInit) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }

        nativeInitLocalStorage(userID, callback);
    }

    @Override
    public void onNetworkChange(boolean connected, int networkType, int ipType, String networkID,
        long wifiNetworkHandle, long xgNetworkHandle, long initializeCostTime) {
        if (connected == mLastNetworkInfo.networkConnected && networkType == mLastNetworkInfo.networkType
            && ipType == mLastNetworkInfo.ipType
            && (null != networkID && networkID.equals(mLastNetworkInfo.networkId))) {
            Log.w(TAG, "onNetworkChange, networkinfo is same");
            return;
        }

        mLastNetworkInfo.networkConnected = connected;
        mLastNetworkInfo.networkType = networkType;
        mLastNetworkInfo.ipType = ipType;
        mLastNetworkInfo.networkId = networkID;
        mLastNetworkInfo.initializeCostTime = initializeCostTime;
        nativeNotifyNetworkChange(
            connected, networkType, ipType, networkID, wifiNetworkHandle, xgNetworkHandle, initializeCostTime);
    }

    @Override
    public void onScreenStatusChange(boolean isScreenOn) {
        nativeNotifyScreenStatusChange(isScreenOn);
    }

    public void notifySelfInfoUpdated(UserInfo selfInfo) {
        if (sdkListenerWeakReference != null) {
            SDKListener listener = sdkListenerWeakReference.get();
            if (listener != null) {
                listener.onSelfInfoUpdated(selfInfo);
            }
        }
    }

    public void notifyUserStatusChanged(List<UserStatus> userStatusList) {
        if (sdkListenerWeakReference != null) {
            SDKListener listener = sdkListenerWeakReference.get();
            if (listener != null) {
                listener.onUserStatusChanged(userStatusList);
            }
        }
    }

    public void notifyUserInfoChanged(List<UserInfo> userInfoList) {
        if (sdkListenerWeakReference != null) {
            SDKListener listener = sdkListenerWeakReference.get();
            if (listener != null) {
                listener.onUserInfoChanged(userInfoList);
            }
        }
    }

    public void notifyAllReceiveMessageOptChanged(ReceiveMessageOptInfo option) {
        if (sdkListenerWeakReference != null) {
            SDKListener listener = sdkListenerWeakReference.get();
            if (listener != null) {
                listener.onAllReceiveMessageOptChanged(option);
            }
        }
    }

    public void notifyExperimentalNotify(String key, Object param) {
        if (sdkListenerWeakReference != null) {
            SDKListener listener = sdkListenerWeakReference.get();
            if (listener != null) {
                listener.onExperimentalNotify(key, param);
            }
        }
    }

    public void enableSignaling(boolean enable) {
        userPreference.setEnableSignaling(enable);
        nativeSetUserPreference(userPreference);
    }

    public void getAIDenoiseSignature(IMCallback callback) {
        if (!isInit) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }

        nativeGetAIDenoiseSignature(callback);
    }

    public void getVideoEditSignature(IMCallback callback) {
        if (!isInit) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }

        nativeGetVideoEditSignature(callback);
    }

    public void sendTRTCCustomData(byte[] data, IMCallback callback) {
        if (!isInit) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }

        nativeSendTRTCCustomData(data, callback);
    }

    public void getBriefGroupMemberList(final String groupID, final IMCallback<List<GroupMemberInfo>> callback) {
        if (!isInit) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }
        nativeGetBriefGroupMemberList(groupID, callback);
    }

    public void reportPushSDKEvent(long eventID, long eventCode, long eventResult, String eventMessage,
        String moreMessage, String extensionMessage, IMCallback callback) {
        nativeReportPushSDKEvent(
            eventID, eventCode, eventResult, eventMessage, moreMessage, extensionMessage, callback);
    }

    public void setCustomLoginInfo(String userID, String customLoginInfo, IMCallback callback) {
        nativeSetCustomLoginInfo(userID, customLoginInfo);

        if (callback != null) {
            callback.success(0);
        }
    }

    public long getMainLooperPointer() {
        return nativeGetMainLooperPointer();
    }

    public void getGroupMessageReceipt(String groupID, Map<Integer, List<String>> requestMessageReadInfo,
        final IMCallback<Map<Integer, Map<Integer, List<String>>>> callback) {
        if (!isInit) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }

        nativeGetGroupMessageReceipt(groupID, requestMessageReadInfo, callback);
    }

    public void setForceRequestCloudMessage(boolean forceRequestCloudMessage) {
        nativeSetForceRequestCloudMessage(forceRequestCloudMessage);
    }

    ///////////////////////////////////////////////////////////////////////////////////////////////////////////////////

    protected native void nativeInitSDK(SDKConfig sdkConfig, boolean needLogCallback, SDKListener sdkListener);

    protected native void nativeUninitSDK();

    protected native void nativeNotifyNetworkChange(boolean connected, int networkType, int ipType, String networkID,
        long wifiNetworkHandle, long xgNetworkHandle, long initializeCostTime);

    protected native void nativeNotifyScreenStatusChange(boolean isScreenOn);

    protected native void nativeLogin(String identifier, String userSig, IMCallback callBack, int accountType);

    protected native void nativeLogout(IMCallback callBack);

    protected native void nativeSetCustomServerInfo(CustomServerInfo customServerInfo);

    protected native void nativeIsCommercialAbilityEnabled(long commercialAbilityKey, IMCallback callBack);

    protected native String nativeGetLoginUser();

    protected native int nativeGetLoginStatus();

    protected native void nativeGetLoginAccountType(IMCallback callBack);

    protected native String nativeGetSDKVersion();

    protected native long nativeGetClockTickInHz();

    protected native long nativeGetTimeTick();

    protected native long nativeGetServerTime();

    protected native void nativeInitLocalStorage(String identifier, IMCallback callBack);

    protected native void nativeReportTUIComponentUsage(Map<Long, Long> tuiComponentMap);

    protected native void nativeReportTUIFeatureUsage(long componentType, long featureType);

    protected native void nativeSetUserPreference(UserPreference userPreference);

    protected native void nativeGetAIDenoiseSignature(IMCallback callBack);

    protected native void nativeGetVideoEditSignature(IMCallback callBack);

    protected native void nativeSendTRTCCustomData(byte[] data, IMCallback callBack);

    protected native void nativeGetBriefGroupMemberList(String groupID, IMCallback callBack);

    protected native void nativeReportPushSDKEvent(long eventID, long eventCode, long eventResult, String eventMessage,
        String moreMessage, String extensionMessage, IMCallback callback);

    protected native void nativeSetCustomLoginInfo(String userID, String customLoginInfo);

    protected native long nativeGetMainLooperPointer();

    protected native void nativeGetGroupMessageReceipt(
        String groupID, Map<Integer, List<String>> requestMessageReadInfo, IMCallback callBack);

    protected native void nativeSetForceRequestCloudMessage(boolean forceRequestCloudMessage);
}
