package com.transsion.ad;

import android.app.Activity;
import android.app.Application;
import android.content.Context;
import android.os.Bundle;
import android.text.TextUtils;
import android.webkit.WebView;

import androidx.annotation.IntDef;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.google.android.ads.mediationtestsuite.MediationTestSuite;
import com.google.android.gms.ads.MobileAds;
import com.google.android.gms.ads.RequestConfiguration;
import com.google.android.gms.ads.initialization.AdapterStatus;
import com.transsion.ad.data.Scenes;
import com.transsion.ad.data.ScenesConfig;
import com.transsion.ad.util.AdEvent;
import com.transsion.ad.util.AdLog;

import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.atomic.AtomicInteger;

public class AdConfigManager {
    private static final String TAG = AdLog.COMMON_TAG + "AdConfigManager";

    public static final int AD_TYPE_NATIVE = 1;
    public static final int AD_TYPE_INTERSTITIAL = 2;
    public static final int AD_TYPE_BANNER = 3;
    public static final int AD_TYPE_OPEN = 4;
    public static final int AD_TYPE_REWARDED = 5;
    public static final int AD_TYPE_REWARDED_INTERSTITIAL = 6;

    private static boolean AD_SDK_INIT = false;
    private final ScenesConfig mAdScenesConfig = new ScenesConfig();
    private final ScenesConfig mBackupAdConfig = new ScenesConfig();
    private final Map<Integer, Scenes> mDefaultScenes = new HashMap<>();

    private final List<ActivityStartedInterface> mActivityStartedInterface = new ArrayList<>();
    private final List<ActivityStoppedInterface> mActivityStoppedInterface = new ArrayList<>();
    private final AtomicInteger foreground = new AtomicInteger();
    private final LinkedList<Activity> activityList = new LinkedList<>();
    private final LinkedList<Activity> adActivityList = new LinkedList<>();

    private AdConfigManager() {
    }


    @IntDef({NativeAdViewType.TYPE_NATIVE_AD_DEFAULT, NativeAdViewType.TYPE_NATIVE_AD_BIG,
            NativeAdViewType.TYPE_NATIVE_AD_SMALL})
    @Retention(RetentionPolicy.SOURCE)
    public @interface NativeAdViewType {
        int TYPE_NATIVE_AD_DEFAULT = 0;
        int TYPE_NATIVE_AD_BIG = 1;
        int TYPE_NATIVE_AD_SMALL = 2;
    }

    public static void init(Application application, @NonNull Builder builder) {
        AdLog.setLogPrint(builder.isLogDebug());
        AdEvent.getInstance().setAdEventCallBack(builder.getAdEventCallBack());
        AdEvent.getInstance().setAdjustEventCallBack(builder.getAdjustEventCallBack());
        Set<Map.Entry<Integer, Scenes>> sceneSet = builder.getDefaultAd().entrySet();
        for (Map.Entry<Integer, Scenes> scenesEntry : sceneSet) {
            getInstance().mDefaultScenes.put(scenesEntry.getKey(), scenesEntry.getValue());
        }
        application.registerActivityLifecycleCallbacks(new ActivityLifecycleCallbacks(builder));
        if (builder.isInitSdk()) {
            initSdk(application);
        }
        if (!TextUtils.isEmpty(builder.getTestDeviceIds())) {
            getInstance().setTestDeviceIds(builder.getTestDeviceIds());
        }
    }

    private static final class AdAdConfigManagerHolder {
        private static final AdConfigManager sInstance = new AdConfigManager();
    }

    public static AdConfigManager getInstance() {
        return AdAdConfigManagerHolder.sInstance;
    }

    public ScenesConfig getAdConfig(boolean backup) {
        return backup ? mAdScenesConfig : mBackupAdConfig;
    }

    public boolean sdkInitialized() {
        return AD_SDK_INIT;
    }

    @NonNull
    public Scenes getBackupScenes(int adType) {
        if (mBackupAdConfig.getScenes() == null) {
            return defaultBackupScenes(adType);
        }
        for (Scenes scenes : mBackupAdConfig.getScenes()) {
            if (scenes.getAdType() == adType) {
                scenes.setBackup(true);
                return scenes;
            }
        }
        return defaultBackupScenes(adType);
    }

    private Scenes defaultBackupScenes(int adType) {
        Scenes scenes = mDefaultScenes.get(adType);
        AdLog.i(TAG, "default backup scenes:" + scenes);
        if (scenes == null) {
            throw new NullPointerException("adId can not be null, must be set default adId by type");
        }
        return scenes;
    }

    @NonNull
    public Scenes getScenesById(int scenesId, int adType) {
        if (mAdScenesConfig.getScenes() == null) {
            Scenes scenes = new Scenes();
            scenes.setScenesId(-1);
            return scenes;
        }
        for (Scenes scenes : mAdScenesConfig.getScenes()) {
            if (scenes.getScenesId() == scenesId && scenes.getAdType() == adType && !TextUtils.isEmpty(scenes.getAdId())) {
                return scenes;
            }
        }
        Scenes scenes = new Scenes();
        scenes.setScenesId(-2);
        return scenes;
    }

    public void launchTestSuite(Context context) {
        MediationTestSuite.launch(context);
    }

    public void updateData(ScenesConfig newData, boolean backup) {
        if (backup) {
            mBackupAdConfig.setScenes(newData.getScenes());
        } else {
            mAdScenesConfig.setScenes(newData.getScenes());
        }
    }

    public static void initSdk(Context context) {
        if (AD_SDK_INIT) {
            return;
        }
        AdLog.i(TAG, "MobileAds init:" + MobileAds.getVersion());
        AD_SDK_INIT = true;
        final long time = System.currentTimeMillis();
        MobileAds.initialize(context, initializationStatus -> {
            Map<String, AdapterStatus> adapterStatus = initializationStatus.getAdapterStatusMap();
            Set<Map.Entry<String, AdapterStatus>> adapterStatusSet = adapterStatus.entrySet();
            StringBuilder sb = new StringBuilder();
            for (Map.Entry<String, AdapterStatus> statusEntry : adapterStatusSet) {
                String description = statusEntry.getValue().getDescription();
                String key = statusEntry.getKey();
                AdLog.i(TAG, "key:" + key + "; description:" + description);
                if (TextUtils.isEmpty(description)) {
                    continue;
                }
                if (TextUtils.isEmpty(key)) {
                    continue;
                }
                int lastIndex = key.lastIndexOf(".");
                sb.append(key.substring(lastIndex+1)).append(":").append(description);
            }
            String reason;
            if (sb.length() <= 0) {
                return;
            } else if (sb.length() > 100) {
                reason = sb.substring(0, 100);
            } else {
                reason = sb.toString();
            }
            AdLog.e(TAG, "ad init error:" + sb);
            Bundle bundle = new Bundle();
            bundle.putString(AdEvent.REASON, reason);
            bundle.putLong(AdEvent.DURATION, System.currentTimeMillis() - time);
            AdEvent.Event event = new AdEvent.Event();
            event.bundle = bundle;
            event.name = AdEvent.AD_INIT;
            AdEvent.getInstance().addEventInfo(event);
        });
    }

    public static void registerWebView(@NonNull WebView webView) {
        MobileAds.registerWebView(webView);
    }

    private void setTestDeviceIds(String deviceId) {
        List<String> testDeviceIds = Collections.singletonList(deviceId);
        RequestConfiguration configuration = new RequestConfiguration.Builder().setTestDeviceIds(testDeviceIds).build();
        MobileAds.setRequestConfiguration(configuration);
    }

    public void addActivityStartedObserver(ActivityStartedInterface activityStartedInterface) {
        mActivityStartedInterface.add(activityStartedInterface);
    }

    public void noticeActivityStartedObserver(@NonNull Activity activity) {
        for (ActivityStartedInterface startedInterface : mActivityStartedInterface) {
            startedInterface.onActivityStarted(activity);
        }
    }

    public void addActivityStoppedObserver(ActivityStoppedInterface activityStoppedInterface) {
        mActivityStoppedInterface.add(activityStoppedInterface);
    }

    public void noticeActivityStoppedObserver(@NonNull Activity activity) {
        for (ActivityStoppedInterface startedInterface : mActivityStoppedInterface) {
            startedInterface.onActivityStopped(activity);
        }
    }

    public boolean isAppForeground() {
        return foreground.get() > 0;
    }

    private void incrementAppPage() {
        foreground.incrementAndGet();
    }

    private void decrementAppPage() {
        foreground.decrementAndGet();
    }

    private void resetForeground() {
        foreground.set(0);
    }

    public int getForegroundCount() {
        return foreground.get();
    }

    public Activity getValidActivity() {
        return activityList.peekLast();
    }

    public Activity getAdActivity() {
        return adActivityList.peek();
    }

    public List<Activity> getAdActivities() {
        return adActivityList;
    }

    private boolean isAdActivity(@NonNull Activity activity) {
        String localClassName = activity.getLocalClassName();
        if (TextUtils.isEmpty(localClassName)) {
            return false;
        }
        return localClassName.startsWith("com.google.android.gms.ads.AdActivity")
                || localClassName.startsWith("com.bytedance.sdk.openadsdk.activity")
                || localClassName.startsWith("com.applovin.adview")
                || localClassName.startsWith("com.facebook.ads");
    }

    private static class ActivityLifecycleCallbacks implements Application.ActivityLifecycleCallbacks {
        private final Builder builder;

        public ActivityLifecycleCallbacks(Builder builder) {
            this.builder = builder;
        }

        @Override
        public void onActivityCreated(@NonNull Activity activity, @Nullable Bundle savedInstanceState) {
            getInstance().activityList.add(activity);
            if (getInstance().isAdActivity(activity)) {
                getInstance().adActivityList.add(activity);
            }
        }

        @Override
        public void onActivityStarted(@NonNull Activity activity) {
            getInstance().incrementAppPage();
            getInstance().noticeActivityStartedObserver(activity);
        }

        @Override
        public void onActivityResumed(@NonNull Activity activity) {
        }

        @Override
        public void onActivityPaused(@NonNull Activity activity) {
        }

        @Override
        public void onActivityStopped(@NonNull Activity activity) {
            getInstance().decrementAppPage();
            getInstance().noticeActivityStoppedObserver(activity);
        }

        @Override
        public void onActivitySaveInstanceState(@NonNull Activity activity, @NonNull Bundle outState) {

        }

        @Override
        public void onActivityDestroyed(@NonNull Activity activity) {
            getInstance().activityList.remove(activity);
            if (getInstance().isAdActivity(activity)) {
                getInstance().adActivityList.remove(activity);
            }
            if (!getInstance().isAppForeground()) {
                getInstance().resetForeground();
            }
        }
    }

    public interface ActivityStartedInterface {
        void onActivityStarted(@NonNull Activity activity);
    }

    public interface ActivityStoppedInterface {
        void onActivityStopped(@NonNull Activity activity);
    }

    public static class Builder {
        private boolean logDebug;
        private boolean initSdk;
        private AdEvent.AdEventCallBack adEventCallBack;
        private AdEvent.AdjustEventCallBack adjustEventCallBack;
        private final Map<Integer, Scenes> defaultAd = new HashMap<>();
        private String deviceId;

        public boolean isLogDebug() {
            return logDebug;
        }

        public void setLogDebug(boolean logDebug) {
            this.logDebug = logDebug;
        }

        public boolean isInitSdk() {
            return initSdk;
        }

        /**
         * 设置是否初始化广告
         *
         * @param initSdk 如果没有授权或者没有网络时，可以设置为false，此时只会初始化log，埋点接口等不涉及权限的初始化，
         *                等授权后可单独调用initSdk(Context)初始化
         */
        public void setInitSdk(boolean initSdk) {
            this.initSdk = initSdk;
        }

        public void addDefaultScenes(String adId, int adType) {
            Scenes scenes = new Scenes();
            scenes.setScenesId(0);
            scenes.setAdType(adType);
            scenes.setBackup(true);
            scenes.setAdId(adId);
            defaultAd.put(adType, scenes);
        }

        public Map<Integer, Scenes> getDefaultAd() {
            return defaultAd;
        }


        public AdEvent.AdEventCallBack getAdEventCallBack(){
            return adEventCallBack;
        }

        /**
         * 埋点上报回调接口
         */
        public void setAdEventCallBack(AdEvent.AdEventCallBack adEventCallBack) {
            this.adEventCallBack = adEventCallBack;
        }

        public AdEvent.AdjustEventCallBack getAdjustEventCallBack() {
            return adjustEventCallBack;
        }

        /**
         * 归因事件回调接口
         */
        public void setAdjustEventCallBack(AdEvent.AdjustEventCallBack adjustEventCallBack) {
            this.adjustEventCallBack = adjustEventCallBack;
        }

        /**
         * 1.Load your ads-integrated app and make an ad request.
         * 2.Check the logcat output for a message that looks like the one below, which shows you your device ID and how to add it as a test device:
         * I/Ads: Use RequestConfiguration.Builder.setTestDeviceIds(Collections.singletonList("33BE2250B43518CCDA7DE426D04EE231"))to get test ads on this device."
         *
         * @param deviceId device ID
         */
        public void setTestDeviceIds(String deviceId) {
            this.deviceId = deviceId;
        }

        public String getTestDeviceIds() {
            return this.deviceId;
        }
    }
}
