package com.transsion.ad.nativead;

import android.content.Context;
import android.os.Bundle;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.FrameLayout;
import android.widget.ImageView;
import android.widget.TextView;

import androidx.annotation.NonNull;

import com.bumptech.glide.Glide;
import com.google.android.gms.ads.AdListener;
import com.google.android.gms.ads.AdLoader;
import com.google.android.gms.ads.AdRequest;
import com.google.android.gms.ads.AdapterResponseInfo;
import com.google.android.gms.ads.LoadAdError;
import com.google.android.gms.ads.VideoController;
import com.google.android.gms.ads.formats.AdManagerAdViewOptions;
import com.google.android.gms.ads.nativead.MediaView;
import com.google.android.gms.ads.nativead.NativeAd;
import com.google.android.gms.ads.nativead.NativeAdOptions;
import com.google.android.gms.ads.nativead.NativeAdView;
import com.google.android.gms.ads.nativead.NativeCustomFormatAd;
import com.transsion.ad.AdConfigManager;
import com.transsion.ad.R;
import com.transsion.ad.data.Scenes;
import com.transsion.ad.sdk.AbstractAdLoader;
import com.transsion.ad.sdk.LoaderCache;
import com.transsion.ad.sdk.wrap.AdErrorWrapper;
import com.transsion.ad.sdk.wrap.AdValueWrapper;
import com.transsion.ad.sdk.wrap.NativeAdWrapper;
import com.transsion.ad.sdk.wrap.NativeCustomFormatAdWrapper;
import com.transsion.ad.util.AdEvent;
import com.transsion.ad.util.AdLog;

public class NativeAdLoader extends AbstractAdLoader<NativeAdListener> {
    private static final String SIMPLE_TEMPLATE_ID = "10104090";
    private NativeAdOptions adOptions;
    private AdManagerAdViewOptions adViewOptions;
    private NativeCustomFormatAd nativeCustomFormatAd;
    private boolean requestNativeAd;
    private boolean requestCustomFormatAd;
    private int maxNumberOfAds;
    private NativeAd nativeAd;

    public NativeAdLoader(Scenes scenes) {
        super(scenes);
    }

    @Override
    public void loadAd(NativeAdListener listener) {
        this.listener = listener;
        load(false, true, false, false, 0);
    }

    @Override
    public void preloadAd(boolean force, NativeAdListener listener) {
        this.listener = listener;
        load(force, true, false, true, 0);
    }

    /**
     * 方法目前仅适用于 AdMob 广告。对于参与中介的广告，请改为使用 loadAd()
     *
     * @param listener       广告回调
     * @param maxNumberOfAds 针对多个广告（最多 5 个）发送请求
     */
    public void loadAds(NativeAdListener listener, int maxNumberOfAds) {
        this.listener = listener;
        load(false, true, false, false, maxNumberOfAds);
    }

    public void loadCustomAd(NativeAdListener listener) {
        this.listener = listener;
        load(false, false, true, false, 0);
    }

    private void load(boolean force, boolean requestNativeAd, boolean requestCustomFormatAd, boolean preload, int maxNumberOfAds) {
        AdLog.i(TAG, "native ad; requestNativeAd = " + requestNativeAd + " preload = " + preload + " maxNumberOfAds = " + maxNumberOfAds);
        this.requestNativeAd = requestNativeAd;
        this.requestCustomFormatAd = requestCustomFormatAd;
        this.maxNumberOfAds = maxNumberOfAds;
        load(force, preload);
    }

    @Override
    public boolean isExpired(boolean show) {
        return (nativeAd == null && nativeCustomFormatAd == null) || super.isExpired(show);
    }

    @Override
    protected void cacheLoaded() {
        if (listener != null) {
            listener.onLoaded(NativeAdLoader.this);
        }
    }

    @Override
    protected boolean noCacheHandled() {
        if (listener != null) {
            return listener.onCacheInvalid(NativeAdLoader.this);
        }
        return super.noCacheHandled();
    }

    @Override
    protected void loadRequest() {
        Context context = getContext();
        if (context == null) {
            AdLog.e(TAG, "end load, context is null");
            return;
        }
        AdLoader.Builder builder = new AdLoader.Builder(context.getApplicationContext(), scenes.getAdId());
        if (requestNativeAd) {
            nativeAdLoad(builder);
        }
        if (requestCustomFormatAd) {
            customFormatAdLoad(builder);
        }
        withAdListener(builder);
        if (adOptions != null) {
            builder.withNativeAdOptions(adOptions);
        }
        if (adViewOptions != null) {
            builder.withAdManagerAdViewOptions(adViewOptions);
        }
        AdLoader adLoader = builder.build();
        AdRequest adRequest = new AdRequest.Builder()/*.setHttpTimeoutMillis(5000)*/.build();
        if (maxNumberOfAds > 0) {
            adLoader.loadAds(adRequest, maxNumberOfAds);
        } else {
            adLoader.loadAd(adRequest);
        }
    }

    private void nativeAdLoad(@NonNull AdLoader.Builder builder) {
        builder.forNativeAd(nativeAd -> {
            AdapterResponseInfo loadedResponseInfo = nativeAd.getResponseInfo() != null ? nativeAd.getResponseInfo().getLoadedAdapterResponseInfo() : null;
            NativeAdLoader.this.responseInfo = loadedResponseInfo != null ? loadedResponseInfo.toString() : null;
            String responseInfoStr = loadedResponseInfo != null ? loadedResponseInfo.getAdSourceName() : "loadedResponseInfo is null";
            NativeAdLoader.this.nativeAd = nativeAd;
            NativeAdLoader.this.time = System.currentTimeMillis();

            Bundle extras = nativeAd.getResponseInfo().getResponseExtras();
            final String mediationGroupName = extras.getString("mediation_group_name");
            final String sourceName = responseInfoStr;
            nativeAd.setOnPaidEventListener(adValue -> {
                AdLog.d(TAG, "paid event valueMicros:" + adValue.getValueMicros() + "; currencyCode:" + adValue.getCurrencyCode() + "; precisionType:" + adValue.getPrecisionType() + "; loader:" + NativeAdLoader.this);
                AdValueWrapper adValueWrapper = new AdValueWrapper(adValue);
                adValueWrapper.setAdRevenueUnit(getScenes().getAdId());
                adValueWrapper.setAdRevenueNetwork(sourceName);
                adValueWrapper.setAdRevenuePlacement(mediationGroupName);
                AdEvent.getInstance().sendAdRevenueInfoToAdjust(adValueWrapper);
            });
            LoaderCache.getInstance().putNativeAdLoaderInCache(NativeAdLoader.this, scenes.getAdId());
            if (listener != null) {
                listener.onNativeAdLoaded(NativeAdLoader.this, new NativeAdWrapper(nativeAd));
            }
        });
    }

    private void customFormatAdLoad(@NonNull AdLoader.Builder builder) {
        builder.forCustomFormatAd(SIMPLE_TEMPLATE_ID, nativeCustomFormatAd -> {
            NativeAdLoader.this.nativeCustomFormatAd = nativeCustomFormatAd;
            if (listener != null) {
                listener.onCustomFormatAdLoaded(NativeAdLoader.this, new NativeCustomFormatAdWrapper(nativeCustomFormatAd));
            }
        }, (nativeCustomFormatAd, s) -> {
            if (listener != null) {
                listener.onCustomClick(NativeAdLoader.this, new NativeCustomFormatAdWrapper(nativeCustomFormatAd), s);
            }
        });
    }

    private void withAdListener(@NonNull AdLoader.Builder builder) {
        builder.withAdListener(new AdListener() {
            @Override
            public void onAdClicked() {
                if (listener != null) {
                    listener.onClicked(NativeAdLoader.this);
                }
            }

            @Override
            public void onAdClosed() {
                if (listener != null) {
                    listener.onClosed(NativeAdLoader.this);
                }
            }

            @Override
            public void onAdFailedToLoad(@NonNull LoadAdError loadAdError) {
                if (listener != null) {
                    listener.onError(NativeAdLoader.this, new AdErrorWrapper(loadAdError));
                }
            }

            @Override
            public void onAdImpression() {
                if (listener != null) {
                    listener.onShow(NativeAdLoader.this);
                }
            }

            @Override
            public void onAdLoaded() {
                if (listener != null) {
                    listener.onLoaded(NativeAdLoader.this);
                }
            }

            @Override
            public void onAdOpened() {
                if (listener != null) {
                    listener.onOpened(NativeAdLoader.this);
                }
            }
        });
    }

    @Override
    public String getSubResponseInfo() {
        if (responseInfo != null && responseInfo.length() > 100) {
            return responseInfo.substring(0, 100);
        } else {
            return responseInfo;
        }
    }

    public boolean bindAdView(@NonNull ViewGroup root){
        return bindAdView(root, AdConfigManager.NativeAdViewType.TYPE_NATIVE_AD_DEFAULT);
    }
    
    public boolean bindAdView(@NonNull ViewGroup root, @AdConfigManager.NativeAdViewType int type) {
        root.removeAllViews();
        LoaderCache.getInstance().removeNativeAdLoaderCache(this, scenes.getAdId());
        if (nativeAd != null) {
            NativeAdView adView;
            switch (type) {
                case AdConfigManager.NativeAdViewType.TYPE_NATIVE_AD_BIG:
                    adView = LayoutInflater.from(root.getContext()).inflate(R.layout.native_ad_layout_big, root).findViewById(R.id.native_ad_view_container);
                    break;
                case AdConfigManager.NativeAdViewType.TYPE_NATIVE_AD_SMALL:
                    adView = LayoutInflater.from(root.getContext()).inflate(R.layout.native_ad_layout_small, root).findViewById(R.id.native_ad_view_container);
                    break;
                case AdConfigManager.NativeAdViewType.TYPE_NATIVE_AD_DEFAULT:
                default:
                    adView = LayoutInflater.from(root.getContext()).inflate(R.layout.native_ad_layout, root).findViewById(R.id.native_ad_view_container);
                    break;
            }
            populateNativeAdView(nativeAd, adView, type);
            isShowed = true;
            return true;
        }
        if (nativeCustomFormatAd != null) {
            View adView = LayoutInflater.from(root.getContext()).inflate(R.layout.ad_simple_custom_template, root, false);
            populateSimpleTemplateAdView(nativeCustomFormatAd, adView);
            root.addView(adView);
            isShowed = true;
            return true;
        }
        return false;
    }

    /**
     * Populates a {@link NativeAdView} object with data from a given {@link NativeAd}.
     *
     * @param nativeAd the object containing the ad's assets
     * @param adView   the view to be populated
     */
    private void populateNativeAdView(NativeAd nativeAd, NativeAdView adView, @AdConfigManager.NativeAdViewType int type) {
        ImageView iconView = adView.findViewById(R.id.ad_app_icon);
        TextView headline = adView.findViewById(R.id.ad_headline);
        TextView adBody = adView.findViewById(R.id.ad_body);
        MediaView mediaView = adView.findViewById(R.id.ad_media);
        TextView action = adView.findViewById(R.id.ad_call_to_action);

        // Set the media view.
        adView.setMediaView(mediaView);

        // Set other ad assets.
        adView.setHeadlineView(headline);
        adView.setBodyView(adBody);
        adView.setCallToActionView(action);
        adView.setIconView(iconView);

        // The headline and mediaContent are guaranteed to be in every NativeAd.
        if (headline != null) {
            headline.setText(nativeAd.getHeadline());
        }
        if (mediaView != null) {
            mediaView.setMediaContent(nativeAd.getMediaContent());
        }

        // These assets aren't guaranteed to be in every NativeAd, so it's important to
        // check before trying to display them.
        if (nativeAd.getBody() == null) {
            adBody.setVisibility(View.INVISIBLE);
        } else {
            adBody.setVisibility(View.VISIBLE);
            adBody.setText(nativeAd.getBody());
        }

        if (nativeAd.getCallToAction() == null) {
            action.setVisibility(View.INVISIBLE);
        } else {
            action.setVisibility(View.VISIBLE);
            action.setText(nativeAd.getCallToAction());
        }

        NativeAd.Image icon;
        if ((icon = nativeAd.getIcon()) != null) {
            if (icon.getDrawable() != null) {
                iconView.setVisibility(View.VISIBLE);
                Glide.with(iconView).load(icon.getDrawable()).into(iconView);
            } else if (icon.getUri() != null) {
                iconView.setVisibility(View.VISIBLE);
                Glide.with(iconView).load(icon.getUri()).into(iconView);
            } else {
                iconView.setVisibility(View.GONE);
            }
        } else {
            if (type == AdConfigManager.NativeAdViewType.TYPE_NATIVE_AD_SMALL) {
                iconView.setVisibility(View.VISIBLE);
                iconView.setImageResource(R.drawable.icon_default);
            } else {
                iconView.setVisibility(View.GONE);
            }
        }

        // This method tells the Google Mobile Ads SDK that you have finished populating your
        // native ad view with this native ad.
        adView.setNativeAd(nativeAd);

        // Get the video controller for the ad. One will always be provided, even if the ad doesn't
        // have a video asset.
        VideoController vc = nativeAd.getMediaContent().getVideoController();

        // Updates the UI to say whether or not this ad has a video asset.
        if (vc.hasVideoContent()) {
//            videoStatus.setText(
//                    String.format(
//                            Locale.getDefault(),
//                            "Video status: Ad contains a %.2f:1 video asset.",
//                            nativeAd.getMediaContent().getAspectRatio()));
            // Create a new VideoLifecycleCallbacks object and pass it to the VideoController. The
            // VideoController will call methods on this object when events occur in the video
            // lifecycle.
            vc.setVideoLifecycleCallbacks(new VideoController.VideoLifecycleCallbacks() {
                @Override
                public void onVideoEnd() {
                    // Publishers should allow native ads to complete video playback before
                    // refreshing or replacing them with another ad in the same UI location.
//                    refresh.setEnabled(true);
//                    videoStatus.setText("Video status: Video playback has ended.");
                    super.onVideoEnd();
                }
            });
        } else {
//            videoStatus.setText("Video status: Ad does not contain a video asset.");
//            refresh.setEnabled(true);
        }
    }

    /**
     * Populates a {@link View} object with data from a {@link NativeCustomFormatAd}. This method
     * handles a particular "simple" custom native ad format.
     *
     * @param nativeCustomFormatAd the object containing the ad's assets
     * @param adView               the view to be populated
     */
    private void populateSimpleTemplateAdView(final NativeCustomFormatAd nativeCustomFormatAd, View adView) {
        TextView headline = adView.findViewById(R.id.simplecustom_headline);
        TextView caption = adView.findViewById(R.id.simplecustom_caption);

        headline.setText(nativeCustomFormatAd.getText("Headline"));
        caption.setText(nativeCustomFormatAd.getText("Caption"));

        FrameLayout mediaPlaceholder = adView.findViewById(R.id.simplecustom_media_placeholder);

        // Get the video controller for the ad. One will always be provided, even if the ad doesn't
        // have a video asset.
        VideoController vc = nativeCustomFormatAd.getVideoController();

        // Apps can check the VideoController's hasVideoContent property to determine if the
        // NativeCustomFormatAd has a video asset.
        if (vc.hasVideoContent()) {
            mediaPlaceholder.addView(nativeCustomFormatAd.getVideoMediaView());
//            videoStatus.setText(
//                    String.format(
//                            Locale.getDefault(),
//                            "Video status: Ad contains a %.2f:1 video asset.",
//                            nativeAd.getMediaContent().getAspectRatio()));
            // Create a new VideoLifecycleCallbacks object and pass it to the VideoController. The
            // VideoController will call methods on this object when events occur in the video
            // lifecycle.
            vc.setVideoLifecycleCallbacks(
                    new VideoController.VideoLifecycleCallbacks() {
                        @Override
                        public void onVideoEnd() {
                            // Publishers should allow native ads to complete video playback before
                            // refreshing or replacing them with another ad in the same UI location.
//                            refresh.setEnabled(true);
//                            videoStatus.setText("Video status: Video playback has ended.");
                            super.onVideoEnd();
                        }
                    });
        } else {
            ImageView mainImage = new ImageView(adView.getContext());
            mainImage.setAdjustViewBounds(true);
            mainImage.setImageDrawable(nativeCustomFormatAd.getImage("MainImage").getDrawable());

            mainImage.setOnClickListener(
                    new View.OnClickListener() {
                        @Override
                        public void onClick(View v) {
                            nativeCustomFormatAd.performClick("MainImage");
                        }
                    });
            mediaPlaceholder.addView(mainImage);
//            refresh.setEnabled(true);
//            videoStatus.setText("Video status: Ad does not contain a video asset.");
        }
    }

    @Override
    public void destroy() {
        super.destroy();
        if (isShowed) {
            if (nativeAd != null) {
                nativeAd.destroy();
                nativeAd = null;
            }
        } else {
            if (!isExpired(false)) {
                //退出时如果当前数据没有被使用，重新放回cache中
                LoaderCache.getInstance().putNativeAdLoaderInCache(this, scenes.getAdId());
            }
        }
        if (nativeCustomFormatAd != null) {
            nativeCustomFormatAd.destroy();
            nativeCustomFormatAd = null;
        }
    }

    @Override
    protected String getClassName() {
        return "NativeAdLoader";
    }

}
