package com.transsion.ad.sdk;

import android.content.Context;
import android.os.Handler;
import android.os.Looper;
import android.os.Message;

import androidx.annotation.NonNull;

import com.transsion.ad.AdConfigManager;
import com.transsion.ad.data.Scenes;
import com.transsion.ad.util.AdLog;


public abstract class AbstractAdLoader<T> implements ILoader<T> {
    protected final String TAG = AdLog.COMMON_TAG + getClassName();
    protected Scenes scenes;
    protected boolean isPreload;
    protected boolean isCache;
    protected boolean isForceLoad;
    protected boolean isShowed;
    protected String responseInfo;
    protected T listener;
    protected long time;

    public AbstractAdLoader(Scenes scenes) {
        this.scenes = scenes;
    }

    protected void load(boolean force, boolean preload) {
        AdLog.i(TAG, "start load ad; scenes = " + scenes + "; preload = " + preload + "; force: " + force);
        isPreload = preload;
        isForceLoad = force;
        if (isCacheValid()) {
            isCache = true;
            AdRequestHandler.getInstance().addCacheTask(this);
            return;
        }
        AdRequestHandler.getInstance().addRequestTask(this);
    }

    @Override
    public Context getContext() {
        return AdConfigManager.getInstance().getValidActivity();
    }

    @Override
    public boolean isPreload() {
        return this.isPreload;
    }

    @Override
    public boolean isCache() {
        return this.isCache;
    }

    @Override
    public boolean isForceLoad() {
        return this.isForceLoad;
    }

    @Override
    public void setScenes(Scenes scenes) {
        this.scenes = scenes;
    }

    @Override
    public Scenes getScenes() {
        return scenes;
    }

    @Override
    public String getResponseInfo() {
        return getSubResponseInfo();
    }

    @Override
    public boolean isExpired(boolean show) {
        //Note: While prefetching ads is a great technique, it's important that publishers not keep
        //old ads around too long without displaying them. Any ad objects that have been held for
        //longer than an hour without being displayed should be discarded and replaced with new ads
        //from a new request.
        long duration = System.currentTimeMillis() - time;
        AdLog.i(TAG, "isExpired time:" + duration + "; show:" + show + "; adId:" + scenes.getAdId());
        return time > 0 && duration >= (show ? 55 * 60 * 1000 : 45 * 60 * 1000);
    }

    @NonNull
    @Override
    public String toString() {
        return "{" +
                "preload:" + isPreload +
                "; cache:" + isCache +
                "; force:" + isForceLoad +
                "; ad showed:" + isShowed +
                "; scenes:" + scenes +
                "}";
    }

    @Override
    public void destroy() {
        AdLog.i(TAG, "loader destroyed:" + this);
    }

    protected boolean noCacheHandled() {
        AdLog.i(TAG, "no cache, no handle; preload = " + isCache + "; scenes = " + scenes);
        return false;
    }

    protected boolean isCacheValid() {
        return !isForceLoad && !isExpired(false);
    }

    protected abstract String getClassName();


    protected abstract void cacheLoaded();

    protected abstract void loadRequest();

    private static class AdRequestHandler extends Handler {
        private static final int WHAT_LOAD_REQUEST = 1;
        private static final int WHAT_LOAD_CACHE = 2;
        private AdRequestHandler() {
            super(Looper.getMainLooper());
        }

        private static class AdRequestHolder {
            private static final AdRequestHandler instance = new AdRequestHandler();
        }

        public static AdRequestHandler getInstance() {
            return AdRequestHolder.instance;
        }

        public <T> void addRequestTask(AbstractAdLoader<T> loader) {
            Message msg = obtainMessage();
            msg.what = WHAT_LOAD_REQUEST;
            msg.obj = loader;
            msg.sendToTarget();
        }

        public <T> void addCacheTask(AbstractAdLoader<T> loader){
            Message msg = obtainMessage();
            msg.what = WHAT_LOAD_CACHE;
            msg.obj = loader;
            msg.sendToTarget();
        }

        @Override
        public void handleMessage(@NonNull Message msg) {
            super.handleMessage(msg);
            AbstractAdLoader loader = (AbstractAdLoader) msg.obj;
            switch (msg.what) {
                case WHAT_LOAD_REQUEST:
                    if (loader.noCacheHandled()) {
                        AdLog.i(loader.TAG, "no cache; preload = " + loader.isPreload + "; scenes = " + loader.scenes);
                        return;
                    }
                    loader.isCache = false;
                    loader.loadRequest();
                    AdLog.i(loader.TAG, "end request; preload = " + loader.isPreload + "; scenes = " + loader.getScenes());
                    break;
                case WHAT_LOAD_CACHE:
                    loader.cacheLoaded();
                    AdLog.i(loader.TAG, "end cache; preload = " + loader.isPreload + "; scenes = " + loader.getScenes());
                    break;
            }
        }
    }
}
