package com.transsion.ad.sdk;

import android.os.Bundle;
import android.os.Handler;
import android.os.Looper;
import android.os.Message;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.transsion.ad.AdConfigManager;
import com.transsion.ad.data.Scenes;
import com.transsion.ad.interstitialad.InterstitialAdListener;
import com.transsion.ad.interstitialad.InterstitialAdLoader;
import com.transsion.ad.nativead.NativeAdListener;
import com.transsion.ad.nativead.NativeAdLoader;
import com.transsion.ad.sdk.wrap.AdErrorWrapper;
import com.transsion.ad.sdk.wrap.NativeAdWrapper;
import com.transsion.ad.sdk.wrap.NativeCustomFormatAdWrapper;
import com.transsion.ad.util.AdEvent;
import com.transsion.ad.util.AdLog;

import java.util.Iterator;
import java.util.LinkedList;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

public class DispatcherLoaderTaskHandler extends Handler {
    private static final String TAG = AdLog.COMMON_TAG + "DispatcherLoaderTaskHandler";

    public static final int NATIVE_AD_TASK = 0;
    public static final int INTERSTITIAL_AD_TASK = 1;
    public static final int BACKUP_NATIVE_AD_TASK = 2;
    public static final int BACKUP_INTERSTITIAL_AD_TASK = 3;
    public static final int REMOVE_NATIVE_LISTENER = 4;
    public static final int REMOVE_INTERSTITIAL_LISTENER = 5;

    private final Map<String, LinkedList<NativeAdListener>> mNativeListMap = new ConcurrentHashMap<>();
    private final Map<String, LinkedList<InterstitialAdListener>> mInterstitialListMap = new ConcurrentHashMap<>();
    //关联广告请求Listener和页面Listener
    private final Map<InterstitialAdListener, InterstitialAdListener> mInterstitialAdListenerMap = new ConcurrentHashMap<>();
    private final Map<NativeAdListener, NativeAdListener> mNativeAdListenerMap = new ConcurrentHashMap<>();

    private final Handler mainHandler;

    public DispatcherLoaderTaskHandler(@NonNull Looper looper) {
        super(looper);
        mainHandler = new TaskMainHandler();
    }

    @Override
    public void handleMessage(@NonNull Message msg) {
        int scenesId = msg.arg1;
        boolean force = msg.arg2 == 1;
        switch (msg.what) {
            case NATIVE_AD_TASK:
                loadNativeAd(scenesId, force, (NativeAdListener) msg.obj);
                break;
            case INTERSTITIAL_AD_TASK:
                loadInterstitialAd(scenesId, force, (InterstitialAdListener) msg.obj);
                break;
            case BACKUP_NATIVE_AD_TASK:
                loadBackupNativeAd(force, (NativeAdListener) msg.obj);
                break;
            case BACKUP_INTERSTITIAL_AD_TASK:
                loadBackupInterstitialAd(force, (InterstitialAdListener) msg.obj);
                break;
            case REMOVE_NATIVE_LISTENER:
                if (msg.obj == null) {
                    break;
                }

                NativeAdListener nativeAdListener = (NativeAdListener) msg.obj;
                for (Iterator<Map.Entry<NativeAdListener, NativeAdListener>> it = mNativeAdListenerMap.entrySet().iterator(); it.hasNext(); ) {
                    Map.Entry<NativeAdListener, NativeAdListener> item = it.next();
                    if (nativeAdListener.equals(item.getValue())) {
                        it.remove();
                    }
                }
                break;
            case REMOVE_INTERSTITIAL_LISTENER:
                if (msg.obj == null) {
                    break;
                }

                InterstitialAdListener interstitialAdListener = (InterstitialAdListener) msg.obj;
                for (Iterator<Map.Entry<InterstitialAdListener, InterstitialAdListener>> it = mInterstitialAdListenerMap.entrySet().iterator(); it.hasNext(); ) {
                    Map.Entry<InterstitialAdListener, InterstitialAdListener> item = it.next();
                    if (interstitialAdListener.equals(item.getValue())) {
                        it.remove();
                    }
                }
                break;
        }
    }

    private void sendNativeCacheInvalidListener(@NonNull NativeAdListener listener) {
        Message msg = mainHandler.obtainMessage();
        msg.what = TaskMainHandler.NATIVE_CACHE_INVALID_TASK;
        msg.obj = listener;
        msg.sendToTarget();
    }

    private void sendInterstitialCacheInvalidListener(@NonNull InterstitialAdListener listener) {
        Message msg = mainHandler.obtainMessage();
        msg.what = TaskMainHandler.INTERSTITIAL_CACHE_INVALID_TASK;
        msg.obj = listener;
        msg.sendToTarget();
    }

    private void loadBackupNativeAd(boolean force, NativeAdListener listener) {
        Scenes scenes = AdConfigManager.getInstance().getBackupScenes(AdConfigManager.AD_TYPE_NATIVE);
        if (listener != null) {
            loadNativeAd(scenes, listener);
        } else {
            preloadBackupAd(AdConfigManager.AD_TYPE_NATIVE, force);
        }
    }

    private void loadNativeAd(int scenesId, boolean force, NativeAdListener listener) {
        Scenes scenes = AdConfigManager.getInstance().getScenesById(scenesId, AdConfigManager.AD_TYPE_NATIVE);
        if (scenes.getScenesId() < 0) {
            if (scenes.getScenesId() == -1) {
                AdLog.e(TAG, "native ad id no cache" + scenesId);
            } else {
                AdLog.e(TAG, "native ad id and scenesId do not match:" + scenesId);
                adMatchEvent(scenesId, AdConfigManager.AD_TYPE_NATIVE);
            }
            if (listener == null) {
                return;
            }
            sendNativeCacheInvalidListener(listener);
            return;
        }
        if (listener != null) {
            loadNativeAd(scenes, listener);
        } else {
            preloadNativeAd(scenes, force);
        }
    }

    private void loadNativeAd(Scenes scenes, NativeAdListener listener) {
        adRequestEvent(scenes.getAdId(), AdConfigManager.AD_TYPE_NATIVE, "load");
        addNativeAdListenerObserver(scenes.getAdId(), listener);
        NativeAdLoader loader = LoaderCache.getInstance().getNativeAdLoaderFromCache(scenes.getAdId(), false);
        if (loader != null) {
            loader.setScenes(scenes);
        } else {
            loader = new NativeAdLoader(scenes);
        }
        loader.loadAd(new NativeRequestListener(this, scenes.getAdId()));
    }

    private void preloadNativeAd(@NonNull Scenes scenes, boolean force) {
        adRequestEvent(scenes.getAdId(), AdConfigManager.AD_TYPE_NATIVE, "preload");
        NativeAdLoader loader = force ? null : LoaderCache.getInstance().getNativeAdLoaderFromCache(scenes.getAdId(), true);
        if (loader != null) {
            loader.setScenes(scenes);
        } else {
            loader = new NativeAdLoader(scenes);
        }
        loader.preloadAd(force, new NativeRequestListener(this, scenes.getAdId()));
    }

    private void loadBackupInterstitialAd(boolean force, InterstitialAdListener listener) {
        Scenes scenes = AdConfigManager.getInstance().getBackupScenes(AdConfigManager.AD_TYPE_INTERSTITIAL);
        if (listener != null) {
            loadInterstitialAd(scenes, listener);
        } else {
            preloadBackupAd(AdConfigManager.AD_TYPE_INTERSTITIAL, force);
        }
    }

    private void loadInterstitialAd(int scenesId, boolean force, InterstitialAdListener listener) {
        Scenes scenes = AdConfigManager.getInstance().getScenesById(scenesId, AdConfigManager.AD_TYPE_INTERSTITIAL);
        if (scenes.getScenesId() < 0) {
            if (scenes.getScenesId() == -1) {
                AdLog.e(TAG, "interstitial ad id no cache" + scenesId);
            } else {
                AdLog.e(TAG, "interstitial ad id and scenesId do not match:" + scenesId);
                adMatchEvent(scenesId, AdConfigManager.AD_TYPE_INTERSTITIAL);
            }
            if (listener == null) {
                return;
            }
            sendInterstitialCacheInvalidListener(listener);
            return;
        }
        if (listener != null) {
            loadInterstitialAd(scenes, listener);
        } else {
            preloadInterstitialAd(scenes, force);
        }
    }

    private void loadInterstitialAd(Scenes scenes, InterstitialAdListener listener) {
        adRequestEvent(scenes.getAdId(), AdConfigManager.AD_TYPE_INTERSTITIAL, "load");
        addInterstitialAdListenerObserver(scenes.getAdId(), listener);
        InterstitialAdLoader loader = LoaderCache.getInstance().getInterstitialAdLoaderFromCache(scenes.getAdId(), false);
        if (loader != null) {
            loader.setScenes(scenes);
        } else {
            loader = new InterstitialAdLoader(scenes);
        }
        loader.loadAd(new InterstitialRequestListener(this, scenes.getAdId()));
    }

    private void preloadInterstitialAd(Scenes scenes, boolean force) {
        adRequestEvent(scenes.getAdId(), AdConfigManager.AD_TYPE_INTERSTITIAL, "preload");
        InterstitialAdLoader loader = force ? null : LoaderCache.getInstance().getInterstitialAdLoaderFromCache(scenes.getAdId(), true);
        if (loader != null) {
            loader.setScenes(scenes);
        } else {
            loader = new InterstitialAdLoader(scenes);
        }
        loader.preloadAd(force, new InterstitialRequestListener(this, scenes.getAdId()));
    }

    private void preloadBackupAd(int adType, boolean force) {
        Scenes scenes = AdConfigManager.getInstance().getBackupScenes(adType);
        switch (adType) {
            case AdConfigManager.AD_TYPE_INTERSTITIAL:
                preloadInterstitialAd(scenes, force);
                break;
            case AdConfigManager.AD_TYPE_NATIVE:
                preloadNativeAd(scenes, force);
                break;
        }
    }

    private void addNativeAdListenerObserver(String adId, NativeAdListener listener) {
        LinkedList<NativeAdListener> listenerList = mNativeListMap.get(adId);
        if (listenerList == null) {
            listenerList = new LinkedList<>();
            mNativeListMap.put(adId, listenerList);
        }
        if (!listenerList.contains(listener)) {
            listenerList.add(listener);
        }
    }

    private void addInterstitialAdListenerObserver(String adId, InterstitialAdListener listener) {
        LinkedList<InterstitialAdListener> listenerList = mInterstitialListMap.get(adId);
        if (listenerList == null) {
            listenerList = new LinkedList<>();
            mInterstitialListMap.put(adId, listenerList);
        }
        if (!listenerList.contains(listener)) {
            listenerList.add(listener);
        }
    }

    private void adRequestEvent(String adId, int adType, String requestType) {
        Bundle bundle = new Bundle();
        bundle.putString(AdEvent.AD_ID, adId);
        bundle.putInt(AdEvent.AD_TYPE, adType);
        bundle.putString(AdEvent.TYPE, requestType);
        AdEvent.Event event = new AdEvent.Event();
        event.name = AdEvent.AD_REQUEST;
        event.bundle = bundle;
        AdEvent.getInstance().addEventInfo(event);
    }

    private void adMatchEvent(int scenesId, int adType) {
        Bundle bundle = new Bundle();
        bundle.putInt(AdEvent.AD_TYPE, adType);
        bundle.putString(AdEvent.AD_ID, String.valueOf(scenesId));
        AdEvent.Event event = new AdEvent.Event();
        event.bundle = bundle;
        event.name = AdEvent.AD_MATCH;
        AdEvent.getInstance().addEventInfo(event);
    }

    private static class TaskMainHandler extends Handler {
        private static final int NATIVE_CACHE_INVALID_TASK = 0;
        private static final int INTERSTITIAL_CACHE_INVALID_TASK = 1;

        public TaskMainHandler() {
            super(Looper.getMainLooper());
        }

        @Override
        public void handleMessage(@NonNull Message msg) {
            switch (msg.what) {
                case NATIVE_CACHE_INVALID_TASK:
                    ((NativeAdListener) msg.obj).onCacheInvalid(null);
                    break;
                case INTERSTITIAL_CACHE_INVALID_TASK:
                    ((InterstitialAdListener) msg.obj).onCacheInvalid(null);
                    break;
            }
        }
    }

    public static class InterstitialRequestListener extends InterstitialAdListener {
        private final DispatcherLoaderTaskHandler loaderTaskHandler;
        private final String adId;

        public InterstitialRequestListener(@NonNull DispatcherLoaderTaskHandler loaderTaskHandler, @NonNull String adId) {
            this.loaderTaskHandler = loaderTaskHandler;
            this.adId = adId;
        }

        @Override
        public void onAdFailedToShowFullScreenContent(@NonNull InterstitialAdLoader loader, @NonNull AdErrorWrapper adError) {
            AdLog.e(TAG, "onAdFailedToShowFullScreenContent loader = " + loader + " adError = " + adError);
            InterstitialAdListener listener = getAdListener(this);
            if (listener != null) {
                listener.onAdFailedToShowFullScreenContent(loader, adError);
            }
        }

        @Override
        public void onAdShowedFullScreenContent(@NonNull InterstitialAdLoader loader) {
            AdLog.i(TAG, "onAdShowedFullScreenContent loader = " + loader);

            InterstitialAdListener listener = getAdListener(this);
            if (listener != null) {
                listener.onAdShowedFullScreenContent(loader);
            }
        }

        @Override
        public void onLoaded(@NonNull InterstitialAdLoader loader) {
            AdLog.i(TAG, "onLoad info loader = " + loader);
            Bundle bundle = new Bundle();
            bundle.putString(AdEvent.AD_ID, loader.getScenes().getAdId());
            bundle.putString(AdEvent.TYPE, loader.isPreload ? "preload" : "load");
            bundle.putString(AdEvent.RESULT, "true");
            bundle.putString(AdEvent.REASON, "-1;success");
            bundle.putString(AdEvent.SOURCE, loader.isCache() ? "cache" : "realtime");
            bundle.putString(AdEvent.AD_SOURCE, loader.getSubResponseInfo());
            bundle.putInt(AdEvent.AD_TYPE, AdConfigManager.AD_TYPE_INTERSTITIAL);
            AdEvent.Event event = new AdEvent.Event();
            event.name = AdEvent.AD_REQUEST_RESULT;
            event.bundle = bundle;
            AdEvent.getInstance().addEventInfo(event);

            InterstitialAdListener listener = getAdListener(this);
            if (listener != null) {
                listener.onLoaded(loader);
            }
        }

        @Override
        public void onError(@NonNull InterstitialAdLoader loader, @NonNull AdErrorWrapper loadAdError) {
            AdLog.e(TAG, "onError" + "loader = " + loader + " errorCode = " + loadAdError);
            Bundle bundle = new Bundle();
            bundle.putString(AdEvent.AD_ID, loader.getScenes().getAdId());
            bundle.putString(AdEvent.TYPE, loader.isPreload() ? "preload" : "load");
            bundle.putString(AdEvent.RESULT, "false");
            bundle.putString(AdEvent.REASON, loadAdError.getSubMessage());
            bundle.putString(AdEvent.SOURCE, loader.isCache() ? "cache" : "realtime");
            bundle.putString(AdEvent.AD_SOURCE, loader.getSubResponseInfo());
            bundle.putInt(AdEvent.AD_TYPE, AdConfigManager.AD_TYPE_INTERSTITIAL);
            AdEvent.Event event = new AdEvent.Event();
            event.name = AdEvent.AD_REQUEST_RESULT;
            event.bundle = bundle;
            AdEvent.getInstance().addEventInfo(event);

            InterstitialAdListener listener = getAdListener(this);
            if (listener != null) {
                listener.onError(loader, loadAdError);
            }
        }

        @Override
        public boolean onCacheInvalid(@Nullable InterstitialAdLoader loader) {
            AdLog.i(TAG, "onCacheInvalid loader = " + loader);
            InterstitialAdListener listener = getAdListener(this);
            if (listener != null) {
                return listener.onCacheInvalid(loader);
            }
            return false;
        }

        @Override
        public void onClicked(@NonNull InterstitialAdLoader loader) {
            AdLog.i(TAG, "onClicked" + "loader = " + loader);
            Bundle bundle = new Bundle();
            bundle.putString(AdEvent.AD_ID, loader.getScenes().getAdId());
            bundle.putString(AdEvent.AD_SOURCE, loader.getSubResponseInfo());
            bundle.putInt(AdEvent.AD_TYPE, AdConfigManager.AD_TYPE_INTERSTITIAL);
            AdEvent.Event event = new AdEvent.Event();
            event.name = AdEvent.AD_CLICK;
            event.bundle = bundle;
            AdEvent.getInstance().addEventInfo(event);
            AdEvent.getInstance().addAdjustEvent("em3g3z");

            InterstitialAdListener listener = getAdListener(this);
            if (listener != null) {
                listener.onClicked(loader);
            }
        }

        @Override
        public void onClosed(@NonNull InterstitialAdLoader loader) {
            AdLog.i(TAG, "onClosed loader = " + loader);
            Bundle bundle = new Bundle();
            bundle.putString(AdEvent.AD_ID, loader.getScenes().getAdId());
            bundle.putString(AdEvent.AD_SOURCE, loader.getSubResponseInfo());
            bundle.putInt(AdEvent.AD_TYPE, AdConfigManager.AD_TYPE_INTERSTITIAL);
            AdEvent.Event event = new AdEvent.Event();
            event.name = AdEvent.AD_CLOSE;
            event.bundle = bundle;
            AdEvent.getInstance().addEventInfo(event);

            InterstitialAdListener listener = getAdListener(this);
            if (listener != null) {
                listener.onClosed(loader);
            }
        }

        @Override
        public void onOpened(@NonNull InterstitialAdLoader loader) {
            AdLog.i(TAG, "onOpened loader = " + loader);

            InterstitialAdListener listener = getAdListener(this);
            if (listener != null) {
                listener.onOpened(loader);
            }
        }

        @Override
        public void onShow(@NonNull InterstitialAdLoader loader) {
            AdLog.i(TAG, "onShow loader = " + loader);
            AdEvent.getInstance().addAdjustEvent("6sbhqa");
            InterstitialAdListener listener = getAdListener(this);
            if (listener != null) {
                listener.onShow(loader);
            }
        }

        @Override
        protected String getClassName() {
            return AdLog.COMMON_TAG + "InterstitialRequestListener";
        }

        private InterstitialAdListener getAdListener(InterstitialAdListener innerListener) {
            InterstitialAdListener outListener = loaderTaskHandler.mInterstitialAdListenerMap.get(innerListener);
            if (outListener != null) {
                return outListener;
            }
            LinkedList<InterstitialAdListener> listenerLinkedList = loaderTaskHandler.mInterstitialListMap.get(adId);
            outListener = listenerLinkedList != null ? listenerLinkedList.pollLast() : null;
            if (outListener != null) {
                loaderTaskHandler.mInterstitialAdListenerMap.put(innerListener, outListener);
            }
            return outListener;
        }
    }

    /**
     * 用于接口返回数据统计等处理，不受页面逻辑影响
     */
    public static class NativeRequestListener extends NativeAdListener {
        private final DispatcherLoaderTaskHandler loaderTaskHandler;
        private final String adId;

        public NativeRequestListener(@NonNull DispatcherLoaderTaskHandler loaderTaskHandler, @NonNull String adId) {
            this.loaderTaskHandler = loaderTaskHandler;
            this.adId = adId;
        }

        @Override
        public void onLoaded(@NonNull NativeAdLoader loader) {
            AdLog.i(TAG, "onLoad info loader = " + loader);
            Bundle bundle = new Bundle();
            bundle.putString(AdEvent.AD_ID, loader.getScenes().getAdId());
            bundle.putString(AdEvent.TYPE, loader.isPreload() ? "preload" : "load");
            bundle.putString(AdEvent.RESULT, "true");
            bundle.putString(AdEvent.REASON, "-1;success");
            bundle.putString(AdEvent.SOURCE, loader.isCache() ? "cache" : "realtime");
            bundle.putString(AdEvent.AD_SOURCE, loader.getSubResponseInfo());
            bundle.putInt(AdEvent.AD_TYPE, AdConfigManager.AD_TYPE_NATIVE);
            AdEvent.Event event = new AdEvent.Event();
            event.name = AdEvent.AD_REQUEST_RESULT;
            event.bundle = bundle;
            AdEvent.getInstance().addEventInfo(event);

            NativeAdListener listener = getAdListener(this);
            if (listener != null) {
                listener.onLoaded(loader);
            }
        }

        @Override
        public void onNativeAdLoaded(@NonNull NativeAdLoader loader, @NonNull NativeAdWrapper nativeAd) {
            AdLog.i(TAG, "onNativeAdLoaded loader = " + loader);
            NativeAdListener listener = getAdListener(this);
            if (listener != null) {
                listener.onNativeAdLoaded(loader, nativeAd);
            }
        }

        @Override
        public void onCustomFormatAdLoaded(@NonNull NativeAdLoader loader, @NonNull NativeCustomFormatAdWrapper nativeCustomFormatAd) {
            AdLog.i(TAG, "onCustomFormatAdLoaded loader = " + loader + "  customId = " + nativeCustomFormatAd.getCustomFormatId());
            NativeAdListener listener = getAdListener(this);
            if (listener != null) {
                listener.onCustomFormatAdLoaded(loader, nativeCustomFormatAd);
            }
        }

        @Override
        public void onCustomClick(@NonNull NativeAdLoader loader, @NonNull NativeCustomFormatAdWrapper nativeCustomFormatAd, @NonNull String assetName) {
            AdLog.i(TAG, "onCustomClick loader = " + loader + "  customId = " + nativeCustomFormatAd.getCustomFormatId() + " assetName = " + assetName);
            NativeAdListener listener = getAdListener(this);
            if (listener != null) {
                listener.onCustomClick(loader, nativeCustomFormatAd, assetName);
            }
        }


        @Override
        public void onError(@NonNull NativeAdLoader loader, @NonNull AdErrorWrapper loadAdError) {
            AdLog.e(TAG, "onError" + "loader = " + loader + " errorCode = " + loadAdError);
            Bundle bundle = new Bundle();
            bundle.putString(AdEvent.AD_ID, loader.getScenes().getAdId());
            bundle.putString(AdEvent.TYPE, loader.isPreload() ? "preload" : "load");
            bundle.putString(AdEvent.RESULT, "false");
            bundle.putString(AdEvent.REASON, loadAdError.getSubMessage());
            bundle.putString(AdEvent.SOURCE, loader.isCache() ? "cache" : "realtime");
            bundle.putString(AdEvent.AD_SOURCE, loader.getSubResponseInfo());
            bundle.putInt(AdEvent.AD_TYPE, AdConfigManager.AD_TYPE_NATIVE);
            AdEvent.Event event = new AdEvent.Event();
            event.name = AdEvent.AD_REQUEST_RESULT;
            event.bundle = bundle;
            AdEvent.getInstance().addEventInfo(event);

            NativeAdListener listener = getAdListener(this);
            if (listener != null) {
                listener.onError(loader, loadAdError);
            }
        }

        @Override
        public boolean onCacheInvalid(@Nullable NativeAdLoader loader) {
            AdLog.i(TAG, "onCacheInvalid loader = " + loader);
            NativeAdListener listener = getAdListener(this);
            if (listener != null) {
                return listener.onCacheInvalid(loader);
            }
            return false;
        }

        @Override
        public void onClicked(@NonNull NativeAdLoader loader) {
            AdLog.i(TAG, "onClicked info loader = " + loader);
            Bundle bundle = new Bundle();
            bundle.putString(AdEvent.AD_ID, loader.getScenes().getAdId());
            bundle.putString(AdEvent.AD_SOURCE, loader.getSubResponseInfo());
            bundle.putInt(AdEvent.AD_TYPE, AdConfigManager.AD_TYPE_NATIVE);
            AdEvent.Event event = new AdEvent.Event();
            event.name = AdEvent.AD_CLICK;
            event.bundle = bundle;
            AdEvent.getInstance().addEventInfo(event);
            AdEvent.getInstance().addAdjustEvent("em3g3z");

            NativeAdListener listener = getAdListener(this);
            if (listener != null) {
                listener.onClicked(loader);
            }
        }

        @Override
        public void onClosed(@NonNull NativeAdLoader loader) {
            AdLog.i(TAG, "onClosed info loader = " + loader);
            Bundle bundle = new Bundle();
            bundle.putString(AdEvent.AD_ID, loader.getScenes().getAdId());
            bundle.putString(AdEvent.AD_SOURCE, loader.getSubResponseInfo());
            bundle.putInt(AdEvent.AD_TYPE, AdConfigManager.AD_TYPE_NATIVE);
            AdEvent.Event event = new AdEvent.Event();
            event.name = AdEvent.AD_CLOSE;
            event.bundle = bundle;
            AdEvent.getInstance().addEventInfo(event);

            NativeAdListener listener = getAdListener(this);
            if (listener != null) {
                listener.onClosed(loader);
            }
        }

        @Override
        public void onOpened(@NonNull NativeAdLoader loader) {
            AdLog.i(TAG, "onOpened info loader = " + loader);

            NativeAdListener listener = getAdListener(this);
            if (listener != null) {
                listener.onOpened(loader);
            }
        }

        @Override
        public void onShow(@NonNull NativeAdLoader loader) {
            AdLog.i(TAG, "onShow info loader = " + loader);
            AdEvent.getInstance().addAdjustEvent("6sbhqa");
            NativeAdListener listener = getAdListener(this);
            if (listener != null) {
                listener.onShow(loader);
            }
        }

        @Override
        protected String getClassName() {
            return AdLog.COMMON_TAG + "NativeRequestListener";
        }

        private NativeAdListener getAdListener(NativeAdListener innerListener) {
            NativeAdListener outListener = loaderTaskHandler.mNativeAdListenerMap.get(innerListener);
            if (outListener != null) {
                return outListener;
            }
            LinkedList<NativeAdListener> listenerLinkedList = loaderTaskHandler.mNativeListMap.get(adId);
            outListener = listenerLinkedList != null ? listenerLinkedList.pollLast() : null;
            if (outListener != null) {
                loaderTaskHandler.mNativeAdListenerMap.put(innerListener, outListener);
            }
            return outListener;
        }
    }
}
