package com.transsion.ad.sdk;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.transsion.ad.sdk.wrap.AdErrorWrapper;

public interface IAdListener<T> {
    /**
     * 广告加载完成后，系统会执行 onLoaded() 方法.该接口会回调多次，比如同一个广告id同时发起多个请求，在当前页面请求结
     * 果返回后会通知到业务层。多次返回结果业务可保存后用于上个展示失败时进行后续补充展示，已提高展示机会。退出页面时调用
     * destroy销毁，没有展示的loader则会保存用于下次展示。
     *
     * @param loader 当前广告加载loader对象
     */
    void onLoaded(@NonNull T loader);

    /**
     * 没有缓存回调。
     *
     * @param loader 当前广告加载loader对象，该接口场景该对象有可能为空。正式场景未匹配到，当前load为空，业务层决定是否请求备选广告
     *               备选广告在接口未下发完成前，可以在应用初始化时调用AdConfigManager.setDefaultScenes设置备选广告id
     * @return 业务层是否已经处理，返回true表示业务层已处理，无需请求广告，反之继续请求广告
     */
    boolean onCacheInvalid(@Nullable T loader);

    /**
     * 广告数据请求失败时加载该方法
     *
     * @param loadAdError LoadAdError 类型的错误参数描述了发生的错误。如需了解详情，
     *                    请参阅“https://developers.google.com/admob/android/ad-load-errors?hl=zh-cn”文档。
     */
    void onError(@NonNull T loader, @NonNull AdErrorWrapper loadAdError);

    /**
     * 记录了广告获得的点击后，系统会调用 onClicked() 方法。
     *
     * @param loader 当前广告加载loader对象
     */
    void onClicked(@NonNull T loader);

    /**
     * 用户在查看广告的目标网址后返回应用时，调用该方法。应用可以使用此方法恢复暂停的活动，或执行任何其他必要的操作，
     * 以做好互动准备。
     *
     * @param loader 当前广告加载loader对象
     */
    void onClosed(@NonNull T loader);

    /**
     * 广告打开覆盖屏幕的叠加层时，系统会调用 onOpened() 方法。
     */
    void onOpened(@NonNull T loader);

    /**
     * 记录了广告获得的展示后，系统会调用 onShow() 方法。
     */
    void onShow(@NonNull T loader);
}
