package com.transsion.ad.sdk;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.transsion.ad.interstitialad.InterstitialAdLoader;
import com.transsion.ad.nativead.NativeAdLoader;
import com.transsion.ad.util.LimitQueue;

import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

public class LoaderCache {
    private final Map<String, LimitQueue<NativeAdLoader>> mNativeAdLoaderCache = new ConcurrentHashMap<>();
    private final Map<String, LimitQueue<InterstitialAdLoader>> mInterstitialAdLoaderCache = new ConcurrentHashMap<>();

    private LoaderCache() {
    }

    private static final class LoaderCacheHolder {
        private static final LoaderCache sInstance = new LoaderCache();
    }

    public static LoaderCache getInstance() {
        return LoaderCacheHolder.sInstance;
    }

    @Nullable
    public NativeAdLoader getNativeAdLoaderFromCache(String adId, boolean preload) {
        LimitQueue<NativeAdLoader> loaderLimitQueue = mNativeAdLoaderCache.get(adId);
        if (loaderLimitQueue == null) {
            return null;
        }
        NativeAdLoader loader = preload ? loaderLimitQueue.peek() : loaderLimitQueue.poll();
        if (loader == null) {
            return null;
        }
        if (loader.isExpired(false)) {
            if (preload) {
                loaderLimitQueue.remove(loader);
            }
            loader = getNativeAdLoaderFromCache(adId, preload);
        }
        return loader;
    }

    public void putNativeAdLoaderInCache(@NonNull NativeAdLoader loader, @NonNull String adId) {
        LimitQueue<NativeAdLoader> loaderLimitQueue = mNativeAdLoaderCache.get(adId);
        if (loaderLimitQueue == null) {
            loaderLimitQueue = new LimitQueue<>(3);
            mNativeAdLoaderCache.put(adId, loaderLimitQueue);
        }
        if (loaderLimitQueue.contains(loader)) {
            return;
        }
        loaderLimitQueue.offer(loader);
    }

    public void removeNativeAdLoaderCache(@NonNull NativeAdLoader loader, @NonNull String adId) {
        LimitQueue<NativeAdLoader> loaderLimitQueue = mNativeAdLoaderCache.get(adId);
        if (loaderLimitQueue == null) {
            return;
        }
        loaderLimitQueue.remove(loader);
    }

    @Nullable
    public InterstitialAdLoader getInterstitialAdLoaderFromCache(String adId, boolean preload) {
        LimitQueue<InterstitialAdLoader> loaderLimitQueue = mInterstitialAdLoaderCache.get(adId);
        if (loaderLimitQueue == null) {
            return null;
        }
        InterstitialAdLoader loader = preload ? loaderLimitQueue.peek() : loaderLimitQueue.poll();
        if (loader == null) {
            return null;
        }
        if (loader.isExpired(false)) {
            if (preload) {
                loaderLimitQueue.remove(loader);
            }
            loader = getInterstitialAdLoaderFromCache(adId, preload);
        }
        return loader;
    }

    public void putInterstitialAdLoaderInCache(@NonNull InterstitialAdLoader loader, @NonNull String adId) {
        LimitQueue<InterstitialAdLoader> loaderLimitQueue = mInterstitialAdLoaderCache.get(adId);
        if (loaderLimitQueue == null) {
            loaderLimitQueue = new LimitQueue<>(3);
            mInterstitialAdLoaderCache.put(adId, loaderLimitQueue);
        }
        if (loaderLimitQueue.contains(loader)) {
            return;
        }
        loaderLimitQueue.offer(loader);
    }

    public void removeInterstitialAdLoaderCache(@NonNull InterstitialAdLoader loader, @NonNull String adId) {
        LimitQueue<InterstitialAdLoader> loaderLimitQueue = mInterstitialAdLoaderCache.get(adId);
        if (loaderLimitQueue == null) {
            return;
        }
        loaderLimitQueue.remove(loader);
    }
}
