package com.transsion.ad.bidding.base

import android.app.Activity
import com.hisavana.common.bean.AdditionalInfo
import com.hisavana.common.bean.TAdErrorCode
import com.hisavana.common.bean.TAdNativeInfo
import com.hisavana.common.interfacz.TAdditionalListener
import com.hisavana.mediation.ad.TBaseAd
import com.transsion.ad.AdLogger
import com.transsion.ad.bidding.BiddingTAdditionalListener
import com.transsion.ad.hi.HiSavanaAdManager
import com.transsion.ad.monopoly.model.MbAdSource
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import java.util.concurrent.atomic.AtomicBoolean

/**
 * @author shmizhangxinbing
 * @date : 2025/6/3 10:48
 * @description:
 */
abstract class AbsBiddingInterceptHiSavanaAdManager : TAdditionalListener() {

    private var mListener: BiddingTAdditionalListener? = null // 外部接收广告状态回调
    private var mPlacementId: String? = null // 程序化 场景ID
    private var mCtxMap: Map<String, Any>? = null
    private var isShowing: AtomicBoolean = AtomicBoolean(false) // 插屏广告是否正在展示
    private var additionalInfo: AdditionalInfo? = null // 下一次加载之前都可以使用

    protected fun getClassTag(): String = javaClass.simpleName


    // =============================================================================================


    abstract fun getAdType(): Int

    abstract fun getAdInstance(): TBaseAd<*>?

    abstract fun show(topActivity: Activity, ctxMap: Map<String, Any>?)

    abstract fun reset()


    // ====================================== 广告回调 ===============================================


    override fun onLoadSuccess(p0: AdditionalInfo) {
        super.onLoadSuccess(p0)
        additionalInfo = p0
        mListener?.onLoadSuccess(p0)
    }

    override fun onLoadFailure(p0: TAdErrorCode?, p1: AdditionalInfo) {
        super.onLoadFailure(p0, p1)
        isShowing.set(false)
        AdLogger.logSdkInterceptE("${getClassTag()} --> onLoadFailure() --> placementId = ${getPlacementId()} --> p0 = $p0")
    }

    override fun onClick(p0: TAdNativeInfo?, p1: AdditionalInfo) {
        super.onClick(p0, p1)
        mListener?.onClick(p0, p1)
    }

    override fun onShow(p0: TAdNativeInfo?, p1: AdditionalInfo) {
        super.onShow(p0, p1)
        isShowing.set(true)
        mListener?.onShow(p0, p1)
    }

    override fun onShowError(p0: TAdErrorCode?, p1: AdditionalInfo) {
        super.onShowError(p0, p1)
        isShowing.set(false)
        mListener?.onShowError(p0, p1)
    }

    override fun onClosed(p0: Int) {
        super.onClosed(p0)
        isShowing.set(false)
        mListener?.onClosed(p0)
        mListener = null

        //getAdInstance()?.destroy()
        reset()

        // 同步加载一个
        loadAd()
    }

    override fun onRewarded() {
        super.onRewarded()
        mListener?.onRewarded()
    }


    // =============================================================================================


    /**
     * 将回调设置进来
     *
     * 展示的时候设置
     */
    fun setListener(listener: BiddingTAdditionalListener?): AbsBiddingInterceptHiSavanaAdManager {
        mListener = listener
        return this
    }

    fun setPlacementId(placementId: String?) {
        mPlacementId = placementId
    }

    fun getPlacementId(): String? = mPlacementId

    fun setCtxMap(ctxMap: Map<String, Any>?) {
        mCtxMap = ctxMap
    }

    /**
     * 资源回收 -- 因为是全局的所以只需要断开引用就可以了
     */
    fun destroy(listener: BiddingTAdditionalListener?) {
        if (mListener == null) {
            return
        }
        if (mListener == listener) {
            setListener(null)
            AdLogger.logSdkIntercept("${getClassTag()} --> 移除监听")
        }
    }


    // ==================================== 广告加载展示 =============================================


    /**
     * 同步获取广告
     */
    fun getAdditionalInfo(): AdditionalInfo? {
        val hasAd = getAdInstance()?.hasAd() == true
        if (hasAd.not()) {
            loadAd()
            return null
        }
        return additionalInfo
    }

    /**
     * 预加载广告
     */
    fun preLoadAd() {
        // 如果没有初始化，那就不进行任何操作
        if (HiSavanaAdManager.isInitialized().not()) {
            return
        }

        // 预加载广告
        if (getAdInstance()?.hasAd() == false) {
            AdLogger.logSdkIntercept("${getClassTag()} --> preLoadAd() --> hiId = ${getPlacementId()}")
            getAdInstance()?.preload()
        }
    }

    /**
     * 加载广告
     */
    fun loadAd() {
        // 如果没有初始化，那就不进行任何操作
        if (HiSavanaAdManager.isInitialized().not()) {
            mListener?.onLoadFailure(
                TAdErrorCode(MbAdSource.MB_AD_SOURCE_HISAVANA, "广告SDK没有初始化"),
                AdditionalInfo()
            )
            return
        }

        // 如果不存在广告，那就重新加载，SDK内部处理了重复调用
        if (isShowing.get()) {
            mListener?.onLoadFailure(
                TAdErrorCode(MbAdSource.MB_AD_SOURCE_HISAVANA, "当前正在展示Hi广告"),
                AdditionalInfo()
            )
            return
        }

        if (getAdInstance() == null) {
            AdLogger.logSdkInterceptE("${getClassTag()} --> loadAd() --> getAdInstance() == null")
            return
        }

        if (getAdInstance()?.hasAd() == false) {
            additionalInfo = null // 重置数据
            getAdInstance()?.loadAd()
        } else {
            // 已经存在广告了直接返回
            additionalInfo?.let {
                mListener?.onLoadSuccess(it)
            }
        }
    }

    /**
     * 展示广告
     */
    fun showAd(activity: Activity?, sceneId: String) {
        if (activity == null) {
            mListener?.onShowError(
                TAdErrorCode(MbAdSource.MB_AD_SOURCE_HISAVANA, "activity is null"), AdditionalInfo()
            )
            return
        }

        if (isShowing.get()) {
            mListener?.onShowError(
                TAdErrorCode(
                    MbAdSource.MB_AD_SOURCE_HISAVANA, "当前正在展示Hi广告"
                ), AdditionalInfo()
            )
            return
        }

        if (getAdInstance()?.hasAd() == true) {
            show(activity, mCtxMap)
        } else {
            mListener?.onShowError(
                TAdErrorCode(MbAdSource.MB_AD_SOURCE_HISAVANA, "当前还没有广告"), AdditionalInfo()
            )
            // 如果直接展示失败了之后就不在继续回调后续的事件了
            //destroy(null)
            setListener(null)
            // 同步加载一个
            loadAd()
        }
    }

    fun enterScene(sceneId: String) {
        CoroutineScope(Dispatchers.Main).launch {
            getAdInstance()?.enterScene(sceneId, 1)
        }
    }
}