package com.transsion.ad.bidding.nativead

import android.os.Handler
import android.os.Looper
import androidx.recyclerview.widget.GridLayoutManager
import androidx.recyclerview.widget.LinearLayoutManager
import androidx.recyclerview.widget.RecyclerView
import com.hisavana.common.bean.TAdErrorCode
import com.transsion.ad.AdLogger
import com.transsion.ad.bidding.BiddingTAdditionalListener
import com.transsion.ad.bidding.base.BiddingIntermediateMaterialBean
import com.transsion.ad.scene.SceneCommonConfig
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext
import java.lang.ref.WeakReference

/**
 * @author shmizhangxinbing
 * @date : 2025/6/5 16:29
 * @description: 列表展示Native广告
 */
class BiddingListManager {

    private var mCtxMap: MutableMap<String, Any> = mutableMapOf() // 上下文数据容器
    private var mScope: CoroutineScope? = null // 传进来的协程
    private var delegate: BiddingNativeManager? = null // 暂存的 -- 回调的时候列表在滑动状态 --> 等待下一次的触发直接展示
    var mLastItemPosition: Int = -1 // 最后一个Item的位置
    var mLastAdPosition: Int = -1 // 最后一个广告的位置
    private var wrRv: WeakReference<RecyclerView?>? = null // 列表
    var mCallback: ((lastAdPosition: Int, current: BiddingNativeManager) -> Unit?)? = null // 状态回调
    private val delegateList = mutableListOf<BiddingNativeManager>() // 已经添加到列表的 NativeSceneDelegate
    private var isIdle = false // 当前是否是静止状态
    private var sceneId: String = "" // 场景ID

    /**
     * 是否有加载广告的能力
     * 在特定情况下需要终止广告的插入
     */
    private var isEnable = true
    private fun getClassTag(): String = javaClass.simpleName
    private val debounceDelay: Long = 300 // 300ms 去抖动延迟
    private val handler = Handler(Looper.getMainLooper())

    private var debounceRunnable: Runnable = Runnable {
        getDelegate()
    }


    // =============================================================================================


    /**
     * 广告加载完成之后会通过这个回调将结果返回
     */
    fun setAdCallback(callback: ((lastAdPosition: Int, current: BiddingNativeManager) -> Unit)?) {
        mCallback = callback
    }

    /**
     * 当前场景信息
     */
    fun setSceneId(sceneId: String) {
        // 有可能无网进入此时并没有scene config
        this.sceneId = sceneId
    }

    /**
     * 禁用加载广告的功能
     */
    fun setEnable(enable: Boolean) {
        isEnable = enable
    }

    /**
     * 上下文信息
     */
    fun setCtxMap(ctxMap: MutableMap<String, Any>) {
        mCtxMap = ctxMap
    }

    /**
     * 协程传进来
     */
    fun setCoroutineScope(scope: CoroutineScope?) {
        mScope = scope
    }

    /**
     * 下拉刷新
     */
    fun refresh() {
        AdLogger.logSdkNative(
            "${getClassTag()} --> refresh() --> 资源回收 --> delegateList.forEach{it.destroy()} -- sceneId = $sceneId",
            writeToFile = false
        )
        delegateList.forEach {
            it.destroy()
        }
        mLastItemPosition = -1
        mLastAdPosition = -1
        delegate?.destroy()
        delegate = null
    }

    /**
     * 页面销毁
     */
    fun destroy() {
        refresh()
        handler.removeCallbacksAndMessages(null)
        wrRv?.clear()
        mCallback = null
    }

    /**
     * 将列表传进来，为了是监听最大可见项
     */
    fun setRecyclerView(recyclerView: RecyclerView?) {
        wrRv = WeakReference<RecyclerView?>(recyclerView)
        recyclerView?.addOnScrollListener(object : RecyclerView.OnScrollListener() {
            override fun onScrollStateChanged(recyclerView: RecyclerView, newState: Int) {
                super.onScrollStateChanged(recyclerView, newState)

                // 当前不支持加载广告
                if (isEnable.not()) {
                    return
                }

                val layoutManager: RecyclerView.LayoutManager? = recyclerView.layoutManager
                val linearManager = layoutManager as? LinearLayoutManager
                linearManager?.apply {
                    //获取最后一个可见view的位置
                    mLastItemPosition = linearManager.findLastVisibleItemPosition()
                }

                val gridManager = layoutManager as? GridLayoutManager
                gridManager?.apply {
                    //获取最后一个可见view的位置
                    mLastItemPosition = gridManager.findLastVisibleItemPosition()
                }

                // 当前是否是停止滚动了
                if (newState == RecyclerView.SCROLL_STATE_IDLE) {
                    isIdle = true
                    // 1. mLastItemPosition 是否大于阈值
                    // 2. mLastItemPosition - mLastAdPosition > 最小间隔阈值
                    if (mLastItemPosition >= getX() && mLastAdPosition == -1) {
                        //getDelegate()
                        postDelayGet()
                    } else if (mLastItemPosition >= getX() && mLastItemPosition - mLastAdPosition >= getY()) {
                        //getDelegate()
                        postDelayGet()
                    }
                } else {
                    isIdle = false
                }
            }
        })
    }


    // =============================================================================================


    /**
     * 获取 XY -- 注意这里的 xy 是服务端配置的
     */
    private fun getX(): Int {
        return SceneCommonConfig.getSceneX(sceneId)
    }

    private fun getY(): Int {
        return SceneCommonConfig.getSceneY(sceneId)
    }

    /**
     * 延迟处理逻辑
     */
    private fun postDelayGet() {
        handler.removeCallbacks(debounceRunnable)
        handler.postDelayed(debounceRunnable, debounceDelay)
    }

    /**
     * 获取广告并且回调给场景自己进行添加到列表
     */
    private fun getDelegate() {

//        AdLogger.logSdkNative(
//            "${getClassTag()} --> getDelegate() --> 滚动结束了 --> 插入广告 -- sceneId = $sceneId",
//            writeToFile = false
//        )

//        // 如果有已完成的但是还没有插入，那就直接插入
        if (delegate?.getMaxEcpmObject() != null) {
            insert()
            return
        }

        // 如果对象已经存在了那就不创建对象了，等待回调之后在插入
        if (null != delegate) {
            AdLogger.logSdkNative(
                "${getClassTag()} --> null != delegate", writeToFile = false
            )
            return
        }

        // 创建新的对象加载广告
        delegate = BiddingNativeManager()
        delegate?.let { nativeManager ->
            // 创建之后就添加到列表
            delegateList.add(nativeManager)
            // 使用传进来的协程
            mScope?.launch {
                nativeManager.setCtxMap(mCtxMap)
                nativeManager.setSceneId(sceneId)
                nativeManager.setListener(object : BiddingTAdditionalListener() {
                    override fun onBiddingLoad(maxEcpmObject: BiddingIntermediateMaterialBean?) {
                        super.onBiddingLoad(maxEcpmObject)
                        // 异步回调的时候需要判断一下当前是否在滑动
                        if (isIdle) {
                            insert()
                        }
                    }

                    override fun onBiddingError(p0: TAdErrorCode?) {
                        super.onBiddingError(p0)
                        AdLogger.logSdkNativeE("${getClassTag()} --> 广告加载失败 废弃当前的重新创建一个对象进行加载 -- p0 = $p0")
                        delegate?.destroy()
                        delegate = null
                    }
                })
                nativeManager.loadAd()
            }
        }
    }

    private fun insert() {
        delegate?.let {
            val lastAdPosition = mLastItemPosition + 1

            // 这里在校验一下
            if (lastAdPosition - mLastAdPosition < getY()) {
                return
            }
            mLastAdPosition = lastAdPosition

            mScope?.launch {
                withContext(Dispatchers.Main) {
                    mCallback?.invoke(mLastAdPosition, it)
                }
            }

            AdLogger.logSdkNative(
                "${getClassTag()} --> insert() --> 触发插入广告 --- mLastItemPosition = $mLastItemPosition -- mLastAdPosition = $mLastAdPosition"
            )

            // 数据重置
            delegate = null
        }
    }
}