package com.transsion.ad.middle.icon

import android.app.Activity
import android.content.Context
import android.text.TextUtils
import android.util.AttributeSet
import android.view.View
import android.view.ViewGroup
import android.widget.FrameLayout
import android.widget.ImageView
import android.widget.TextView
import com.blankj.utilcode.util.ConvertUtils.byte2FitMemorySize
import com.bumptech.glide.Glide
import com.transsion.ad.ps.PsLinkUtils
import com.transsion.ad.ps.model.RecommendInfo
import com.transsion.ad.strategy.AdClickManager
import com.transsion.ad.strategy.AdLayoutProviderBean
import com.transsion.ad.strategy.MeasureManager
import com.transsion.ad.view.AbsBuyOutNativeView
import com.transsion.ad.view.blur.BlurTransformation
import kotlinx.coroutines.runBlocking

/**
 * @author: zhangxinbing
 * @date : 2025/5/14 14:47
 * @description: Icon 广告组装
 */
class BuyOutIconView @JvmOverloads constructor(
    context: Context, attrs: AttributeSet? = null
) : AbsBuyOutNativeView(context, attrs) {

    private var mAdLayoutProvider: AdLayoutProviderBean? = null  // 布局封装
    private var mRecommendInfo: RecommendInfo? = null // APK 信息


    // ================================= View 曝光监测 ===============================================


    override fun display() {
        onShow()
    }

    override fun adShowTime() {
        //TODO("Not yet implemented")
    }


    // ===================================== API ===================================================


    /**
     * 资源回收
     */
    fun destroy() {
        removeAllViews()
        // 添加检测
        MeasureManager.removeSession(this)
        mCallback = null
    }

    /**
     * 模拟点击事件
     */
    fun clickAd() {
        mAdLayoutProvider?.layout?.performClick()
    }

    /**
     * 绑定原生广告
     */
    fun bindNativeView(
        recommendInfo: RecommendInfo?,
        adLayoutProvider: AdLayoutProviderBean?,
    ) {

        // 数据校验
        if (recommendInfo == null || adLayoutProvider == null) {
            return
        }

        // 保存数据
        mRecommendInfo = recommendInfo
        mAdLayoutProvider = adLayoutProvider

        // 展示数据
        showData(getSceneId(), adLayoutProvider, recommendInfo)

        // 组装View 将广告布局添加到容器中
        removeAllViews()
        addView(
            adLayoutProvider.layout, LayoutParams(
                ViewGroup.LayoutParams.MATCH_PARENT, ViewGroup.LayoutParams.MATCH_PARENT
            )
        )

        // 添加检测
        MeasureManager.addSession(this)
    }


    // =================================== 广告展示 =================================================


    /**
     * 广告展示
     */
    private fun showData(
        sceneId: String, adLayoutProvider: AdLayoutProviderBean?, recommendInfo: RecommendInfo?
    ) {

        // 修复线上问题
        // 在Android中使用Glide加载图片时，如果传入的context是view的context，可能会导致上述的IllegalArgumentException。
        // 这是因为view的context可能与Activity的生命周期不同步。
        // Exception java.lang.IllegalArgumentException: You cannot start a load for a destroyed activity
        // at com.bumptech.glide.manager.RequestManagerRetriever.assertNotDestroyed (RequestManagerRetriever.java:353)
        // at com.bumptech.glide.manager.RequestManagerRetriever.get (RequestManagerRetriever.java:153)
        // at com.bumptech.glide.manager.RequestManagerRetriever.get (RequestManagerRetriever.java:133)
        // at com.bumptech.glide.Glide.with (Glide.java:818)
        // at com.transsion.wrapperad.middle.nativead.NonNativeView.showData (NonNativeView.kt:266)
        if ((context as? Activity)?.isDestroyed == true) {
            //AdLogger.logSdkNative("${getClassTag()} --> 当前Activity销毁，广告就不需要展示了", writeToFile = false)
            return
        }

        // 设置标题
        adLayoutProvider?.titleId?.let { titleId ->
            adLayoutProvider.layout.findViewById<TextView>(titleId)?.apply {
                text = recommendInfo?.detail?.name
            }
        }

        // 设置描述
        adLayoutProvider?.descriptionId?.let { descriptionId ->
            adLayoutProvider.layout.findViewById<TextView>(descriptionId)?.apply {
                text = recommendInfo?.detail?.simpleDescription
            }
        }

        // 按钮文案
        adLayoutProvider?.callToActionId?.let { callToActionId ->
            adLayoutProvider.layout.findViewById<TextView>(callToActionId)?.apply {
                text = recommendInfo?.buttonText
                if (TextUtils.isEmpty(recommendInfo?.buttonText)) {
                    visibility = View.GONE
                }
                setOnClickListener {
                    clickAd(sceneId, recommendInfo)
                }
            }
        }

        // 大小
        runBlocking {
            adLayoutProvider?.apkSizeId?.let { apkSizeId ->
                adLayoutProvider.layout.findViewById<TextView>(apkSizeId)?.apply {
                    text = byte2FitMemorySize(recommendInfo?.detail?.sourceSize?.toLong() ?: 0L, 1)
                }
            }
        }

        // 评分
        adLayoutProvider?.ratingId?.let { ratingId ->
            adLayoutProvider.layout.findViewById<TextView>(ratingId)?.apply {
                text = recommendInfo?.detail?.star
            }
        }

        // 广告点击事件
        adLayoutProvider?.layout?.setOnClickListener {
            clickAd(sceneId, recommendInfo)
        }

        // Icon
        showImage(adLayoutProvider = adLayoutProvider, recommendInfo = recommendInfo)

        // 高斯模糊
        showBlur(adLayoutProvider = adLayoutProvider, recommendInfo = recommendInfo)
    }

    /**
     * 模糊背景
     */
    private fun showBlur(adLayoutProvider: AdLayoutProviderBean?, recommendInfo: RecommendInfo?) {
        adLayoutProvider?.burlBgId?.let { mediaId ->
            val imageViewBlur = adLayoutProvider.layout.findViewById<ImageView>(mediaId)
            imageViewBlur.scaleType = ImageView.ScaleType.CENTER_CROP
            // 如果本地地址是空的，那就不使用线上地址在线加载
            Glide.with(context).load(recommendInfo?.iconUrl).transform(BlurTransformation(25))
                .into(imageViewBlur)
        }
    }

    /**
     * 广告点击事件处理
     */
    private fun clickAd(sceneId: String, recommendInfo: RecommendInfo?) {
        // 广告点击回调
        AdClickManager.onClick(null)

        onClick()
        // 本地检测没有安装 -- 打开PS半屏
        // 需要检测已安装的ps版本是否支持半屏，不支持的走全屏
        PsLinkUtils.adClick(recommendInfo, isAutoDownload = true, scene = sceneId)
    }

    /**
     * 展示图片
     */
    private fun showImage(adLayoutProvider: AdLayoutProviderBean?, recommendInfo: RecommendInfo?) {
        // 主图 -- 这里需要判断是视频还是图片 -- 测试 先开发图片
        adLayoutProvider?.iconId?.let { iconId ->
            adLayoutProvider.layout.findViewById<FrameLayout>(iconId)?.apply {
                removeAllViews()
                // 大图 TODO 待优化
                val imageView = ImageView(context)
                imageView.scaleType = ImageView.ScaleType.FIT_CENTER
                Glide.with(context).load(recommendInfo?.iconUrl).into(imageView)
                // 全屏展示
                addView(imageView)
            }
        }
    }


    // =============================================================================================


    /**
     * 广告曝光
     */
    private fun onClick() {
        mCallback?.onMbIconClick(mRecommendInfo)
    }

    /**
     * 广告点击
     */
    private fun onShow() {
        mCallback?.onMbIconShow(mRecommendInfo)
    }
}