package com.transsion.ad.monopoly.intercept

import android.os.Parcelable
import android.text.TextUtils
import androidx.annotation.Keep
import com.blankj.utilcode.util.Utils
import com.transsion.ad.AdLogger
import com.transsion.ad.db.MbAdDatabase
import com.transsion.ad.db.plan.MbAdDbPlans
import com.transsion.ad.util.TimeUtil
import com.transsion.ad.monopoly.plan.AdPlansTransform
import com.transsion.ad.monopoly.model.AdPlans
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import kotlinx.parcelize.Parcelize
import java.io.Serializable
import java.util.concurrent.ConcurrentHashMap

/**
 * @author: zhangxinbing
 * @date : 2024/5/10 19:30
 * @description: 获取广告计划展示次数
 */
object NonAdShowedTimesManager {

    private fun getClassTag(): String = javaClass.simpleName

    /**
     * 内存保存一份
     */
    private val plansMap = ConcurrentHashMap<String, ShowedTimesMemoryBean>()

    /**
     * 通过广告ID获取展示次数
     */
    suspend fun getShowedTimes(plans: MbAdDbPlans?): Int {

        // 无论什么情况输出一下数据库里面的信息
        val latestNonAdPlans =
            MbAdDatabase.getInstance(Utils.getApp()).mbAdPlansDao().getPlanById(plans?.id)
        //AdLogger.logSdkE("${getClassTag()} --> getShowedTimes() --> 数据库的情况 --> latestNonAdPlans = $latestNonAdPlans")

        return plans?.let { nonAdPlans ->
            // 先从内存中获取
            plansMap[nonAdPlans.id]?.let { showedTimesMemoryBean ->
                // 同一天数据
                if (TextUtils.equals(showedTimesMemoryBean.date, TimeUtil.getCurrentDate())) {
                    showedTimesMemoryBean.showedTimes
                } else {
                    // 不是同一天数据直接返回0
                    //AdLogger.logSdkE("${getClassTag()} --> getShowedTimes() --> 先从内存中获取 --> 不是同一天,数据直接返回0")
                    0
                }
            } ?: run {
                // 同一天数据
                if (TextUtils.equals(latestNonAdPlans?.showDate, TimeUtil.getCurrentDate())) {
                    //AdLogger.logSdkE("${getClassTag()} --> getShowedTimes() --> 内存中不存在 --> 是同一天,数据返回 -- ${nonAdPlans.showedTimes ?: 0}")
                    // 内存中不存在就直接使用广告计划的，并且内存保存一份
                    plansMap[nonAdPlans.id] = ShowedTimesMemoryBean(
                        nonAdPlans.showedTimes ?: 0, TimeUtil.getCurrentDate()
                    )
                    latestNonAdPlans?.showedTimes ?: 0
                } else {
                    //AdLogger.logSdkE("${getClassTag()} --> getShowedTimes() --> 内存中不存在 --> 不是同一天,数据直接返回0")
                    // 内存中不存在就直接使用广告计划的，并且内存保存一份
                    plansMap[nonAdPlans.id] = ShowedTimesMemoryBean(0, TimeUtil.getCurrentDate())
                    0
                }
            }
        } ?: run {
            // 没有获取到数据默认0
            //AdLogger.logSdkE("${getClassTag()} --> getShowedTimes() --> plans == null --> 不是同一天,数据直接返回0")
            0
        }
    }

    /**
     * 保存
     */
    fun saveShowedTimes(mAdPlans: AdPlans?) {

        if (mAdPlans == null) {
            //AdLogger.logSdkE("${getClassTag()} --> saveShowedTimes() --> mAdPlans == null")
            return
        }

        val nonAdPlans = AdPlansTransform.transformAdPlans2NonAdPlans(mAdPlans)
        // 更新内存信息
        plansMap[nonAdPlans.id]?.let { showedTimesMemoryBean ->
            // 同一天数据
            if (TextUtils.equals(showedTimesMemoryBean.date, TimeUtil.getCurrentDate())) {
                showedTimesMemoryBean.showedTimes += 1
                plansMap.put(nonAdPlans.id, showedTimesMemoryBean)
            } else {
                // 不是同一天数据
                plansMap.put(nonAdPlans.id, ShowedTimesMemoryBean(1, TimeUtil.getCurrentDate()))
            }
        } ?: run {
            if (TextUtils.equals(nonAdPlans.showDate, TimeUtil.getCurrentDate())) {
                nonAdPlans.showedTimes = nonAdPlans.showedTimes?.plus(1)
                plansMap.put(
                    nonAdPlans.id,
                    ShowedTimesMemoryBean(nonAdPlans.showedTimes ?: 1, TimeUtil.getCurrentDate())
                )
            } else {
                // 不是同一天数据
                plansMap.put(nonAdPlans.id, ShowedTimesMemoryBean(1, TimeUtil.getCurrentDate()))
            }
        }

        //AdLogger.logSdkE("${getClassTag()} --> saveShowedTimes() --> plansMap = $plansMap")

        // 更新数据库信息
        CoroutineScope(Dispatchers.IO).launch {
            // 需要取最新的展示次数
//            val latestNonAdPlans =
//                AppDatabase.getInstance(Utils.getApp()).nonAdPlansDao().getPlanById(nonAdPlans.id)
//            val showedTimes = nonAdPlans.showedTimes ?: 0
            // 数据同步从缓存里面获取已经展示的数量
            nonAdPlans.showedTimes = plansMap[nonAdPlans.id]?.showedTimes ?: 0
            nonAdPlans.showDate = TimeUtil.getCurrentDate()
            MbAdDatabase.getInstance(Utils.getApp()).mbAdPlansDao().update(nonAdPlans)
            //AdLogger.logSdkE(getClassTag() + " saveShowCount() --> showedTimes = ${nonAdPlans.showedTimes} -- displayTimes = ${nonAdPlans.displayTimes}")

            //val latestNonAdPlans = MbAdDatabase.getInstance(Utils.getApp()).mbAdPlansDao().getPlanById(nonAdPlans.id)
            //AdLogger.logSdkE("${getClassTag()} --> saveShowCount() --> 保存数据之后重新查询一下,验证数据保存延时 --> latestNonAdPlans.showedTimes = ${latestNonAdPlans?.showedTimes}")
        }
    }
}


@Keep
@Parcelize
data class ShowedTimesMemoryBean(var showedTimes: Int, var date: String) : Parcelable, Serializable
