package com.transsion.ad.monopoly.intercept

import com.transsion.ad.db.plan.MbAdDbPlans
import com.transsion.ad.monopoly.model.AdPlansHitResponse
import com.transsion.ad.util.TimeUtil
import java.text.ParseException
import java.text.SimpleDateFormat
import java.util.Calendar
import java.util.Locale

/**
 * @author: zhangxinbing
 * @date : 2024/3/18 20:54
 * @description: 当前广告计划的开始时间和结束时间判断
 */
internal class ValidDispatchTimeHandler : AbsAdPlanInterceptHandler() {

    override suspend fun doHandle(
        sceneId: String, plans: MbAdDbPlans?, ctxMap: Map<String, Any>
    ): AdPlansHitResponse {

        // 获取当前时间戳
        val currentTimestamp = TimeUtil.getCurrentTimestamp()

        // 定义时间格式
        val timeFormat = SimpleDateFormat("HH:mm:ss", Locale.getDefault())

        // 检查 dispatchTimeStart 和 dispatchTimeEnd 是否为空字符串
        val dispatchStartTime = plans?.dispatchTimeStart
        val dispatchEndTime = plans?.dispatchTimeEnd
        if (dispatchStartTime.isNullOrBlank() || dispatchEndTime.isNullOrBlank()) {
            return getChain()?.doHandle(sceneId, plans, ctxMap) ?: getDefaultRes()
        }

        try {
            // 获取当前日期
            val calendar = Calendar.getInstance()
            val currentDate = calendar.time

            // 解析并设置当天的开始时间
            val startCalendar = Calendar.getInstance().apply {
                time = timeFormat.parse(dispatchStartTime) ?: currentDate
                set(Calendar.YEAR, calendar.get(Calendar.YEAR))
                set(Calendar.MONTH, calendar.get(Calendar.MONTH))
                set(Calendar.DAY_OF_MONTH, calendar.get(Calendar.DAY_OF_MONTH))
            }
            val todayStartTime = startCalendar.timeInMillis

            // 解析并设置当天的结束时间
            val endCalendar = Calendar.getInstance().apply {
                time = timeFormat.parse(dispatchEndTime) ?: currentDate
                set(Calendar.YEAR, calendar.get(Calendar.YEAR))
                set(Calendar.MONTH, calendar.get(Calendar.MONTH))
                set(Calendar.DAY_OF_MONTH, calendar.get(Calendar.DAY_OF_MONTH))
            }
            val todayEndTime = endCalendar.timeInMillis

            // 判断当前时间是否在范围内
            val isValidTime = currentTimestamp in todayStartTime..todayEndTime

            // 返回结果
            return if (isValidTime) {
                getChain()?.doHandle(sceneId, plans, ctxMap) ?: getDefaultRes()
            } else {
                AdPlansHitResponse(isHit = false, rejectMsg = "当前不在展示时间内--时间段")
            }
        } catch (e: ParseException) {
            //logErrorMsg("解析时间失败：${e.message} --> sceneId = $sceneId --> plans.id = ${plans.id}")
            return getChain()?.doHandle(sceneId, plans, ctxMap) ?: getDefaultRes()
        }
    }
}