package com.transsion.ad.monopoly.manager

import android.text.TextUtils
import com.blankj.utilcode.util.GsonUtils
import com.blankj.utilcode.util.Utils
import com.transsion.ad.AdLogger
import com.transsion.ad.MbAdContents
import com.transsion.ad.db.plan.MbAdDbPlans
import com.transsion.ad.monopoly.plan.AdPlansTransform
import com.transsion.ad.monopoly.model.MbAdPlansDto
import com.transsion.ad.strategy.AdMmkv
import com.transsion.ad.util.AssetsUtil
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.withContext
import java.io.File

/**
 * @author: zhangxinbing
 * @date : 2024/7/29 14:50
 * @description: 内置广告计划处理
 */
internal object AdPlansAssetsManager {

    private fun getClassTag(): String = javaClass.simpleName

    /**
     * 将资源文件保存到下载目录中
     */
    suspend fun saveAssets2Download() {
        withContext(Dispatchers.IO) {
            kotlin.runCatching {
                // 从数据库获取版本号，如果不存在那就将内置的配置写入数据库，如果存在那就发起请求判断是否需要更新配置
                val firstAdPlan = AdPlansStorageManager.getAllAdPlans()
                if (firstAdPlan.isNullOrEmpty()) {
                    // 读取assets文件夹下的配置文件
                    val readAssets = AssetsUtil.readAssets("local_non_ad_plan.json")
                    // 非空判断
                    if (TextUtils.isEmpty(readAssets).not()) {
                        // 解析json
                        // 输出流解析 JavaBean 集合
                        val analysisMbAdPlansDto =
                            GsonUtils.fromJson(readAssets, MbAdPlansDto::class.java)
                        val nonAdPlanDto = copyResource(analysisMbAdPlansDto)
                        // 处理完成之后保存数据到本地数据库
                        nonAdPlanDto.let {
                            // 将计划保存到数据库
                            saveAdPlanToDb(it)
                            // 保存版本号
                            AdMmkv.mmkvWithID.putString(
                                MbAdContents.NON_AD_PLAN_VERSION, it.data?.version
                            )
                            AdLogger.logAdInit("${getClassTag()} --> saveAssets2Download() --> success --> adPlans?.size -- ${it.data?.adPlans?.size} --> adPlans = ${it.data?.adPlans}")
                        }
                    } else {
                        AdLogger.logAdInitE("${getClassTag()} --> saveAssets2Download() --> assets数据 -- 保存数据库失败 -- fail fail fail -- readAssets = null")
                    }
                } else {
                    AdLogger.logAdInit("${getClassTag()} --> saveAssets2Download() --> 本地存在非标广告计划，不需要从assets下载")
                }
            }.getOrElse {
                AdLogger.logAdInitE("${getClassTag()} --> saveAssets2Download() --> 保存本地数据异常 -- error error error --> it = $it")
            }
        }
    }


    // =============================================================================================


    /**
     * 解析assets文件夹下的非标广告计划
     */
    private fun copyResource(mbAdPlansDto: MbAdPlansDto): MbAdPlansDto {
        // 遍历计划里面的素材进行下载
        mbAdPlansDto.data?.adPlans?.forEach {
            // 广告主头像保存本地
            val avatarPath = saveResFromAssets(it.advertiserAvatar ?: "")
            // 更新图片路径
            if (TextUtils.isEmpty(avatarPath).not()) {
                // 更新广告主头像路径
                it.advertiserAvatarPath = avatarPath
            }

            // 下载素材
            it.adMaterialList?.forEach { adMaterial ->
                // 视频资源
                adMaterial.video?.let { video ->
                    val path = saveResFromAssets(video.url ?: "")
                    // 更新图片路径
                    if (TextUtils.isEmpty(path).not()) {
                        video.path = path
                        adMaterial.downloadMaterialSuccess = true
                    }
                }
                // 图片资源
                adMaterial.image?.let { image ->
                    val path = saveResFromAssets(image.url ?: "")
                    // 更新图片路径
                    if (TextUtils.isEmpty(path).not()) {
                        image.path = path
                        adMaterial.downloadMaterialSuccess = true
                    }
                }
            }
        }
        return mbAdPlansDto
    }

    /**
     * 保存资源
     */
    private fun saveResFromAssets(url: String): String? {
        val fileName = File(
            MbAdContents.NON_AD_VIDEO_DOWNLOAD_FILE_PATH,
            "${System.currentTimeMillis()}.${MbAdContents.MINE_TYPE}"
        )
        return if (AssetsUtil.copyAssetFile(Utils.getApp(), url, fileName)) {
            // 更新图片路径
            fileName.absolutePath
        } else {
            null
        }
    }

    /**
     * 保存到本地数据库
     */
    private suspend fun saveAdPlanToDb(mbAdPlansDto: MbAdPlansDto) {
        mbAdPlansDto.data?.let {
            val list = mutableListOf<MbAdDbPlans>()
            // 数据转化
            it.adPlans?.forEach { adPlan ->
                val transformAdPlans = AdPlansTransform.transformAdPlans2NonAdPlans(adPlan)
                transformAdPlans.version = it.version
                list.add(transformAdPlans)
            }
            // 2. 将非标广告计划列表保存到数据库
            AdPlansStorageManager.insert(list)
        }
    }
}