package com.transsion.ad.ps

import android.os.SystemClock
import android.text.TextUtils
import com.blankj.utilcode.util.GsonUtils
import com.blankj.utilcode.util.Utils
import com.cloud.hisavana.sdk.common.util.MD5Utils
import com.tn.lib.net.manager.NetServiceGenerator
import com.tn.lib.net.utils.ParamHelper
import com.transsion.ad.AdLogger
import com.transsion.ad.MbAdContents
import com.transsion.ad.config.TestConfig
import com.transsion.ad.db.MbAdDatabase
import com.transsion.ad.db.plan.MbAdDbPlans
import com.transsion.ad.db.pslink.PsLinkAdPlan
import com.transsion.ad.monopoly.manager.AdPlansStorageManager
import com.transsion.ad.monopoly.model.MbAdImage
import com.transsion.ad.monopoly.plan.AdPlanMaterialManager
import com.transsion.ad.ps.model.PsLinkAdInfo
import com.transsion.ad.ps.model.PsLinkDto
import com.transsion.ad.scene.SceneGlobalConfig
import com.transsion.ad.strategy.AdMmkv
import com.transsion.ad.monopoly.plan.AdPlanSourceManager
import com.transsion.ad.strategy.AdResDownloadManager
import com.transsion.ad.util.FileUtil
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.withContext
import org.json.JSONObject
import java.io.File

/**
 * @author: zhangxinbing
 * @date : 2025/3/19 17:50
 * @description: PS 接口请求
 */
internal object PSAdPlanRequestManager {

    const val SCENE_CON = 14 // 默认样式
    private const val PAGE_SIZE = 10 // 每页数量
    private const val KEY_PS_LINK_REQUEST_GAP = "key_ps_link_request_gap" // PS 接口请求时间间隔

    private fun getClassTag(): String = javaClass.simpleName

    private val service by lazy { NetServiceGenerator.instance.getService(PsLinkApi::class.java) }

    private val psLinkAdPlanDao by lazy {
        MbAdDatabase.getInstance(Utils.getApp()).psLinkAdPlanDao()
    }


    // ====================================== 更新数据 ===============================================


    /**
     * 更新PS数据
     *
     * 1. 只要出发了 ad/config 就会请求PS分发接口
     * 2. PS分发接口 有请求间隔
     */
    suspend fun updatePsLink() {
        // 获取所有PS虚拟计划
        // 发起 ps广告计划拉取
        AdPlansStorageManager.getPlansBySource(AdPlanSourceManager.AdPlanEnum.AD_PLAN_AD_SOURCE_PS.value)
            ?.forEach { plan ->
                // 资源是否已经下载完成了
                // 如果有资源 就需要判断请求间隔，没有资源直接请求。
                if (AdPlanMaterialManager.hasPsOffer(plan)) {
                    val lastTimeStamp =
                        AdMmkv.getAdMMKV().getLong(KEY_PS_LINK_REQUEST_GAP + plan.id, 0)
                    val currentTime = SystemClock.elapsedRealtime()
                    val psLinkInterval = SceneGlobalConfig.getPsLinkInterval() * 1000L // 间隔时间 单位秒
                    // 判断时间隔
                    if (currentTime - lastTimeStamp > psLinkInterval) {
                        getList(plan)
                        // 保存请求时间
                        AdMmkv.getAdMMKV().putLong(KEY_PS_LINK_REQUEST_GAP + plan.id, currentTime)
                    } else {
                        val adSlot = plan.extAdSlot ?: SCENE_CON // PS 广告位ID
                        AdLogger.logPSW("${getClassTag()} --> updatePsLink() --> ID = ${plan.id} --> name = ${plan.name} --> adSlot = $adSlot --> 当前有资源，且在请求间隔内 --> 不做处理")
                    }
                } else {
                    getList(plan)
                    // 保存请求时间
                    AdMmkv.getAdMMKV()
                        .putLong(KEY_PS_LINK_REQUEST_GAP + plan.id, SystemClock.elapsedRealtime())
                }
            }
    }


    // ===================================== 内部逻辑 ================================================


    /**
     * 发起请求
     */
    suspend fun getPsLinkListBySlot(
        pageIndex: Int = 0,
        retry: Boolean = true,
        pageSize: Int = PAGE_SIZE,
        psScene: Int? = 14,
        nonId: String,
    ) {

        // 埋点
        PSReportUtil.reportPsLinkRequest(
            psState = PSReportUtil.PSRequestState.REQUEST_TRIGGER,
            count = null,
            errorMsg = null,
            psScene = psScene.toString(),
            scene = PSReportUtil.PsDistributeSceneEnum.PS_DISTRIBUTE_SCENE_AD_PLAN.value
        )

        withContext(Dispatchers.IO) {
            kotlin.runCatching {
                // 获取请求参数
                val jsonObject = getParams(pageIndex, pageSize, psScene)
                val url =
                    if (TestConfig.isReleasePsApi()) "https://feature-api.palmplaystore.com" else "https://test-feature-api.palmplaystore.com"
                //AdLogger.logPS("${getClassTag()} --> getPsLinkListBySlot() --> psScene = $psScene --> url = $url --> jsonObject = $jsonObject")

                // 请求
                request(jsonObject, psScene, nonId, url)
            }.getOrElse {
                AdLogger.logPSE("${getClassTag()} --> getPsLinkList() --> psScene = $psScene --> it = $it")

                // Ps 商单请求失败
                PSReportUtil.reportPsLinkRequest(
                    psState = PSReportUtil.PSRequestState.REQUEST_FAIL,
                    count = null,
                    errorMsg = "${it.message}",
                    scene = psScene.toString(),
                    psScene = PSReportUtil.PsDistributeSceneEnum.PS_DISTRIBUTE_SCENE_AD_PLAN.value
                )
            }
        }
    }


    // ===================================== 内部逻辑 ===============================================


    private suspend fun getList(plan: MbAdDbPlans) {
        var psScene = SCENE_CON
        if (!TextUtils.isEmpty(plan.extAdSlot)) {
            psScene = plan.extAdSlot?.toInt() ?: SCENE_CON
        }
        // 触发PS的下载
        getPsLinkListBySlot(0, false, 10, psScene = psScene, nonId = plan.id)
    }

    /**
     * 请求参数
     */
    private fun getParams(pageIndex: Int, pageSize: Int, scene: Int?): JSONObject {
        // 请求参数组装
        val jsonObject = JSONObject()
        jsonObject.put("bu", "mb")
        jsonObject.put("pageIndex", pageIndex) // 页码，从0开始
        jsonObject.put("pageSize", pageSize) // 每页数量，默认10
        jsonObject.put("media", 1)// 固定值
        jsonObject.put("scene", scene)// 下图 场景代码

        // 通用数据
        val commonParams = PsLinkApi.commonParams(System.currentTimeMillis())
        commonParams.forEach {
            jsonObject.put(it.key, it.value)
        }

        // 返回
        return jsonObject
    }

    /**
     * 发起请求
     */
    private suspend fun request(
        jsonObject: JSONObject, scene: Int?, adPlanId: String, url: String
    ) {

        val postPsRecommendList =
            service.postPsCandidateList(ParamHelper.createBody(jsonObject.toString()), url)

        // 输出请求结果
        AdLogger.logPS("${getClassTag()} --> request() --> PS 接口返回 --> psScene = $scene --> adPlanId = $adPlanId --> data = ${postPsRecommendList?.data?.size}")

        // PS 商单结果 埋点
        PSReportUtil.reportPsLinkRequest(
            psState = PSReportUtil.PSRequestState.REQUEST_SUCCESS,
            count = postPsRecommendList?.data?.size,
            errorMsg = "",
            psScene = scene.toString(),
            scene = PSReportUtil.PsDistributeSceneEnum.PS_DISTRIBUTE_SCENE_AD_PLAN.value
        )

        // 解析数据
        analyzeData(scene.toString(), adPlanId, postPsRecommendList)
    }

    /**
     * 解析数据
     */
    private suspend fun analyzeData(
        sceneStr: String, adPlanId: String, postPsRecommendList: PsLinkDto?
    ) {
        // 1.删除原slot对应的list
        // 2.下载对应素材
        // 3.下载成功后 插入到db 意味着db一定是可用数据
        // 4.全部slot场景 处理完 移除无用资源
        //step 1
        psLinkAdPlanDao.deleteExpiredPlansBySlot(sceneStr)

        //step 2
        var rank = 0
        postPsRecommendList?.data?.forEach {
            //如果不支持pslink 且没有gplink链接 过滤掉商单
            if (!PsLinkUtils.isSupportPsLink() && TextUtils.isEmpty(it.gpLink)) {
                return@forEach
            }

            // 触发埋点
            PSReportUtil.reportMaterialState(
                psState = PSReportUtil.PSRequestState.REQUEST_TRIGGER,
                recommendInfo = it,
                sceneStr = sceneStr,
                adPlanId
            )

            val itemDetail = it.detail
            val adImage = MbAdImage() // 图片素材对象
            adImage.url = it.showContent ?: itemDetail?.img0 // 图片素材地址

            // 下载素材
            val result = downloadImage(adImage)
            if (result) {
                val psLinkAdInfo = PsLinkAdInfo(
                    advertiserName = it.name,
                    advertiserAvatar = it.iconUrl,
                    advertiserAvatarPath = downloadAdvertiserAvatar(it.iconUrl),
                    title = itemDetail?.name,
                    desc = itemDetail?.simpleDescription,
                    buttonText = it.buttonText, // 广告按钮文案
                    url = adImage.url,
                    path = adImage.path,
                )
                val id = (sceneStr + it.id).hashCode()
                val adPlan = PsLinkAdPlan(
                    id = id,
                    nonId = adPlanId,
                    adSource = AdPlanSourceManager.AdPlanEnum.AD_PLAN_AD_SOURCE_PS.value,
                    extAdSlot = sceneStr,
                    rank = rank++,
                    psPlanId = it.id.toString(),
                    psLinkAdInfoStr = GsonUtils.toJson(psLinkAdInfo),
                    psInfoJson = GsonUtils.toJson(it),
                    updateTimestamp = System.currentTimeMillis()
                )

                // 下载成功埋点
                PSReportUtil.reportMaterialState(
                    psState = PSReportUtil.PSRequestState.REQUEST_SUCCESS,
                    recommendInfo = it,
                    sceneStr = sceneStr,
                    adPlanId
                )
                //step 3
                psLinkAdPlanDao.insertPslinkAd(adPlan)
            } else {
                // 下载失败埋点
                PSReportUtil.reportMaterialState(
                    psState = PSReportUtil.PSRequestState.REQUEST_FAIL,
                    recommendInfo = it,
                    sceneStr = sceneStr,
                    adPlanId = adPlanId
                )
            }
        }
    }

    /**
     * 下载图片素材
     */
    private fun downloadImage(mbAdImage: MbAdImage): Boolean {
        // 隐藏后缀 .mp4这样的
        val mineType = MbAdContents.MINE_TYPE
        val url = mbAdImage.url
        val fileName = MD5Utils.toMd5(url)
        val destination =
            MbAdContents.NON_AD_PS_DOWNLOAD_FILE_PATH + File.separatorChar + "$fileName.$mineType"
        val file = File(destination)
        //下载图类的  可能被压缩 判断条件不一致
        if (file.isFile && file.exists() && file.length() > 0) {
            //不需要重新下载 copy一次 都处理完后可删除old文件
            mbAdImage.path = destination
            return true
        } else {
            val downloadFileSuccess = AdResDownloadManager.downloadFile(url, destination)
            // size check
            val localFileSize = file.length()
            if (downloadFileSuccess && localFileSize > 0) {
                AdLogger.logPS("${getClassTag()} --> downloadAdPlan() --> 图片素材下载成功 --> -- destination = $destination")
                mbAdImage.path = destination
                return true
            } else {
                // 下载失败不处理
                AdLogger.logPSE("${getClassTag()} --> downloadAdPlan() --> 图片素材下载失败 --  downloadFileSuccess = $downloadFileSuccess -- localFileSize = $localFileSize -- destination = $destination")
                return false
            }
        }
    }

    /**
     * 下载广告主图标
     */
    private fun downloadAdvertiserAvatar(advertiserAvatar: String?): String {
        // 文件本地命名
        val fileName = System.currentTimeMillis().toString()
        val mineType = MbAdContents.MINE_TYPE

        // 文件本地存储路径
        val destination =
            MbAdContents.NON_AD_PS_DOWNLOAD_FILE_PATH + File.separatorChar + "$fileName.$mineType"

        // 判断一下文件是否已经存在
        if (FileUtil.isFileExists(destination)) {
            return destination
        }

        // 下载文件
        val downloadFile = AdResDownloadManager.downloadFile(advertiserAvatar, destination)
        return if (downloadFile) {
            destination
        } else {
            ""
        }
    }
}