package com.transsion.ad.ps

import android.text.TextUtils
import com.hisavana.common.bean.TAdErrorCode
import com.tn.lib.net.Http
import com.tn.lib.net.manager.NetServiceGenerator
import com.tn.lib.net.utils.ParamHelper
import com.tn.lib.util.networkinfo.NetworkUtil
import com.transsion.ad.AdLogger
import com.transsion.ad.config.TestConfig
import com.transsion.ad.middle.WrapperAdListener
import com.transsion.ad.ps.model.PsLinkDto
import com.transsion.ad.ps.model.RecommendInfo
import org.json.JSONObject

/**
 * @author: zhangxinbing
 * @date : 2025/3/20 10:12
 * @description: PS 分发接口请求
 */
object PSDistributionRequestManager {

    const val PAGE_SIZE = 10 // 每页数量

    private fun getClassTag(): String = javaClass.simpleName
    private val service by lazy { NetServiceGenerator.instance.getService(PsLinkApi::class.java) }


    // =============================================================================================


    suspend fun getPsLinkList(
        pageIndex: Int = 0,
        retry: Boolean = true,
        pageSize: Int = PAGE_SIZE,
        scene: String,
        pushModel: String = "",
        callback: ((data: PsLinkDto?) -> Unit?)? = null
    ) {

        // 获取请求参数
        val jsonObject = getParams(pageIndex, pageSize, pushModel)
        val url =
            if (TestConfig.isReleasePsApi()) "https://feature-api.palmplaystore.com" else "https://test-feature-api.palmplaystore.com"
        AdLogger.logAdInit("${getClassTag()} --> getPsLinkListBySlot() --> scene = $scene --> url = $url --> jsonObject = $jsonObject")

        // Ps 商单请求触发请求
        PSReportUtil.reportPsLinkRequest(
            psState = PSReportUtil.PSRequestState.REQUEST_TRIGGER,
            count = null,
            errorMsg = "",
            psScene = "",
            scene = scene
        )

        kotlin.runCatching {
            val postPsRecommendList =
                service.postPsCandidateList(ParamHelper.createBody(jsonObject.toString()), url)

            // PS 商单结果
            PSReportUtil.reportPsLinkRequest(
                psState = PSReportUtil.PSRequestState.REQUEST_SUCCESS,
                count = postPsRecommendList?.data?.size,
                errorMsg = "",
                psScene = "",
                scene = scene
            )
            val subData: MutableList<RecommendInfo> = mutableListOf()
            postPsRecommendList?.data?.forEach {
                //如果不支持pslink 且没有gplink链接 过滤掉商单
                if (!PsLinkUtils.isSupportPsLink() && TextUtils.isEmpty(it.gpLink)) {
                    return@forEach
                }
                subData.add(it)
            }
            postPsRecommendList?.data = subData
            // 回调结果
            callback?.invoke(postPsRecommendList)
        }.getOrElse {
            AdLogger.logPSE("${getClassTag()} --> getPsLinkList() --> 加载失败了 it = $it")
            // Ps 商单请求失败
            PSReportUtil.reportPsLinkRequest(
                psState = PSReportUtil.PSRequestState.REQUEST_FAIL,
                count = null,
                errorMsg = it.message,
                psScene = "",
                scene = scene
            )

            // 如果接口返回的异常中包含该错误，retry 一次
            if (it.message?.contains("Exception received from UploadDataProvider") == true && retry && NetworkUtil.hasCapabilityAsync()) {
                AdLogger.logPSE("${getClassTag()} --> getPsLinkList() --> retry")
                getPsLinkList(pageIndex, false, scene = scene)
            } else {
                callback?.invoke(null)
            }
        }
    }

    suspend fun getPsLinkListV2(
        pageIndex: Int = 0,
        retry: Boolean = true,
        pageSize: Int = PAGE_SIZE,
        scene: String,
        pushModel: String = "",
        listener: WrapperAdListener?
    ) {

        // 获取请求参数
        val jsonObject = getParams(pageIndex, pageSize, pushModel)
        val url =
            if (Http.isDev()) "https://test-feature-api.palmplaystore.com" else "https://feature-api.palmplaystore.com"
        AdLogger.logSdkIcon("${getClassTag()} --> getPsLinkListBySlot() --> scene = $scene --> url = $url --> jsonObject = $jsonObject")

        // Ps 商单请求触发请求
        PSReportUtil.reportPsLinkRequest(
            psState = PSReportUtil.PSRequestState.REQUEST_TRIGGER,
            count = null,
            errorMsg = "",
            psScene = "",
            scene = scene
        )

        kotlin.runCatching {
            val postPsRecommendList =
                service.postPsCandidateList(ParamHelper.createBody(jsonObject.toString()), url)

            // PS 商单结果
            PSReportUtil.reportPsLinkRequest(
                psState = PSReportUtil.PSRequestState.REQUEST_SUCCESS,
                count = postPsRecommendList?.data?.size,
                errorMsg = "",
                psScene = "",
                scene = scene
            )

            // 过滤掉无效的商单
            val subData: MutableList<RecommendInfo> = mutableListOf()
            postPsRecommendList?.data?.forEach {
                //如果不支持pslink 且没有gplink链接 过滤掉商单
                if (!PsLinkUtils.isSupportPsLink() && TextUtils.isEmpty(it.gpLink)) {
                    return@forEach
                }
                subData.add(it)
            }

            // 回调结果
            listener?.onPSDistributionReady(subData)
        }.getOrElse {
            AdLogger.logPSE("${getClassTag()} --> getPsLinkList() --> 加载失败了 it = $it")
            // Ps 商单请求失败
            PSReportUtil.reportPsLinkRequest(
                psState = PSReportUtil.PSRequestState.REQUEST_FAIL,
                count = null,
                errorMsg = it.message,
                psScene = "",
                scene = scene
            )

            // 如果接口返回的异常中包含该错误，retry 一次
            if (it.message?.contains("Exception received from UploadDataProvider") == true && retry && NetworkUtil.hasCapabilityAsync()) {
                AdLogger.logPSE("${getClassTag()} --> getPsLinkList() --> retry")
                getPsLinkListV2(pageIndex, false, scene = scene, listener = listener)
            } else {
                listener?.onError(TAdErrorCode(0, "${it.message}"))
            }
        }
    }


    // =============================================================================================


    /**
     * 请求参数
     */
    private fun getParams(pageIndex: Int, pageSize: Int, pushModel: String?): JSONObject {
        val jsonObject = JSONObject()
        jsonObject.put("bu", "mb")
        jsonObject.put("pageIndex", pageIndex) // 页码，从0开始
        jsonObject.put("pageSize", pageSize) // 每页数量，默认10
        jsonObject.put("recommendCode", "mb_apps")

        // TODO 待PS通知新增字段
        if (TextUtils.isEmpty(pushModel).not()) {
            jsonObject.put("pushModel", pushModel)
        }

        // 通用数据
        val commonParams = PsLinkApi.commonParams(System.currentTimeMillis())
        commonParams.forEach {
            jsonObject.put(it.key, it.value)
        }

        // 返回
        return jsonObject
    }

}