package com.transsion.ad.ps.attribution

import com.transsion.ad.AdLogger
import com.transsion.ad.db.pslink.AttributionPoint.AttributionType
import com.transsion.ad.monopoly.model.AdPlans
import com.transsion.ad.monopoly.plan.AdPlanSourceManager
import com.transsion.ad.ps.model.RecommendInfo
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch

/**
 * @author shmizhangxinbing
 * @date : 2025/7/14 19:44
 * @description: 数据生产
 */
object AttributionProduceManager : BaseAttributionProvider() {

    /**
     * 广告曝光
     *
     * 返回了字段就上报，没有返回就 不上报
     */
    fun onBiddingBuyOutDisplay(plans: AdPlans?) {
        // 收口判断，进处理PS类型包断广告
        if (AdPlanSourceManager.isPsAdPlan(plans)) {
            getAttributionUrlAndReport(plans, AttributionType.DISPLAY)
        }
    }

    /**
     * 广告点击
     *
     * 1. deeplink
     * 2. gpLink
     */
    fun onBiddingBuyOutClick(plans: AdPlans?, type: AttributionType) {
        // 收口判断，进处理PS类型包断广告
        if (AdPlanSourceManager.isPsAdPlan(plans)) {
            getAttributionUrlAndReport(plans, type)
        }
    }


    // =============================================================================================


    /**
     *
     */
    private fun getAttributionUrlAndReport(plans: AdPlans?, type: AttributionType) {
        kotlin.runCatching {
            val adMaterialList = plans?.adMaterialList // 素材集合
            if (adMaterialList?.isNotEmpty() == true) {
                val adMaterial = adMaterialList[0] // 素材
                val recommendInfo = adMaterial.psRecommendInfo // PS Offer
                val reportUrl = when (type) { //
                    AttributionType.DISPLAY -> {
                        recommendInfo?.showAttributionLink ?: ""
                    }

                    AttributionType.CLICK -> {
                        recommendInfo?.attributionLink ?: ""
                    }

                    AttributionType.GP_CLICK -> {
                        recommendInfo?.gpAttributionLink ?: ""
                    }
                }
                reportUrl(recommendInfo = recommendInfo, type = type, reportUrl = reportUrl)
            }
        }
    }

    /**
     * 创建任务
     */
    private fun reportUrl(recommendInfo: RecommendInfo?, type: AttributionType, reportUrl: String) {
        CoroutineScope(Dispatchers.IO).launch {

            // 触发数据消费
            AttributionConsumeManager.consume()

            //
            if (reportUrl.startsWith("http").not()) {
                AdLogger.logPSE("${getClassTag()} --> reportUrl() --> 创建一个归因任务 --> psId = ${recommendInfo?.id} --> type = $type --> reportUrl = $reportUrl --> 链接异常 --> 抛弃")
                return@launch
            }

            // 插入数据前判断一下是否已经存在对应Offer的任务了
            if (isAttributionPointExists(
                    psId = recommendInfo?.id.toString(), type = type, reportUrl = reportUrl
                )
            ) {
                AdLogger.logPS("${getClassTag()} --> reportUrl() --> Offer 归因任务已经存在 --> psId = ${recommendInfo?.id} --> type = $type --> reportUrl = $reportUrl")
                return@launch
            }

            // 插入任务
            insert(reportUrl = reportUrl, type = type, psId = recommendInfo?.id.toString())
            // 日志输出一下
            AdLogger.logPS("${getClassTag()} --> reportUrl() --> 创建一个归因任务 --> psId = ${recommendInfo?.id} --> type = $type --> reportUrl = $reportUrl")
            // 触发数据消费
            AttributionConsumeManager.consume()
        }
    }

}