package com.transsion.ad.strategy

import android.annotation.SuppressLint
import android.app.KeyguardManager
import android.content.Context
import android.os.Build
import android.os.Handler
import android.os.Looper
import android.os.PowerManager
import android.os.SystemClock
import android.text.TextUtils
import androidx.appcompat.app.AppCompatActivity
import androidx.lifecycle.lifecycleScope
import com.blankj.utilcode.util.ActivityUtils
import com.hisavana.common.bean.TAdErrorCode
import com.transsion.ad.AdLogger
import com.transsion.ad.MBAd
import com.transsion.ad.bidding.BiddingTAdditionalListener
import com.transsion.ad.bidding.base.BiddingIntermediateMaterialBean
import com.transsion.ad.bidding.base.InterceptUtil
import com.transsion.ad.bidding.interstitial.BiddingInterstitialManager
import com.transsion.ad.config.FullSearchModel
import com.transsion.ad.config.HotSceneManager
import com.transsion.ad.scene.SceneCommonConfig
import com.transsion.ad.scene.SceneStorage
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext

/**
 * @author shmizhangxinbing
 * @date : 2025/8/4 17:03
 * @description:
 */
object HotSplashManager : BiddingTAdditionalListener() {

    private var inBackgroundTime = 0L   // 在后台的累计时间
    private var inBackground = false    // 当前是否是后台
    private fun getClassTag(): String = javaClass.simpleName

    @SuppressLint("StaticFieldLeak")
    private var interstitialManager: BiddingInterstitialManager? = null
    private val mHandler = Handler(Looper.getMainLooper())


    // =============================================================================================


    /**
     * App 进入前台
     */
    fun onStart() {
        inBackground = false
        // 加载并展示广告
        mHandler.removeCallbacksAndMessages(null)
        mHandler.postDelayed({
            loadHotSplashAd()
        }, 300)
    }

    /**
     * App 进入后台
     */
    fun onStop() {
        AdLogger.logSdkIntercept("${getClassTag()} --> onStop() --> App进入后台")
        inBackgroundTime = SystemClock.elapsedRealtime()
        inBackground = true
        mHandler.removeCallbacksAndMessages(null)

        // 包断默认直接可用，这里仅需要触发Hi程序化即可
        BiddingInterstitialManager.preLoadAd(HotSceneManager.getHotSceneId())
    }


    // ==================================== 广告回调 =================================================


    private fun resetAd() {
        interstitialManager?.destroy()
        interstitialManager = null
    }

    override fun onBiddingLoad(maxEcpmObject: BiddingIntermediateMaterialBean?) {
        super.onBiddingLoad(maxEcpmObject)
        ActivityUtils.getTopActivity()?.let { topAct ->
            (topAct as? AppCompatActivity)?.let {
                it.lifecycleScope.launch {
                    withContext(Dispatchers.Main) {
                        interstitialManager?.startAdActivity(activity = topAct)
                    }
                }
            }
        }
    }

    override fun onBiddingWrapperAdClose(adSource: Int) {
        super.onBiddingWrapperAdClose(adSource)
        resetAd()
    }

    override fun onBiddingError(p0: TAdErrorCode?) {
        super.onBiddingError(p0)
        resetAd()
    }


    // =============================================================================================


    /**
     * 加载并且展示热启动开屏广告
     */
    private fun loadHotSplashAd() {

        val msg = isIntercept()
        if (TextUtils.isEmpty(msg).not()) {
            AdLogger.logSdkInterceptE("${getClassTag()} --> loadHotSplashAd() --> msg = $msg")
            return
        }

        CoroutineScope(Dispatchers.IO).launch {
            AdLogger.logSdk("${getClassTag()} --> loadHotSplashAd() --> App 进入前台符合条件 --> 准备展示开屏广告")

            // 这里需要判断一下，否则调用loadAd之后，因为已经展示了会回调加载失败，进而导致调用destroy,导致后续Activity事件无法回调
            if (InterceptUtil.isInterceptAdShowed()) {
                return@launch
            }

            if (interstitialManager == null) {
                interstitialManager = BiddingInterstitialManager()
                interstitialManager?.setSceneId(HotSceneManager.getHotSceneId())
                    ?.setListener(this@HotSplashManager)
            }
            interstitialManager?.loadAd()
        }
    }


    // =============================================================================================


    /**
     * 检查展示条件是否满足
     */
    private fun isIntercept(): String {
        // 处理App在锁屏状态下被拉活
        val topAct = ActivityUtils.getTopActivity()
        topAct ?: return "topAct is null"
        if (topAct.isDestroyed) {
            return "topAct?.isDestroyed == true"
        }

        val isFakeLaunch = isScreenOff(topAct) || isScreenLocked(topAct)
        if (isFakeLaunch) {
            return "处理App在锁屏状态下被拉活"
        }

        // 全搜的时候热启动不展示热启动广告
        if (FullSearchModel.isSkipHotSplash()) {
            // 重置数据
            FullSearchModel.setSkipHotSplash(isSkip = false)
            return "全搜的时候热启动不展示热启动广告"
        }

        // 在后台累计时间判断
        if (inBackgroundTime <= 0) {
            return "inBackgroundTime <= 0"
        }

        // 获取远端配置信息
        if (MBAd.getAdInitParams()?.isDebug == false) {
            // 新用户屏蔽 -- 目前只有热启动才有新用户屏蔽
            if (NewUserManager.isNewUserMask()) {
                return "新用户保护期间不展示热启动广告"
            }

            val time = System.currentTimeMillis() - inBackgroundTime
            val sceneConfig = SceneStorage.getSceneConfig(HotSceneManager.getHotSceneId())
            val hotSplashInterval =
                (sceneConfig?.get(SceneCommonConfig.SCENE_CONFIG_KEY_HOT_SPLASH_INTERVAL)?.asInt
                    ?: 1) * 60 * 1000
            if (time < hotSplashInterval) {
                return "不满足最小间隔时间 -- background_time = $time --- TIME_IN_THE_BACKGROUND = $hotSplashInterval"
            }
        }

        return ""
    }

    private fun isScreenOff(context: Context): Boolean {
        val powerManager = context.getSystemService(Context.POWER_SERVICE) as PowerManager
        val isScreenOff = !powerManager.isScreenOn
        return isScreenOff
    }

    private fun isScreenLocked(context: Context): Boolean {
        val keyguardManager = context.getSystemService(Context.KEYGUARD_SERVICE) as KeyguardManager
        val isLocked = if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
            keyguardManager.isDeviceLocked || keyguardManager.isKeyguardLocked
        } else {
            keyguardManager.isKeyguardLocked
        }
        return isLocked
    }

}