package com.transsion.ad.strategy

import android.content.Context
import android.text.TextUtils
import com.blankj.utilcode.util.GsonUtils
import com.blankj.utilcode.util.Utils
import com.google.gson.reflect.TypeToken
import com.tn.lib.net.cons.HeaderConst
import com.tn.lib.net.utils.MkvHeader
import com.tn.lib.util.device.TNDeviceHelper
import com.transsion.ad.AdLogger
import com.transsion.ad.db.mcc.LocalMcc
import com.transsion.ad.db.MbAdDatabase
import com.transsion.ad.util.AssetsUtil
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.withContext

/**
 * @author: zhangxinbing
 * @date : 2025/2/13 16:39
 * @description: 国家信息管理
 */
object NationalInformationManager {

    private fun getClassTag(): String = javaClass.simpleName

    /**
     * 将Assets数据保存到数据库
     */
    suspend fun initLocalMcc(context: Context) {
        // 读取本地的 local_mcc.json 文件
        withContext(Dispatchers.IO) {
            kotlin.runCatching {
                // 读取数据库是耗时工作
                // MCC(移动国家码) 这个是国际移动用户识别码，是区别移动用户所属国家的标志，共3位，中国为460
                // 不是经常变化的东西，所以这里只需要保存一次就可以了
                val firstLocalMcc =
                    MbAdDatabase.getInstance(context).localMccDao().getFirstLocalMcc()
                if (firstLocalMcc != null) {
                    AdLogger.logAdInit("${getClassTag()} --> initLocalMcc() --> 数据库已存在 --> success")
                    return@withContext
                }
                val readAssets = AssetsUtil.readAssets("local_mcc.json")
                // 输出流解析 JavaBean 集合
                val listType = object : TypeToken<List<LocalMcc>>() {}.type
                val codeList = GsonUtils.fromJson<List<LocalMcc>>(readAssets, listType)
                // 保存到数据库
                MbAdDatabase.getInstance(context).localMccDao().insert(codeList)
                AdLogger.logAdInit("${getClassTag()} --> initLocalMcc() --> assets数据 -- 保存数据库成功 -- success")
            }.getOrElse {
                AdLogger.logAdInitE("${getClassTag()} --> initLocalMcc() --> assets数据 -- 保存数据库失败 -- error -- it = $it")
            }
        }
    }

    /**
     * 获取当前国家的 MCC
     */
    fun getCurrentMcc(): String {
        var simOperator = MkvHeader.netMmkv?.getString(HeaderConst.SP_CODE, "") ?: ""
        if (TextUtils.isEmpty(simOperator)) {
            simOperator = TNDeviceHelper.getSimOperator()
        }
        if (simOperator.isNotEmpty() && simOperator.length >= 3) {
            simOperator = simOperator.substring(0, 3)
        }
        return simOperator
    }

    /**
     * 获取所有国家信息
     */
    suspend fun getAllNationalInformationList(): List<LocalMcc> {
        return withContext(Dispatchers.IO) {
            MbAdDatabase.getInstance(Utils.getApp()).localMccDao().getAllLocalMccList()
        }
    }

    /**
     * 通过ISO获取国家信息
     */
    suspend fun getNationalInformation(iso: String): LocalMcc? {
        return withContext(Dispatchers.IO) {
            MbAdDatabase.getInstance(Utils.getApp()).localMccDao().getLocalMccByColumnValue(iso)
        }
    }
}