package com.transsion.ad.util

import androidx.fragment.app.Fragment
import androidx.fragment.app.FragmentManager
import androidx.fragment.app.FragmentTransaction

/**
 * @author: zhangxinbing
 * @date : 2025/3/17 16:16
 * @description: Fragment 工具类
 */
internal object FragmentUtils {

    /**
     * 安全地添加 Fragment 到容器中
     *
     * @param fragmentManager FragmentManager（来自 Activity 或 Fragment）
     * @param containerId 容器的 ID（FrameLayout 的 ID）
     * @param fragment 要添加的 Fragment
     * @param addToBackStack 是否添加到回退栈
     * @param tag Fragment 的唯一标识（可选）
     */
    fun addFragmentSafely(
        fragmentManager: FragmentManager,
        containerId: Int,
        fragment: Fragment,
        addToBackStack: Boolean = false,
        tag: String? = fragment::class.java.simpleName
    ) {
        // 检查 FragmentManager 是否已经销毁
        if (fragmentManager.isStateSaved || fragmentManager.isDestroyed) {
            return
        }

        // 检查 Fragment 是否已经添加到容器中
        val existingFragment = tag?.let { fragmentManager.findFragmentByTag(it) }
        if (existingFragment != null && existingFragment.isAdded) {
            return
        }

        // 开启事务
        val transaction = fragmentManager.beginTransaction()

        // 添加 Fragment
        transaction.replace(containerId, fragment, tag)

        // 是否添加到回退栈
        if (addToBackStack) {
            transaction.addToBackStack(tag)
        }

        // 提交事务
        commitFragmentTransaction(fragmentManager, transaction)
    }

    /**
     * 安全地提交 Fragment 事务
     *
     * @param fragmentManager FragmentManager
     * @param transaction FragmentTransaction
     */
    private fun commitFragmentTransaction(
        fragmentManager: FragmentManager, transaction: FragmentTransaction
    ) {
        // 检查 FragmentManager 是否已经销毁
        if (fragmentManager.isStateSaved || fragmentManager.isDestroyed) {
            return
        }

        // 提交事务
        try {
            transaction.commitAllowingStateLoss()
        } catch (e: IllegalStateException) {
            e.printStackTrace()
        }
    }
}