package com.transsion.ad.view

import android.content.Context
import android.graphics.*
import android.util.AttributeSet
import android.view.ViewGroup
import com.transsion.ad.R

/**
 * @author shmizhangxinbing
 * @date : 2025/7/25 15:23
 * @description: 圆弧ViewGroup 可以设置控件的一边为圆弧样式
 */
class ArcEdgeLayout @JvmOverloads constructor(
    context: Context,
    attrs: AttributeSet? = null,
    defStyleAttr: Int = 0,
) : ViewGroup(context, attrs, defStyleAttr) {

    /**
     * 圆弧位置枚举：用于指定圆弧出现在布局的哪一边
     */
    enum class ArcPosition {
        TOP, RIGHT, BOTTOM, LEFT
    }

    // =============================================================================================


    private var arcPosition = ArcPosition.TOP   // 当前圆弧位置
    private var arcHeight = 0f                  // 圆弧高度
    private val path = Path()                   // 用于裁剪和绘制的路径
    private val rect = RectF()                  // 用于存储布局的矩形区域
    private val arcRect = RectF() // 圆弧的矩形范围

    // 用于绘制背景的画笔
    private val paint = Paint(Paint.ANTI_ALIAS_FLAG).apply {
        color = Color.WHITE
        style = Paint.Style.FILL
        isAntiAlias = true
    }

    init {
        // 使用硬件加速以提升性能
        setLayerType(LAYER_TYPE_HARDWARE, null)

        // 读取自定义属性
        context.obtainStyledAttributes(attrs, R.styleable.ArcEdgeLayout).apply {
            try {
                // 获取圆弧位置
                arcPosition = ArcPosition.values()[getInt(
                    R.styleable.ArcEdgeLayout_arcPosition, 0
                ).coerceIn(0, 3)]
                // 获取圆弧高度
                arcHeight = getDimension(R.styleable.ArcEdgeLayout_arcHeight, 40f)
            } finally {
                recycle()
            }
        }
    }


    // =============================================================================================


    override fun onSizeChanged(w: Int, h: Int, oldw: Int, oldh: Int) {
        super.onSizeChanged(w, h, oldw, oldh)
        // 更新布局矩形区域
        rect.set(0f, 0f, w.toFloat(), h.toFloat())
        // 更新裁剪路径
        updatePath()
    }

    override fun dispatchDraw(canvas: Canvas) {
        // 剪裁画布为圆弧路径区域
        canvas.clipPath(path)
        super.dispatchDraw(canvas)
    }

    override fun onDraw(canvas: Canvas) {
        // 绘制圆弧背景
        canvas.drawPath(path, paint)
        super.onDraw(canvas)
    }

    override fun onLayout(changed: Boolean, l: Int, t: Int, r: Int, b: Int) {
        // 布局所有子View到 (0,0)，并保留各自的测量宽高
        for (i in 0 until childCount) {
            val child = getChildAt(i)
            child.layout(0, 0, child.measuredWidth, child.measuredHeight)
        }
    }

    override fun onMeasure(widthMeasureSpec: Int, heightMeasureSpec: Int) {
        // 先测量自身
        super.onMeasure(widthMeasureSpec, heightMeasureSpec)
        // 再测量所有子View
        for (i in 0 until childCount) {
            val child = getChildAt(i)
            measureChild(child, widthMeasureSpec, heightMeasureSpec)
        }
    }


    // =============================================================================================


    /**
     * 设置圆弧位置，并更新路径和重绘
     */
    fun setArcPosition(position: ArcPosition) {
        if (arcPosition != position) {
            arcPosition = position
            updatePath()
            invalidate()
        }
    }

    /**
     * 设置圆弧高度，并更新路径和重绘
     */
    fun setArcHeight(height: Float) {
        if (arcHeight != height) {
            arcHeight = height
            updatePath()
            invalidate()
        }
    }


    // =============================================================================================


    /**
     * 更新裁剪和绘制路径，根据圆弧位置和高度动态生成
     */
    private fun updatePath() {
        path.reset()

        when (arcPosition) {
            ArcPosition.TOP -> {
                // 顶部圆弧
                arcRect.set(
                    -arcHeight, 0f, width + arcHeight, arcHeight * 2
                )
                path.moveTo(0f, arcHeight)
                path.arcTo(arcRect, 180f, 180f, false)
                path.lineTo(width.toFloat(), height.toFloat())
                path.lineTo(0f, height.toFloat())
            }

            ArcPosition.RIGHT -> {
                // 右侧圆弧
                arcRect.set(
                    width - arcHeight * 2, -arcHeight, width.toFloat(), height + arcHeight
                )
                path.moveTo(0f, 0f)
                path.lineTo(width - arcHeight, 0f)
                path.arcTo(arcRect, 270f, 180f, false)
                path.lineTo(0f, height.toFloat())
            }

            ArcPosition.BOTTOM -> {
                // 底部圆弧
                arcRect.set(
                    -arcHeight, height - arcHeight * 2, width + arcHeight, height.toFloat()
                )
                path.moveTo(0f, 0f)
                path.lineTo(width.toFloat(), 0f)
                path.lineTo(width.toFloat(), height - arcHeight)
                path.arcTo(arcRect, 0f, 180f, false)
            }

            ArcPosition.LEFT -> {
                // 左侧圆弧
                arcRect.set(
                    0f, -arcHeight, arcHeight * 2, height + arcHeight
                )
                path.moveTo(arcHeight, 0f)
                path.lineTo(width.toFloat(), 0f)
                path.lineTo(width.toFloat(), height.toFloat())
                path.lineTo(arcHeight, height.toFloat())
                path.arcTo(arcRect, 90f, 180f, false)
            }
        }

        path.close()
    }
}
